﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using CsGL.OpenGL;
using System.ComponentModel;
using System.Runtime.InteropServices;
using MyGeometry;
using System.Drawing;
using NumericalRecipes;

using MyCholmodSolver;
using System.IO;

namespace DotScissor
{
	public class VertexPair
	{
		public int I;
		public int J;
		public VertexPair(int i, int j)
		{
			this.I = i;
			this.J = j;
		}
	}
	public class Circle2d
	{
		public Vector2d centriod;
		public double radii = 0;
		public Circle2d() { }
		public Circle2d(Vector2d c, double r)
		{
			this.centriod = c;
			this.radii = r;
		}
		public void SetCentroid(Vector2d c)
		{
			this.centriod = c;
		}
		public void SetRadious(double r)
		{
			this.radii = r;
		}
		public List<Vector2d> GetSamples(int nsamples)
		{
			return this.GetSamples(this.radii, 0, nsamples);
		}
		public List<Vector2d> GetSamples(double radius, int nsamples)
		{
			return this.GetSamples(radius, 0, nsamples);
		}
		public List<Vector2d> GetSamples(double radius, double theta, int nsamples)
		{
			List<Vector2d> points = new List<Vector2d>();

			double dtheta = 2 * Math.PI / nsamples;
			for (int i = 0; i < nsamples; ++i)
			{
				double phi = theta + i * dtheta;
				double x = radius * Math.Cos(phi);
				double y = radius * Math.Sin(phi);

				Vector2d pt = new Vector2d(x, y) + this.centriod;
				points.Add(pt);
			}

			return points;
		}
	}

	[TypeConverterAttribute(typeof(DeformerConverter))]
	public unsafe class HarmonicSolver : IDisposable
	{
		#region IDisposable Members

		public void Dispose()
		{
			this.mesh = null;
			this.opt = null;
			this.isolineLists = null;
			this.fullIsolineList = null;

			this.isolinesSets = null;
			this.facesOnStrokes = null;
			this.isolinesOnStroke = null;

			this.triArea = null;
			this.isovalues = null;
			this.vertexConcaveNess = null;
			this.currHarmonicField = null;
			this.vtxDisplayColor = null;
			this.triDisplayColor = null;

			this.scalarFields = null;
			this.gradientFields = null;

			this.faceScores = null;
			this.faceGradient = null;
			this.faceRegionIndex = null;

			if (this.sparseSolver != null)
				this.sparseSolver.Release();
		}

		#endregion
		
		public class Option
		{
			public enum EnumFieldType
			{
				Harmonic = 0,
				IG,
				LG,
			};
			private EnumFieldType fieldType = EnumFieldType.IG;
			
			[Category("0.General Options")]
			public EnumFieldType FieldType
			{
				get { return fieldType; }
				set { fieldType = value; }
			}


			[Category("0.General Options")]
			public bool PartSegmentation { get; set; }
			
			[Category("0.General Options")]
			public bool UsePrevConstraints { get; set; }
			
			[Category("0.General Options")]
			public int NSamplesOnCircle { get; set; }
			
			[Category("1.Harmonic Options")]
			public int NumberOfIsolines { get; set; }

			[Category("1.Harmonic Options")]
			public bool SolverUseAtb { get; set; }

			[Category("2.IG Options")]
			public double SmallConstant { get; set; }

			[Category("2.IG Options")]
			public double ConcaveWeight { get; set; }
			
			[Category("2.IG Options")]
			public double GradientScale { get; set; }
			
			[Category("3.LG Options")]
			public double Alpha { get; set; }

			public Option()
			{
				this.SolverUseAtb = false;
				this.PartSegmentation = true;
				this.NumberOfIsolines = 15;
				this.ConcaveWeight = 0.01;
				this.SmallConstant = 1.0;
				this.GradientScale = 1.0;

				this.NSamplesOnCircle = 8;

				this.Alpha = 0.5;
			}
		}
		public Option opt = null;
		private Mesh mesh = null;
		private CholmodSolver sparseSolver = null;
		public bool hasHarmonic = false; /// -- indicate whether the hasHarmonic filed is generated
		public bool started = false;

		public HarmonicSolver(Mesh _mesh, Option _opt)
		{
			this.mesh = _mesh; this.opt = _opt; this.regionNum = opt.NumberOfIsolines;
			
			// initialize triareas 
			int fn = mesh.FaceCount, vn = mesh.VertexCount;
			this.triArea = new double[fn];
			for (int i = 0; i < fn; ++i)
			{
				triArea[i] = mesh.ComputeFaceArea(i);
			}
			this.vtxDisplayColor = new double[vn * 3];
			this.triDisplayColor = new double[fn * 3];
			this.linesOnFace = new List<int>[fn];
			for (int i = 0; i < fn; ++i)
			{
				this.linesOnFace[i] = new List<int>();
			}
			
			
			this.scalarFields = new List<double[]>();
			this.CreateIsoValues();

			this.concaveVertices = FindConcaveVertices();
            this.gaussianCurvature = ComputeGaussianCurvature();
            NormalizeData(this.gaussianCurvature);

			// for computing shortest path later on.
			
			this.ComputeEdgesInfo();

			/// initial a field
			/// 
			VertexPair pair = new VertexPair(0, mesh.VertexCount - 1);
			this.ComputeHarmonicField_IG(pair);
			this.prevPair = pair;
			this.hasHarmonic = true;


		}
		~HarmonicSolver()
		{
			GC.Collect();
		}


		public bool ObtainConstraints(List<VertexPair> vtxPairs)
		{
			if (vtxPairs == null) return false;

			this.vertexPairs = vtxPairs;

			return true;
		}
		public void Update()
		{
			this.ObtainIsolinesSets();
			this.ComputeIsolineScores();
			this.VoteBestCut();
		}

		private bool ShowPatches = false;

		private double[] triArea = null;
		private double[] isovalues = null;
		private double[] vertexConcaveNess = null;
		private double[] currHarmonicField = null;
		private double[] vtxDisplayColor = null;
		private double[] triDisplayColor = null;
		private List<double[]> scalarFields = null;
		private List<double[]> gradientFields = null;
		private int regionNum = 15;
		private double ConstantWeight = 1000;


		public class Isoline : PriorityQueueElement
		{
			public List<FaceRecord> faces = null;
			public BoundaryCurve boundaryCurve = null;
			public Vector3d centriod;
			public Vector2d scrCentriod;
			public bool lineAsCut = true;
			public bool isPatchType = false;
			public double radious = 0;
			public double value = -1;
			public int id = -1;
			public double score = 0;
			public bool isValidElector = true;
			public Isoline(int id_)
			{
				id = id_;
			}
			public Isoline()
			{
			}

			#region PriorityQueueElement Members
			private int pqIndex = -1;
			public int PQIndex
			{
				get
				{
					return pqIndex;
				}
				set
				{
					pqIndex = value;
				}
			}
			#endregion

			#region IComparable<PriorityQueueElement> Members
			public int CompareTo(Object other)
			{
				Isoline r = other as Isoline;
				if (this.radious < r.radious) return -1;
				if (this.radious > r.radious) return 1;
				return 0;
			}
			#endregion
		}
		private List<List<Isoline>> isolineLists = new List<List<Isoline>>();
		private List<Isoline> fullIsolineList = new List<Isoline>();
		private VertexPair prevPair = null;
		public List<VertexPair> vertexPairs = null;
		private List<List<Isoline>> isolinesSets = new List<List<Isoline>>();
		private List<List<int>> facesOnStrokes = new List<List<int>>();
		private List<Isoline> isolinesOnStroke = new List<Isoline>();


		#region Harmonic filed + isolines
		private double[] GetIsovalues(double min, double max, int steps)
		{
			double[] isovalues = new double[steps - 1];
			double s = (max - min) / steps;
			for (int i = 0; i < steps - 1; i++)
				isovalues[i] = s * (i + 1);
			return isovalues;
		}
		private void AddPositionalWeights(SparseMatrix M, int columnIndex, double weight)
		{
			M.AddRow();
			M.AddElement(M.RowSize - 1, columnIndex, weight);
		}
		private void ComputeHarmonicField_Cot(VertexPair pair)
		{
			/// --- build up lefthand side matrix A
			int vn = mesh.VertexCount;
			int fn = mesh.FaceCount;
			int bn = 2;
			int m = vn+bn;

			if (this.sparseSolver != null) this.sparseSolver.Release();
			this.sparseSolver = new CholmodSolver();
			
			this.sparseSolver.InitializeMatrixA(m, vn);
			for (int i = 0, j = 0; i < fn; i++, j += 3)
			{
				int c1 = mesh.FaceIndex[j];
				int c2 = mesh.FaceIndex[j + 1];
				int c3 = mesh.FaceIndex[j + 2];
				Vector3d v1 = new Vector3d(mesh.VertexPos, c1 * 3);
				Vector3d v2 = new Vector3d(mesh.VertexPos, c2 * 3);
				Vector3d v3 = new Vector3d(mesh.VertexPos, c3 * 3);
				double cot1 = (v2 - v1).Dot(v3 - v1) / (v2 - v1).Cross(v3 - v1).Length();
				double cot2 = (v3 - v2).Dot(v1 - v2) / (v3 - v2).Cross(v1 - v2).Length();
				double cot3 = (v1 - v3).Dot(v2 - v3) / (v1 - v3).Cross(v2 - v3).Length();
				sparseSolver.Add_Coef(c2, c2, -cot1); sparseSolver.Add_Coef(c2, c3, cot1);
				sparseSolver.Add_Coef(c3, c3, -cot1); sparseSolver.Add_Coef(c3, c2, cot1);
				sparseSolver.Add_Coef(c3, c3, -cot2); sparseSolver.Add_Coef(c3, c1, cot2);
				sparseSolver.Add_Coef(c1, c1, -cot2); sparseSolver.Add_Coef(c1, c3, cot2);
				sparseSolver.Add_Coef(c1, c1, -cot3); sparseSolver.Add_Coef(c1, c2, cot3);
				sparseSolver.Add_Coef(c2, c2, -cot3); sparseSolver.Add_Coef(c2, c1, cot3);
			}

			/// add positional weights
			int I = pair.I, J = pair.J;

			sparseSolver.Add_Coef(vn, I, 1.0 * ConstantWeight);
			sparseSolver.Add_Coef(vn + 1, J, 1.0 * ConstantWeight);

			if (!opt.SolverUseAtb) m = vn;

			double[] b = new double[m];
			double[] x = new double[vn];

			/// -- assign values to the right hand side b,-- Ax=b
			if (opt.SolverUseAtb)
			{
				b[vn] = 1.0 * ConstantWeight;
				b[vn + 1] = 0.0;
			}
			else /// set by user.. 
			{
				b[I] = 1.0 * ConstantWeight * ConstantWeight;
			}

			fixed (double* _b = b)
			{
				sparseSolver.InitializeMatrixB(_b, m, 1);
			}
			sparseSolver.InitializeSolver();
			sparseSolver.SetFinalPack(0);
			sparseSolver.Factorize();

			fixed (double* _x = x)
			{
				sparseSolver.Linear_Solve(_x, !opt.SolverUseAtb);
			}

			//for (int j = 0; j < vn; j++)
			//    if (x[j] < 0)
			//        x[j] = 0;
			this.currHarmonicField = x;
			this.hasHarmonic = true;
		}
		private void ComputeHarmonicField_IG(VertexPair pair)
		{
            /// build up lefthand side matrix A
			int n = mesh.VertexCount;
			int m = n+2;

			if (this.sparseSolver != null) this.sparseSolver.Release();
			this.sparseSolver = new CholmodSolver();
			
			this.sparseSolver.InitializeMatrixA(m, n);
            for (int i = 0; i < n; i++)
            {
                double tot = 0;

                Vector3d v1 = new Vector3d(mesh.VertexPos, i * 3);
                Vector3d n1 = new Vector3d(mesh.VertexNormal, i * 3);

				int nn = mesh.AdjVV[i].Length;
				double[] wei = new double[nn];
				int count = 0; 
				
				/// Part 1: change weight here and remember to change Part 2 of weight
                foreach (int adj in mesh.AdjVV[i])
                {
                    Vector3d v2 = new Vector3d(mesh.VertexPos, adj * 3);
                    
					double gau = (this.gaussianCurvature[adj] + this.gaussianCurvature[i]) / 2;
					double w = 1.0;// / (gau + opt.SmallConstant);
					
				//	w /=(this.avgEdgeLength + (v1 - v2).Length()); // consider edge length or area

					double eij = (v1-v2).Length() / this.avgEdgeLength;

					if (this.concaveVertices.Contains(adj) || this.concaveVertices.Contains(i))
					{
						w *= (opt.ConcaveWeight * eij);
					}

					wei[count++] = w;
                    tot += w;
                }

				count = 0;
                foreach (int adj in mesh.AdjVV[i])
                {
                    sparseSolver.Add_Coef(i, adj, wei[count++] / tot);
                }
                sparseSolver.Add_Coef(i, i, -1.0);
            }

			/// add positional weights
			int I = pair.I, J = pair.J;

			sparseSolver.Add_Coef(n, I, 1.0 * ConstantWeight);
			sparseSolver.Add_Coef(n + 1, J, 1.0 * ConstantWeight);

			if (!opt.SolverUseAtb) m = n;

			double[] b = new double[m];
			double[] x = new double[n];

			/// -- assign values to the right hand side b,-- Ax=b
			if (opt.SolverUseAtb)
			{
				b[n] = 1.0 * ConstantWeight;
				b[n + 1] = 0.0;
			}
			else /// set by user.. 
			{
				b[I] = 1.0 * ConstantWeight * ConstantWeight;
			}

			fixed (double* _b = b)
			{
				sparseSolver.InitializeMatrixB(_b, m, 1);
			}
			sparseSolver.InitializeSolver();
			sparseSolver.SetFinalPack(0);
			sparseSolver.Factorize();

			fixed (double* _x = x)
			{
				sparseSolver.Linear_Solve(_x, !opt.SolverUseAtb);
			}

			this.currHarmonicField = x;
			this.hasHarmonic = true;
		}
		private void ComputeHarmonicField_LG(VertexPair pair)
		{
            /// build up lefthand side matrix A
			int n = mesh.VertexCount;
			int m = n+2;

			double alpha = opt.Alpha;

			if (this.sparseSolver != null) this.sparseSolver.Release();
			this.sparseSolver = new CholmodSolver();
			
			this.sparseSolver.InitializeMatrixA(m, n);
            for (int i = 0, j = 0; i < n; i++,j+=3)
            {
                Vector3d v1 = new Vector3d(mesh.VertexPos, j);
                Vector3d n1 = new Vector3d(mesh.VertexNormal, j);

				int nn = mesh.AdjVV[i].Length;
				
				double[] angle = new double[nn];

				int count = 0; double avgAngle = 0;
				foreach (int adj in mesh.AdjVV[i])
				{
					Vector3d n2 = new Vector3d(mesh.VertexNormal, adj * 3);
					double cos = Math.Abs(n1.Dot(n2));

					if (cos > 1) cos = 1;

					double ang = Math.Acos(cos);
					angle[count++] = ang;
					avgAngle += ang;
				}
				avgAngle /= nn;
				
				/// Part 1: change weight here and remember to change Part 2 of weight
				count = 0;
				double[] wei = new double[nn]; double tot = 0;
                foreach (int adj in mesh.AdjVV[i])
                {
                    Vector3d v2 = new Vector3d(mesh.VertexPos, adj * 3);
                    
					Vector3d v12 = v2-v1;

					double len = v12.Length();
					double kn = 2*v12.Dot(n1) / len / len;

					double w1 = 1.0 / (1 + angle[count] / avgAngle);
					double w2 = 1.0 / Math.Sqrt(Math.Abs(kn)+1);


					double w = (1 - alpha) * w1 + alpha * w2;
					wei[count++] = w;
                    tot += w;
                }

				count = 0;
                foreach (int adj in mesh.AdjVV[i])
                {
                    sparseSolver.Add_Coef(i, adj, wei[count++] / tot);
                }
                sparseSolver.Add_Coef(i, i, -1.0);
            }

			/// add positional weights
			int I = pair.I, J = pair.J;

			sparseSolver.Add_Coef(n, I, 1.0 * ConstantWeight);
			sparseSolver.Add_Coef(n + 1, J, 1.0 * ConstantWeight);

			if (!opt.SolverUseAtb) m = n;

			double[] b = new double[m];
			double[] x = new double[n];

			/// -- assign values to the right hand side b,-- Ax=b
			if (opt.SolverUseAtb)
			{
				b[n] = 1.0 * ConstantWeight;
				b[n + 1] = 0.0;
			}
			else /// set by user.. 
			{
				b[I] = 1.0 * ConstantWeight * ConstantWeight;
			}

			fixed (double* _b = b)
			{
				sparseSolver.InitializeMatrixB(_b, m, 1);
			}
			sparseSolver.InitializeSolver();
			sparseSolver.SetFinalPack(0);
			sparseSolver.Factorize();

			fixed (double* _x = x)
			{
				sparseSolver.Linear_Solve(_x, !opt.SolverUseAtb);
			}

			this.currHarmonicField = x;
			this.hasHarmonic = true;
		}

		private double[] gaussianCurvature = null;
		private HashSet<int> concaveVertices = null;
		private HashSet<int> FindConcaveVertices()
        {
            HashSet<int> cv = new HashSet<int>();
            int n = mesh.VertexCount;
			this.vertexConcaveNess = new double[n];
            for (int i = 0; i < n; i++)
            {
                Vector3d v = new Vector3d(mesh.VertexPos, i * 3);
                Vector3d normal = new Vector3d(mesh.VertexNormal, i * 3);
                Vector3d center = new Vector3d();
                double d = 0.0;
                foreach (int adj in mesh.AdjVV[i])
                {
                    Vector3d u = new Vector3d(mesh.VertexPos, adj * 3);
                    center += u;
                    d += (u - v).Length();
                }
                center /= (double)mesh.AdjVV[i].Length;
                d /= (double)mesh.AdjVV[i].Length;

				double conv = (v - center).Dot(normal);
				this.vertexConcaveNess[i] = conv;

                if (conv < -1e-6)
                    cv.Add(i);
            }

            return cv;
        }
		private void NormalizeData(double[] arr)
        {
            int n = arr.Length;

            double min = double.MaxValue;
            double max = double.MinValue;
            for (int i = 0; i < n; i++)
            {
                if (arr[i] < min) min = arr[i];
                if (arr[i] > max) max = arr[i];
            }

            if (max != min)
            {
                for (int i = 0; i < n; i++)
                    arr[i] = (arr[i] - min) / (max - min);
            }
        }
        private double[] ComputeGaussianCurvature()
        {
            int n = mesh.VertexCount;
            int fn = mesh.FaceCount;
            double[] angleSum = new double[n];

            for (int i = 0; i < n; i++) angleSum[i] = 0;

            for (int i = 0; i < fn; i++)
            {
                int b = i * 3;
                int c1 = mesh.FaceIndex[b];
                int c2 = mesh.FaceIndex[b + 1];
                int c3 = mesh.FaceIndex[b + 2];
                Vector3d v1 = new Vector3d(mesh.VertexPos, c1 * 3);
                Vector3d v2 = new Vector3d(mesh.VertexPos, c2 * 3);
                Vector3d v3 = new Vector3d(mesh.VertexPos, c3 * 3);
                double len1 = (v2 - v3).Length();
                double len2 = (v3 - v1).Length();
                double len3 = (v1 - v2).Length();
                double sq1 = len1 * len1;
                double sq2 = len2 * len2;
                double sq3 = len3 * len3;

                // use law of cosines to compute the angle
                double a1 = Math.Acos((sq2 + sq3 - sq1) / (2.0 * len2 * len3));
                double a2 = Math.Acos((sq3 + sq1 - sq2) / (2.0 * len3 * len1));
                double a3 = Math.Acos((sq1 + sq2 - sq3) / (2.0 * len1 * len2));

                double ct1 = (v2 - v1).Dot(v3 - v1) / (len2 * len3);
                double ct2 = (v3 - v2).Dot(v1 - v2) / (len3 * len1);
                double ct3 = (v1 - v3).Dot(v2 - v3) / (len1 * len2);
                a1 = Math.Acos(ct1);
                a2 = Math.Acos(ct2);
                a3 = Math.Acos(ct3);

                if (double.IsNaN(a1) || double.IsNaN(a2) || double.IsNaN(a3))
                {
                    angleSum[c1] += Math.PI / 3;
                    angleSum[c2] += Math.PI / 3;
                    angleSum[c3] += Math.PI / 3;
                }
                else
                {
                    angleSum[c1] += a1;
                    angleSum[c2] += a2;
                    angleSum[c3] += a3;
                }
            }

            // compute gaussian curvature as angle excess
            double pi2 = 2.0 * Math.PI;
            for (int i = 0; i < n; i++)
            {
				double gau = angleSum[i] - pi2;

				angleSum[i] = Math.Abs(gau);

            }

            angleSum = SmoothData(angleSum);
            angleSum = SmoothData(angleSum);
            angleSum = SmoothData(angleSum);
            angleSum = SmoothData(angleSum); 
            angleSum = SmoothData(angleSum);
            angleSum = SmoothData(angleSum);
            angleSum = SmoothData(angleSum);
            angleSum = SmoothData(angleSum); 
            angleSum = SmoothData(angleSum);
            angleSum = SmoothData(angleSum);
            angleSum = SmoothData(angleSum);
            angleSum = SmoothData(angleSum); 
            angleSum = SmoothData(angleSum);
            angleSum = SmoothData(angleSum);
            angleSum = SmoothData(angleSum);
            angleSum = SmoothData(angleSum);

            return angleSum;
        }
        private double[] SmoothData(double[] data)
        {
            int n = mesh.VertexCount;
            double[] smoothdata = new double[n];

            for (int i = 0; i < n; i++)
            {
                double sum = 0.0;
                foreach (int adj in mesh.AdjVV[i])
                {
                    sum += data[adj];
                }
                sum /= mesh.AdjVV[i].Length;
            }

            return data;
        }
		private double ComputeFaceGradient(double[] h, int f)
        {
			int j = f * 3;
 
            int c1 = mesh.FaceIndex[j];
            int c2 = mesh.FaceIndex[j+1];
            int c3 = mesh.FaceIndex[j+2];
            
			Vector3d v1 = new Vector3d(mesh.VertexPos, c1 * 3);
            Vector3d v2 = new Vector3d(mesh.VertexPos, c2 * 3);
            Vector3d v3 = new Vector3d(mesh.VertexPos, c3 * 3);
            
			double d1 = h[c1];
            double d2 = h[c2];
            double d3 = h[c3];

            /// find closest point v on V21 to V3
            double r = (v3 - v1).Dot(v2 - v1) / (v2 - v1).Dot(v2 - v1);
            Vector3d v = v1 + r * (v2 - v1);
            double d = d1 + r * (d2 - d1);

            /// compute gradient vector1 
            Vector3d gv1 = ((v - v3) / (v - v3).Length()) * ((d - d3) / (v - v3).Length());
            Vector3d gv2 = ((v - v1) / (v - v1).Length()) * ((d - d1) / (v - v1).Length());

            
            double g = (gv1 + gv2).Length() / this.triArea[f];
            g = Math.Pow(g, opt.GradientScale);

			if (double.IsNaN(g))
			{
				g = 1e-5;
			}

			return g;
        }
		private void CreateIsoValues()
		{
			this.isovalues = GetIsovalues(0.0, 1.0, regionNum);
		}
		private void ObtainIsolines()
		{
			LocateFacesIsoPosition(this.currHarmonicField, this.isovalues);
			LocateIsoContours();
		}


		/// class for iso-contours extraction
		public class PQPairs : ICloneable
		{
			public int findex, lindex; // lindex-->contour line index
			public Vector3d p, q, n;
			public int pId = -1, qId = -1; // stand for new vertex index of p and q after subdivision
			public int e1, e2;
			public double ratio1, ratio2;
			public double isovalue;
			public PQPairs() { e1 = e2 = -1; }
			#region ICloneable Members
			public object Clone()
			{
				PQPairs pair = new PQPairs();
				pair.p = this.p;
				pair.q = this.q;
				pair.n = this.n;
				pair.pId = this.pId;
				pair.e1 = this.e1;
				pair.e2 = this.e2;
				pair.findex = this.findex;
				pair.lindex = this.lindex;
				pair.ratio1 = this.ratio1;
				pair.ratio2 = this.ratio2;
				pair.isovalue = this.isovalue;

				return pair;
			}

			#endregion
		}
		public class IsoFaceRec
		{
			public int index;
			public bool valid = false;
			public bool subdived = false;
			public List<PQPairs> pqPairs = new List<PQPairs>();
			public IsoFaceRec() { }
		}
		public class FaceRecord
		{
			public IsoFaceRec face = null;
			public PQPairs pq = null;
			public FaceRecord(IsoFaceRec r, PQPairs _pq)
			{
				face = r;
				pq = _pq;
			}
		} // each face <-> a unique pqpair
		private IsoFaceRec[] isofaces = null;
		private void LocateFacesIsoPosition(double[] f, double[] isovals)
		{
			if (f == null) throw new ArgumentException();

			// initialize isofaces
			this.isofaces = new IsoFaceRec[mesh.FaceCount];
			for (int i = 0; i < mesh.FaceCount; ++i)
				this.isofaces[i] = new IsoFaceRec();
			for (int i = 0; i < isovalues.Length; ++i)
			{
				double v = isovals[i];
				for (int k = 0, m = 0; k < this.mesh.FaceCount; ++k, m += 3)
				{
					int c1 = this.mesh.FaceIndex[m];
					int c2 = this.mesh.FaceIndex[m + 1];
					int c3 = this.mesh.FaceIndex[m + 2];
					PQPairs r = null;
					if ((f[c1] <= v && f[c2] >= v) || (f[c2] <= v && f[c1] >= v))
					{
						if (r == null) { r = new PQPairs(); r.findex = k; }
						if (r.e1 == -1)
						{
							r.e1 = 0; r.ratio1 = (v - f[c1]) / (f[c2] - f[c1]);
						}
						else
						{
							r.e2 = 0; r.ratio2 = (v - f[c1]) / (f[c2] - f[c1]);
						}
					}
					if ((f[c2] <= v && f[c3] >= v) || (f[c3] <= v && f[c2] >= v))
					{
						if (r == null) { r = new PQPairs(); r.findex = k; }
						if (r.e1 == -1)
						{
							r.e1 = 1; r.ratio1 = (v - f[c2]) / (f[c3] - f[c2]);
						}
						else
						{
							r.e2 = 1; r.ratio2 = (v - f[c2]) / (f[c3] - f[c2]);
						}
					}
					if ((f[c3] <= v && f[c1] >= v) || (f[c1] <= v && f[c3] >= v))
					{
						if (r == null) { r = new PQPairs(); r.findex = k; }
						if (r.e1 == -1)
						{
							r.e1 = 2; r.ratio1 = (v - f[c3]) / (f[c1] - f[c3]);
						}
						else
						{
							r.e2 = 2; r.ratio2 = (v - f[c3]) / (f[c1] - f[c3]);
						}
					}
					if (r == null) continue;
					if (r.e1 == -1 || r.e2 == -1) continue;

					r.isovalue = v;
					r.lindex = i;

					Vector3d v1 = new Vector3d(mesh.VertexPos, c1 * 3);
					Vector3d v2 = new Vector3d(mesh.VertexPos, c2 * 3);
					Vector3d v3 = new Vector3d(mesh.VertexPos, c3 * 3);
					Vector3d n1 = new Vector3d(mesh.FaceNormal, k * 3);
					Vector3d p = new Vector3d(), q = new Vector3d();
					switch (r.e1)
					{
						case 0: p = v2 * r.ratio1 + v1 * (1.0 - r.ratio1); break;
						case 1: p = v3 * r.ratio1 + v2 * (1.0 - r.ratio1); break;
						case 2: p = v1 * r.ratio1 + v3 * (1.0 - r.ratio1); break;
					}
					switch (r.e2)
					{
						case 0: q = v2 * r.ratio2 + v1 * (1.0 - r.ratio2); break;
						case 1: q = v3 * r.ratio2 + v2 * (1.0 - r.ratio2); break;
						case 2: q = v1 * r.ratio2 + v3 * (1.0 - r.ratio2); break;
					}
					r.n = n1; r.p = p; r.q = q;
					isofaces[k].pqPairs.Add(r);
					isofaces[k].index = k;
					isofaces[k].valid = true;
				}
			}
		}
		private void LocateIsoContours()
		{
			// -- find out all the faces belong to a group (with the same iso-value)
			int n = mesh.FaceCount, m = opt.NumberOfIsolines;
			List<FaceRecord>[] faces = new List<FaceRecord>[m];
			for (int i = 0; i < m; ++i)
			{
				faces[i] = new List<FaceRecord>();
			}
			for (int i = 0; i < isofaces.Length; ++i)
			{
				if (isofaces[i] != null && isofaces[i].valid)
				{
					foreach (PQPairs pq in isofaces[i].pqPairs)
					{
						int idx = pq.lindex;
						faces[idx].Add(new FaceRecord(isofaces[i], pq));
					}
				}
			}
			// -- further divid each set into connected groups --> isocontours
			bool[] tag = new bool[mesh.FaceCount]; int lineIndex = 0;
			this.fullIsolineList.Clear();
			for (int i = 0; i < m; ++i)
			{
				List<FaceRecord> s = faces[i];
				foreach (FaceRecord rec in s)
					tag[rec.face.index] = true;
				FaceRecord prev = null;
				foreach (FaceRecord rec in s)
				{
					int v = rec.face.index;
					if (tag[v])
					{
						List<FaceRecord> flist = new List<FaceRecord>();

						int next = v; flist.Add(rec); tag[next] = false; prev = rec; // visited
						bool nextExist = true;
						while (nextExist)
						{
							nextExist = false;
							foreach (int adj in mesh.AdjFF[next])
							{
								if (tag[adj])
								{
									next = adj;

									FaceRecord fr = null;
									foreach (FaceRecord r in s)
									{
										if (adj == r.face.index)
										{
											fr = r;
											break;
										}
									}
									
									if (fr == null) throw new Exception();

									if (!HasCommonPorQ(prev, fr))
									{
										continue;
									}

									flist.Add(fr); prev = fr;
									tag[next] = false;
									nextExist = true;
									break;
								}
							}
						}
						Isoline line = new Isoline(lineIndex++);
						line.faces = flist;
						line.value = isovalues[i];
						this.fullIsolineList.Add(line);
					}
				}
			}
		}
		private bool HasCommonPorQ(FaceRecord f, FaceRecord g)
		{
			return IsTheSamePosition(f.pq.p, g.pq.p) || IsTheSamePosition(f.pq.p, g.pq.q)
				|| IsTheSamePosition(f.pq.q, g.pq.p) || IsTheSamePosition(f.pq.q, g.pq.q);
		}
		private bool IsTheSamePosition(Vector3d s, Vector3d t)
		{
			if ((s - t).Length() < 1e-08)
				return true;
			return false;
		}


		/// for segmentation
		private Set<int> ObtainStrokeFaces(VertexPair pair)
		{			
			int n = mesh.VertexCount;
			int source = pair.I;
			int target = pair.J;

			int[] pred = new int[n];
			for (int i = 0; i < n; ++i) pred[i] = -1;
			bool[] mark = new bool[n];
			Queue<int> iQue = new Queue<int>();
			iQue.Enqueue(source); mark[source] = true;
			bool find = false;
			while (iQue.Count > 0 && !find)
			{
				int seed = iQue.Dequeue();
				foreach (int j in mesh.AdjVV[seed])
				{
					if (!mark[j])
					{
						mark[j] = true;
						pred[j] = seed;
						if (j == target) { find = true; break; }
						iQue.Enqueue(j);
					}
				}
			}
			List<int> vset = new List<int>();
			int tmp = target; 
			while (pred[tmp] != -1)
			{
				vset.Add(tmp);
				tmp = pred[tmp];
			}
			vset.Add(tmp); vset.Reverse(0,vset.Count);
			Set<int> fSet = new Set<int>();
			foreach (int v in vset)
			{
				foreach (int f in mesh.AdjVF[v])
				{
					fSet.Add(f);
				}
			}
			return fSet;
		}
		private List<int>[] linesOnFace = null; // record isolines that passing through each face

		public Isoline currBestCut = null;
		public int currClickVertex = -1;

		private List<Isoline> bestCuts = new List<Isoline>();
		private void ObtainIsolinesOnStroke(VertexPair pair)
		{
			int n = mesh.FaceCount;

			/// -- obtain face-isoline mapping --
			for (int i = 0; i < n; ++i) this.linesOnFace[i].Clear();
			int index = 0;
			foreach (Isoline line in this.fullIsolineList)
			{
				foreach (FaceRecord f in line.faces)
				{
					int b = f.face.index;
					if (!linesOnFace[b].Contains(index))
						this.linesOnFace[b].Add(index);
				}
				index++;
			}
			/// -- get lines that passing through vertices on stroke --
			List<int> linesOnStroke = new List<int>();
			Set<int> fOnStrokes = ObtainStrokeFaces(pair);
			foreach (int f in fOnStrokes)
			{
				List<int> list = this.linesOnFace[f];
				foreach (int lineid in list)
					if (!linesOnStroke.Contains(lineid))
					{
						linesOnStroke.Add(lineid);
					}
			}
			
			SortLines(linesOnStroke); // --- sort according to the isovalues

			this.isolinesOnStroke = new List<Isoline>();
			foreach (int id in linesOnStroke)
			{
				Isoline line = this.fullIsolineList[id];
				ObtainIsolineProperties(line);
				this.isolinesOnStroke.Add(line);
			}
		}
		private void AddCurrIsolinesSet()
		{
			this.scalarFields.Add(this.currHarmonicField);
			this.isolinesSets.Add(this.isolinesOnStroke);
		}
		private void JudgeIsolinesType()
		{
			double[] normal = mesh.FaceNormal;
			foreach (List<Isoline> list in this.isolinesSets)
			{
				foreach (Isoline line in list)
				{
					int f = line.faces[0].face.index, count = 0;
					Vector3d ni = new Vector3d(normal, f * 3);
					foreach (FaceRecord rec in line.faces)
					{
						int j = rec.face.index;
						Vector3d nj = new Vector3d(normal, j * 3);
						if (ni.Dot(nj) < 0) // opposite
						{
							++count;
						}
					}
					if (count < 1)
					{
						line.isPatchType = true;
					}
				}
			}
		}


		private double[] faceScores = null;
		private double[] faceGradient = null;
		private Isoline baseLine = null;


		/// sort isolines according to a desending order of isovalue --
		private void SortLines(List<int> linesOnV)
		{
			if (linesOnV.Count <= 1) return;

			List<double> arr = new List<double>();
			Dictionary<double, int> dict = new Dictionary<double, int>();

			foreach (int v in linesOnV)
			{
				double val = this.fullIsolineList[v].value;

				if (dict.ContainsKey(val))
				{
					Program.PrintText("Key exisit: " + val.ToString());
					continue;
				}
				dict.Add(val, v);
				arr.Add(val);
			}
			arr.Sort();
			linesOnV.Clear();
			for (int i = arr.Count - 1; i >= 0; --i)
			{
				double val = arr[i];
				linesOnV.Add(dict[val]);
			}
		}
		private void ObtainIsolineProperties(Isoline line)
		{
			// obtain radius --
			double C = 0;
			foreach (FaceRecord f in line.faces)
			{
				if (f.face.valid == false) throw new ArgumentException();
				PQPairs pair = f.pq;
				C += (pair.p - pair.q).Length();
			}
			line.radious = C / (2 * Math.PI);

			Vector3d center = new Vector3d();
			foreach (FaceRecord f in line.faces)
			{
				if (f.face.valid == false) throw new ArgumentException();
				
				PQPairs pair = f.pq;
				center += (pair.p + pair.q) / 2;
			}
			line.centriod = center / line.faces.Count;
		}
		private void ObtainIsolinesScreenProjectionPosition()
		{
			MeshView view = Program.FormMain.MeshView;

			Rectangle viewport = new Rectangle(0, 0, view.Width, view.Height);
			OpenGLProjector projector = new OpenGLProjector();


			foreach (List<Isoline> isolines in this.isolinesSets)
			{
				foreach (Isoline line in isolines)
				{
					Vector3d v = projector.Project(line.centriod);
					Vector2d u = new Vector2d(v.x, v.y);

					if (!viewport.Contains((int)v.x, (int)v.y)) continue;

					line.scrCentriod = u;
				}
			}
		}
		private void ObtainIsolinesSets()
		{
			this.scalarFields.Clear();
			this.isolinesSets.Clear();

			foreach (VertexPair pair in this.vertexPairs)
			{
				if (!this.hasHarmonic)
				{
					switch (opt.FieldType)
					{
						case Option.EnumFieldType.Harmonic:
							this.ComputeHarmonicField_Cot(pair);
							break;
						case Option.EnumFieldType.IG:
							this.ComputeHarmonicField_IG(pair);
							break;
						case Option.EnumFieldType.LG:
							this.ComputeHarmonicField_LG(pair);
							break;
					}
				}
				else
				{
					this.UpdateHarmonicField(pair);
					this.started = true;
				}

				this.ObtainIsolines();
				this.ObtainIsolinesOnStroke(pair);
				this.AddCurrIsolinesSet();

				this.prevPair = pair;
			}

			this.JudgeIsolinesType();
			this.currHarmonicField = this.scalarFields[0];
			
		//	this.vtxDisplayColor = this.AssignColors(this.currHarmonicField);


		//	this.ObtainIsolinesScreenProjectionPosition();


		}
		private void ComputeIsolineScores()
		{
			if (this.currClickVertex == -1) throw new ArgumentException("No click point!");


			// ---------------------------------------------------------------------
			// globally, find the sudden jumping isoline.
			// ---------------------------------------------------------------------
			PriorityQueue Q = new PriorityQueue(); bool isPartType = false;
			foreach (List<Isoline> list in this.isolinesSets)
			{
				foreach (Isoline line in list)
				{
					if (!line.isPatchType) isPartType = true;
					Q.Insert(line);
				}
			}

			List<Isoline> lineList = new List<Isoline>();
			List<double> radii = new List<double>();
			while (!Q.IsEmpty())
			{
				Isoline line = Q.DeleteMin() as Isoline;
				lineList.Add(line);
				radii.Add(line.radious);
			}
			
			int k = radii.Count; double thr = 1.5;
			int jumpIndex = k-1; double jumpRadius = radii[jumpIndex];
			int baseIndex = 0;
			if (isPartType)
			{
				for (int i = 0; i < k; ++i)
				{
					if (!lineList[i].isPatchType)
					{
						baseIndex = i;
						break;
					}
				}
			}

	//		Program.PrintText("baseIndex = " + baseIndex.ToString());
			this.baseLine = lineList[baseIndex];

			double baseRadius = radii[baseIndex];
			for (int i = 0; i < k; ++i)
			{
				double dt = radii[i] / baseRadius;
				if (dt > thr)
				{
					jumpRadius = radii[i];
					jumpIndex = i;
					break;
				}
			}

			double maxRadius = radii[jumpIndex - 1];

	//		Program.PrintText("Jumping index: " + jumpIndex.ToString());

			double tightU = radii[0]; // tightness sigma
			for (int i = 0; i < jumpIndex; ++i)
			{
				double ratio = maxRadius / radii[i];
				if (ratio < thr)
				{
					tightU = radii[i];
					break;
				}
			}


			for (int i = 0; i < baseIndex; ++i)
			{
				lineList[i].isValidElector = false;
				lineList[i].score = 1e-6;
			}
			for (int i = jumpIndex; i < k; ++i)
			{
				lineList[i].isValidElector = false;
				lineList[i].score = 1e-6;
			}


			Vector3d clickPoint = new Vector3d(mesh.VertexPos, this.currClickVertex * 3);

			// now for individual set of isolines
			foreach (List<Isoline> list in this.isolinesSets)
			{
				int n = list.Count;

				// ---------------------------------------------------------------------
				// compute average isoline to clicking point distance
				// ---------------------------------------------------------------------
				double[] line2ClickDist = new double[n];
				for (int i = 0; i < n; ++i)
				{
					Isoline line = list[i]; double dis = 0;
					foreach (FaceRecord rec in line.faces)
					{
						Vector3d o = (rec.pq.p + rec.pq.q) / 2;
						dis += (clickPoint - o).Length();
					}
					line2ClickDist[i] = dis / line.faces.Count;
				}

				double minLine2ClickDis = double.MaxValue;
				for (int i = 0; i < n; ++i)
				{
					if (minLine2ClickDis > line2ClickDist[i])
					{
						minLine2ClickDis = line2ClickDist[i];
					}
				}

				double userDistU = minLine2ClickDis;


				// ---------------------------------------------------------------------
				// compute the score for each isoline, reject if neccessary
				// ---------------------------------------------------------------------
				double[] proximity = new double[n];
				double[] concavity = new double[n];
				double[] tightness = new double[n];
				for (int i = 0; i < n; ++i)
				{
					double r = list[i].radious; double r1 = r, r2 = r;
					
					// ---------------------------------------------------------------------
					// reject the isoline if its radii is large than the sudden jump one.
					// ---------------------------------------------------------------------
					if (!list[i].isValidElector)
					{
						continue;
					}


					if (i > 0) r1 = list[i - 1].radious;
					if (i < n - 1) r2 = list[i + 1].radious;


					double dd = (line2ClickDist[i] - userDistU);
					proximity[i] = dd;


					// ---------------------------------------------------------------------
					// concavity of the isoline -- multi-scale
					// ---------------------------------------------------------------------
					if (i == 0 || i == n-1)
					{
						concavity[i] = (2 * r - r2 - r1) / maxRadius;
					}
					else
					{
						int s = i - 1, t = i + 1, count = 0;
						double w_sum = 0;
						while (s >= 0 && t < n)
						{
							r1 = list[s].radious;
							r2 = list[t].radious;
							double w = Math.Exp(-(double)count * count / (double)(4 * 2));
							concavity[i] += (2 * r - r1 - r2) / maxRadius * w;
							s--; t++; count++;
							w_sum += w;
						}
						concavity[i] /= w_sum;
					}

					// ---------------------------------------------------------------------
					// tightness of the isoline
					// ---------------------------------------------------------------------

					tightness[i] = r - tightU;

				}

				// normalization
				double maxProximity = double.MinValue;
				double maxConcavity = double.MinValue;
				double maxTightness = double.MinValue;
				for (int i = 0; i < n; ++i)
				{
					if (maxProximity < proximity[i])
					{
						maxProximity = proximity[i];
					}
					if (maxTightness < tightness[i])
					{
						maxTightness = tightness[i];
					}

					double absConcavity = Math.Abs(concavity[i]);
					if (maxConcavity < absConcavity)
					{
						maxConcavity = absConcavity;
					}
				}


				double epsilon = 1e-5;
				for (int i = 0; i < n; ++i)
				{
					proximity[i] /= (maxProximity+epsilon);
					concavity[i] /= (maxConcavity+epsilon);
					tightness[i] /= (maxTightness+epsilon);
				}

				for (int i = 0; i < n; ++i)
				{
					Isoline line = list[i];

					if (!line.isValidElector) continue;

					double tc = (1.0 + concavity[i]) / 2;

					double tt = F(proximity[i]) * F(tightness[i]) * F(tc);

					list[i].score = tt;
				}
			}
		}
		private double F(double x)
		{
			double d = 1.0 / (1.0 + x * x);
			return d;
		}
		private void VoteBestCut()
		{
			/// do isoline score voting on mesh faces
			this.faceScores = new double[mesh.FaceCount];
			foreach (List<Isoline> list in this.isolinesSets)
			{
				foreach (Isoline line in list)
				{
					double sum = line.radious;   
					foreach (FaceRecord rec in line.faces)
					{
						int f = rec.face.index;
						
						double len = (rec.pq.p - rec.pq.q).Length();

						faceScores[f] += line.score;/// *len / sum;

					}
				}
			}

			/// compute face gradient
			this.faceGradient = new double[mesh.FaceCount]; int index = 0;
			foreach (List<Isoline> list in this.isolinesSets)
			{
				double[] scalarField = this.scalarFields[index++];
				bool[] computed = new bool[mesh.FaceCount];
				foreach (Isoline line in list)
				{
					foreach (FaceRecord rec in line.faces)
					{
						int f = rec.face.index;
						if (!computed[f])
						{
							faceGradient[f] += this.ComputeFaceGradient(scalarField, f);

							computed[f] = true;
						}
					}
				}
			}
			NormalizeData(faceGradient);

			/// recompute isoline scores
			foreach (List<Isoline> list in this.isolinesSets)
			{
				foreach (Isoline line in list)
				{
					if (!line.isValidElector)
					{
						line.score = 0;
						continue;
					}

					double score = 0;
					foreach (FaceRecord rec in line.faces)
					{
						int f = rec.face.index;
						score += faceScores[f] * faceGradient[f];
					}
					line.score = score / line.faces.Count;
				}
			}


			// ---------------------------------------------------------
			// select the isoline with best score as the cut		
			// ---------------------------------------------------------

			int currSetIndex = -1;
			
			double maxScore = double.MinValue; Isoline best = null; int count = 0;
			foreach (List<Isoline> list in this.isolinesSets)
			{
				foreach (Isoline line in list)
				{
					if (line.score > maxScore)
					{
						maxScore = line.score;
						best = line;

						currSetIndex = count;
					}
				}
				++count;
			}

			/// record this, in case the user wants to update the cut position.
	//		Program.PrintText("Best field index: " + currSetIndex.ToString());
	//		Program.PrintText("Best isoline score: " + best.score.ToString());
	//		if (best.isPatchType && !best.isValidElector) Program.PrintText("best is a patch type!");
			this.currBestVertexPair = this.vertexPairs[currSetIndex];
			this.currBestCutValue = best.value;

			// test 
			bool test = false;
			if (test && best != null)
			{
				double maxdis = double.MinValue;
				foreach (VertexPair pair in this.vertexPairs)
				{
					Vector3d p = new Vector3d(mesh.VertexPos,pair.I*3);
					Vector3d q = new Vector3d(mesh.VertexPos,pair.J*3);
					double minpdis = double.MaxValue;
					double minqdis = double.MaxValue;
					foreach (FaceRecord rec in best.faces)
					{
						double pdis1 = (rec.pq.p - p).Length();
						double pdis2 = (rec.pq.q - p).Length();
						double pdis = pdis1 > pdis2 ? pdis2 : pdis1;

						double qdis1 = (rec.pq.p - q).Length();
						double qdis2 = (rec.pq.q - q).Length();
						double qdis = qdis1 > qdis2 ? qdis2 : qdis1;

						if (pdis < minpdis)
						{
							minpdis = pdis;
						}
						if (qdis < minqdis)
						{
							minqdis = qdis;
						}
					}

					double dist = (minpdis + minqdis)/2;
					if (dist > maxdis)
					{
						this.currBestVertexPair = pair;
						maxdis = dist;
					}
				}

				currSetIndex = this.vertexPairs.IndexOf(this.currBestVertexPair);
				maxScore = double.MinValue;
				foreach (Isoline line in this.isolinesSets[currSetIndex])
				{
					if (line.score > maxScore)
					{
						maxScore = line.score;
						best = line;
					}
				}
				

				this.currBestCutValue = best.value;

				Program.PrintText("Best field index: " + currSetIndex.ToString());
				Program.PrintText("Best isoline value: " + best.value.ToString());
			}



			if (best != null)
			{

				this.currBestCut = best;
				this.bestCuts.Add(best);

				this.ObtainNewBoundary(best);
				this.FloodFill();
				this.AssignComponentsColors();
			}

		}



		public class BoundaryCurve
		{
			public List<int> vertices = null;
			public List<int> smoothPoints = null;
			public int index = -1;
			public BoundaryCurve(List<int> vtxs)
			{
				this.vertices = vtxs;
			}

			public List<int> walkingFaces = null;
		}
		private List<BoundaryCurve> boundaryCurves = new List<BoundaryCurve>();

		private int[] faceRegionIndex = null;
		private List<List<int>> facePatches = new List<List<int>>();
		private List<List<int>> components = null;
		private int[] componentIndex = null;
		/// update harmonic field ...
		private VertexPair currBestVertexPair = null;
		private List<int> multiClickVertices = new List<int>();
		private List<int> prevFixedVertices = new List<int>();
		private List<int> currFixedVertices = new List<int>();
		private double currBestCutValue = 0;


		private void ObtainNewBoundary(Isoline line)
		{
			// ----------------------------------------------------------------
			// assign boundary faces that lies on the isoline
			// ----------------------------------------------------------------
			bool[] isOnlineFace = new bool[mesh.FaceCount];
			foreach (FaceRecord rec in line.faces)
			{
				int f = rec.face.index;
				isOnlineFace[f] = true;
			}

			// ----------------------------------------------------------------
			// collects edges that lies on the boundary
			// ----------------------------------------------------------------
			List<Edge> edjes = new List<Edge>(); Edge prevE = null;
			foreach (FaceRecord rec in line.faces)
			{
				int j = rec.face.index;
				int b = j * 3; bool find = false;
				for (int i = 0; i < 3; ++i)
				{
					int c1 = mesh.FaceIndex[b+i];
					int c2 = mesh.FaceIndex[b+(i+1)%3];

					Edge e = new Edge(c1,c2);

					foreach (int f in edgeAdjFaces[e])
					{
						if (f != j && !isOnlineFace[f])
						{
							if (prevE != null)
							{
								if (prevE.I == c1 || prevE.J == c1 || // the same side edge
									prevE.I == c2 || prevE.J == c2)
								{
									edjes.Add(e);
									prevE = e;
								}
							}
							else
							{
								edjes.Add(e);
								prevE = e;
							}

							find = true;
							break;
						}
					}

					if (find) break;

				}
			}

			if (edjes.Count < 2) return;


			// ----------------------------------------------------------------
			// collects vertices from one-ring boundary edges
			// ----------------------------------------------------------------
			List<int> oneRingVertices = new List<int>();
			int startVetex = -1;

			Edge e0 = edjes[0], e1 = edjes[1];
			if (e0.I == e1.I || e0.I == e1.J)
			{
				startVetex = e0.J;
			}
			else
			{
				startVetex = e0.I;
			}

			oneRingVertices.Add(startVetex);
			foreach (Edge e in edjes)
			{
				if (e.I == startVetex)
				{
					oneRingVertices.Add(e.J);
					startVetex = e.J;
				}
				else
				{
					oneRingVertices.Add(e.I);
					startVetex = e.I;
				}
			}
			oneRingVertices.RemoveAt(oneRingVertices.Count - 1);

			BoundaryCurve cur = new BoundaryCurve(oneRingVertices);
			line.boundaryCurve = cur;
			this.boundaryCurves.Add(cur);

		}
		private void GetCommonIndices(int i, int j, out int index1, out int index2)
		{
			index1 = -1; index2 = -1;
			
			int b = i * 3, c = j * 3;
			for (int k = 0; k < 3; ++k)
			{
				int bk = mesh.FaceIndex[b + k];
				for (int l = 0; l < 3; ++l)
				{
					int cl = mesh.FaceIndex[c + l];
					if (bk == cl)
					{
						if (index1 == -1)
						{
							index1 = bk;
						}
						else
						{
							index2 = bk;
						}
						break;
					}
				}
			}
		}
		private void GetDiffIndices(int i, int j, out int index1, out int index2)
		{
			index1 = -1; index2 = -1;
			
			int b = i * 3, c = j * 3;
			List<int> ISet = new List<int>();
			List<int> JSet = new List<int>();
			for (int k = 0; k < 3; ++k)
			{
				ISet.Add(mesh.FaceIndex[b+k]);
				JSet.Add(mesh.FaceIndex[c+k]);
			}
			List<int> diff = ISet.Intersect(JSet).ToList();

			index1 = diff[0];
			index2 = diff[1];
		}
		private void FloodFill()
		{
			if (this.componentIndex == null)
			{
				this.componentIndex = new int[mesh.VertexCount];
			}

			bool[] isboundary = new bool[mesh.VertexCount];
			foreach (Isoline line in this.bestCuts)
			{
				if (line.lineAsCut)
				{
					foreach (FaceRecord rec in line.faces)
					{
						int f = rec.face.index * 3;
						for (int i = 0; i < 3; ++i)
						{
							int c = mesh.FaceIndex[f + i];
							isboundary[c] = true;
						}
					}
				}
				else // using the boundary curve
				{
					BoundaryCurve curve = line.boundaryCurve;
					foreach (int v in curve.vertices)
					{
						isboundary[v] = true;
					}
				}
			}

			int prevNumComponents = this.components == null ? 1 : this.components.Count;

			this.components = new List<List<int>>();

			bool[] visited = new bool[mesh.VertexCount]; int index = 0;
			int[] tag = new int[mesh.VertexCount];
			for (int i = 0; i < mesh.VertexCount; ++i)
			{
				if (!isboundary[i] && !visited[i])
				{
					List<int> vertices = new List<int>();
					Queue<int> Q = new Queue<int>();
					vertices.Add(i);
					Q.Enqueue(i);
					visited[i] = true;

					while (Q.Count > 0)
					{
						int s = Q.Dequeue();
						foreach (int j in mesh.AdjVV[s])
						{
							if (!visited[j])
							{
								visited[j] = true;
								vertices.Add(j);
								if (!isboundary[j])
								{
									Q.Enqueue(j);
								}
							}
						}
					}

					foreach (int v in vertices)
						tag[v] = index;
					index++;

					this.components.Add(vertices);
				}
			}

			for (int i = 0; i < mesh.VertexCount; ++i)
			{
				if (!visited[i])
				{
					foreach (int j in mesh.AdjVV[i])
					{
						this.components[tag[j]].Add(i);
						visited[i] = true;
						break;
					}
				}
			}


			int k = this.components.Count;
			if (k == 1) return;

			int currSelectedComponent = componentIndex[this.vertexPairs[0].I];


			List<List<int>> affectedComponents = new List<List<int>>();
			foreach (List<int> component in this.components)
			{
				foreach (int j in component)
				{
					if (isboundary[j]) continue;
					if (componentIndex[j] == currSelectedComponent)
					{
						affectedComponents.Add(component);
						break;
					}
				}
			}

			if (affectedComponents.Count > 0)
			{
				List<int> component = null;
				if (affectedComponents.Count >= 3)
				{
					int max = int.MinValue, secMax = int.MinValue;
					int maxId = -1, secMaxId = -1, count = 0;
					foreach (List<int> comp in affectedComponents)
					{
						if (max < comp.Count)
						{
							max = comp.Count;
							maxId = count;
						}
						++count;
					}
					count = 0;
					foreach (List<int> comp in affectedComponents)
					{
						if (count == maxId)
						{
							++count;
							continue;
						}
						if (secMax < comp.Count)
						{
							secMax = comp.Count;
							secMaxId = count;
						}
						++count;
					}

					component = affectedComponents[secMaxId];
				}
				else if (affectedComponents.Count == 2)
				{
					if (affectedComponents[0].Count < affectedComponents[1].Count)
					{
						component = affectedComponents[0];
					}
					else
					{
						component = affectedComponents[1];
					}
				}

				if (component != null)
				{
					foreach (int j in component)
					{
						this.componentIndex[j] = k - 1;
					}
				}
			}

			List<List<int>> newCompoennts = new List<List<int>>();
			for (int i = 0; i < k; ++i)
			{
				newCompoennts.Add(new List<int>());
			}
			foreach (List<int> compnt in this.components)
			{
				foreach (int j in compnt)
				{
					if (isboundary[j]) continue;

					int J = this.componentIndex[compnt[0]];
					newCompoennts[J].AddRange(compnt);
					break;
				}
			}

			this.components = newCompoennts;

			GC.Collect();
		}
		private void FloodFill_FaceBased()
		{
			// --------------------------------------------------------------
			// update the boundary curve first.
			// --------------------------------------------------------------
		//	this.BoundaryCurveSelfUpdate();

			// --------------------------------------------------------------
			// flood fill by faces.
			// --------------------------------------------------------------
			bool[] isCurveVertex = new bool[mesh.VertexCount];
			foreach (BoundaryCurve curve in this.boundaryCurves)
			{
				foreach (int v in curve.vertices)
				{
					isCurveVertex[v] = true;
				}
			}

			List<int> suckFaces = new List<int>();

			bool[] isboundary = new bool[mesh.FaceCount];
			foreach (BoundaryCurve curve in this.boundaryCurves)
			{
				foreach (int v in curve.vertices)
				{
					foreach (int f in mesh.AdjVF[v])
					{
						int count = 0, b = f*3;
						for (int j = 0; j < 3; ++j)
						{
							int vId = mesh.FaceIndex[b+j];
							if (isCurveVertex[vId])
							{
								count++;
							}
						}
						if (count >= 2)
						{
							isboundary[f] = true;
							if (count == 3)
							{
								suckFaces.Add(f);
							}
						}
					}
				}
			}

			this.facePatches.Clear();

			bool[] visited = new bool[mesh.FaceCount]; int index = 0;
			int[] tag = new int[mesh.FaceCount];
			for (int i = 0; i < mesh.FaceCount; ++i)
			{
				if (!isboundary[i] && !visited[i])
				{
					List<int> faces = new List<int>();
					Queue<int> Q = new Queue<int>();
					faces.Add(i);
					Q.Enqueue(i);
					visited[i] = true;

					while (Q.Count > 0)
					{
						int s = Q.Dequeue();
						foreach (int j in mesh.AdjFF[s])
						{
							if (!visited[j])
							{
								visited[j] = true;
								faces.Add(j);
								if (!isboundary[j])
								{	
									Q.Enqueue(j);
								}
							}
						}
					}

					foreach (int f in faces)
						tag[f] = index;
					
					index++;

					this.facePatches.Add(faces);

				}
			}

			for (int i = 0; i < mesh.FaceCount; ++i)
			{
				if (!visited[i])
				{
					foreach (int j in mesh.AdjFF[i])
					{
						this.facePatches[tag[j]].Add(i);
						visited[i] = true;
						break;
					}
				}
			}

			int regionIndex = 0;
			this.faceRegionIndex = new int[mesh.FaceCount];
			foreach (List<int> patch in this.facePatches)
			{
				foreach (int f in patch)
				{
					faceRegionIndex[f] = regionIndex;
				}
				regionIndex++;
			}

			GC.Collect();

		}
		private void UpdateComponentByIndex()
		{
			if (this.componentIndex == null) return;

			int n = this.componentIndex.Length;
			int max = int.MinValue;
			for (int i = 0; i < n; ++i)
			{
				if (componentIndex[i] > max)
				{
					max = componentIndex[i];
				}
			}

			int k = max + 1;
			this.components = new List<List<int>>();
			for (int i = 0; i < k; ++i)
			{
				this.components.Add(new List<int>());
			}
			int index = 0;
			foreach (int j in this.componentIndex)
			{
				this.components[j].Add(index++);
			}
		}
		public void AssignComponentsColors()
		{
			int index = 0;
			Color[] colors = Program.displayProperty.ColorMall;

			int sum = 0;

			this.vtxDisplayColor = new double[mesh.VertexCount*3];
			foreach (List<int> vertices in this.components)
			{
				sum += vertices.Count;

				Color c = colors[(index++) % 33];
				double r = c.R / 255.0;
				double g = c.G / 255.0;
				double b = c.B / 255.0;
				foreach (int v in vertices)
				{
					int j = v * 3;
					this.vtxDisplayColor[j] = r;
					this.vtxDisplayColor[j+1] = g;
					this.vtxDisplayColor[j+2] = b;
				}
			}

			if (sum != mesh.VertexCount)
			{
				Program.PrintText("Something vertices are missing ...");
			}
		}
		public void ComputeFaceGradient()
		{
			int n = mesh.FaceCount; this.gradientFields = new List<double[]>();
			foreach (double[] h in this.scalarFields)
			{
				double[] gradient = new double[n];
				for (int i = 0, j = 0; i < n; ++i, j+=3)
				{
					int c1 = mesh.FaceIndex[j];
					int c2 = mesh.FaceIndex[j + 1];
					int c3 = mesh.FaceIndex[j + 2];

					Vector3d v1 = new Vector3d(mesh.VertexPos, c1 * 3);
					Vector3d v2 = new Vector3d(mesh.VertexPos, c2 * 3);
					Vector3d v3 = new Vector3d(mesh.VertexPos, c3 * 3);

					double d1 = h[c1];
					double d2 = h[c2];
					double d3 = h[c3];

					/// find closest point v on V21 to V3
					double r = (v3 - v1).Dot(v2 - v1) / (v2 - v1).Dot(v2 - v1);
					Vector3d v = v1 + r * (v2 - v1);
					double d = d1 + r * (d2 - d1);

					/// compute gradient vector1 
					Vector3d gv1 = ((v - v3) / (v - v3).Length()) * ((d - d3) / (v - v3).Length());
					Vector3d gv2 = ((v - v1) / (v - v1).Length()) * ((d - d1) / (v - v1).Length());


					double g = (gv1 + gv2).Length() / this.triArea[i];
					g = Math.Pow(g, opt.GradientScale);

					if (double.IsNaN(g))
					{
						g = 1e-5;
					}

					gradient[i] = g;
				}

				NormalizeData(gradient);

				this.gradientFields.Add(gradient);
			}
		}
		public void AssignPatchColors()
		{
			if (this.triDisplayColor == null) this.triDisplayColor = new double[mesh.FaceCount*3];

			Color[] colors = Program.displayProperty.ColorMall; int index = 0;
			foreach (List<int> patch in this.facePatches)
			{
				Color c = colors[(index++) % 33];
				foreach (int v in patch)
				{
					int j = v * 3;
					this.triDisplayColor[j] = c.R / 255.0;
					this.triDisplayColor[j + 1] = c.G / 255.0;
					this.triDisplayColor[j + 2] = c.B / 255.0;
				}
			}
		}


		private void UpdateHarmonicField(VertexPair pair)
		{
			if (this.sparseSolver == null || pair == null) return;

			int I = pair.I, J = pair.J;
			int prevI = this.prevPair.I, prevJ = this.prevPair.J;

			if (I < 0 || J < 0) return;

			int n = mesh.VertexCount;


			// -------------------------------------------------------------
			// update factorization, A + WW^T
			// -------------------------------------------------------------
			this.sparseSolver.InitializeMatrixC(n,n);

			sparseSolver.AddCoef_C(I, I, 1.0 * ConstantWeight);
			sparseSolver.AddCoef_C(J, J, 1.0 * ConstantWeight);

			if (!sparseSolver.UpdateFactorization(true))
			{
				Program.PrintText("fail to update factorization ...\n");
			}


			// -------------------------------------------------------------
			// downdate factorization, A - WW^T
			// -------------------------------------------------------------
			sparseSolver.AddCoef_C(I, I, -1.0 * ConstantWeight);
			sparseSolver.AddCoef_C(J, J, -1.0 * ConstantWeight);
	
			sparseSolver.AddCoef_C(prevI, prevI, 1.0 * ConstantWeight);
			sparseSolver.AddCoef_C(prevJ, prevJ, 1.0 * ConstantWeight);

			/// remove the prev user multi-click
			foreach (int v in this.multiClickVertices)
			{
				sparseSolver.AddCoef_C(v, v, 1.0 * ConstantWeight);
			}
			
			if (!sparseSolver.UpdateFactorization(false))
			{
				Program.PrintText("fail to downdate factorization ...\n");
			}

			// directly modify "b"
			sparseSolver.SetCoef_B(prevI, 0, 0.0);
			sparseSolver.SetCoef_B(I, 0, 1.0 * ConstantWeight * ConstantWeight);
			foreach (int v in this.multiClickVertices)
			{
				sparseSolver.SetCoef_B(v, 0, 0);
			}


			double[] x = new double[n];
			fixed (double* _x = x)
			{
				sparseSolver.Linear_Solve(_x, !opt.SolverUseAtb);
			}

			this.currHarmonicField = x;

		}
		

		public class Edge
		{
			public int I = -1;
			public int J = -1;
			public Edge(int i, int j)
			{
				if (i > j) { int tmp = i; i = j; j = tmp; } /// make sure i < j

				this.I = i;
				this.J = j;
			}
			
			public override bool Equals(object obj)
			{
				Edge rec = obj as Edge;
				return I == rec.I && J == rec.J;
			}
			public override int GetHashCode()
			{
				return I + J;
			}
		}
		private HashSet<Edge> meshEdges = null;
		private Dictionary<Edge, double> edgeLength = null;
		private Dictionary<Edge, double> edgeDihedralAngle = null;
		private Dictionary<Edge, double> edgeCost = null;
		private Dictionary<Edge, List<int>> edgeAdjFaces = null;
		private double avgEdgeLength = 0;
		private void ComputeEdgesInfo()
		{
			/// find all the edges
			this.meshEdges = new HashSet<Edge>();
			for (int i = 0, j = 0; i < mesh.FaceCount; ++i,j+=3)
			{
				int c0 = mesh.FaceIndex[j];
				int c1 = mesh.FaceIndex[j+1];
				int c2 = mesh.FaceIndex[j+2];

				Edge e0 = new Edge(c0, c1);
				Edge e1 = new Edge(c1, c2);
				Edge e2 = new Edge(c2, c0);

				this.meshEdges.Add(e0);
				this.meshEdges.Add(e1);
				this.meshEdges.Add(e2);
			}

			this.edgeAdjFaces = new Dictionary<Edge, List<int>>();
			foreach (Edge e in this.meshEdges)
			{
				List<int> adjFaces = mesh.AdjVF[e.I].Intersect(mesh.AdjVF[e.J]).ToList();
				this.edgeAdjFaces.Add(e, adjFaces);
			}

			/// compute the length & dihedral angle for each edge
			int nedges = this.meshEdges.Count;
			this.edgeLength = new Dictionary<Edge, double>();
			this.edgeDihedralAngle = new Dictionary<Edge, double>();
			this.edgeCost = new Dictionary<Edge, double>();

			this.avgEdgeLength = 0;
			foreach (Edge e in this.meshEdges)
			{
				Vector3d u = new Vector3d(mesh.VertexPos, e.I * 3);
				Vector3d v = new Vector3d(mesh.VertexPos, e.J * 3);
				double len = (u-v).Length();
				this.edgeLength.Add(e, len);

				this.avgEdgeLength += len;

				double cos = 1;
				if (this.edgeAdjFaces[e].Count == 2)
				{
					int s = this.edgeAdjFaces[e][0], t = this.edgeAdjFaces[e][1];
					
					Vector3d n1 = new Vector3d(mesh.FaceNormal, s * 3);
					Vector3d n2 = new Vector3d(mesh.FaceNormal, t * 3);

					int index1, index2;
					this.GetDiffIndices(s, t, out index1, out index2);

					Vector3d c = new Vector3d(mesh.DualVertexPos, s * 3);
					Vector3d p = new Vector3d(mesh.VertexPos, index2 * 3);

					bool concv = (p - c).Dot(n1) > 0;

					cos = Math.Abs(n1.Dot(n2));

					if (!concv) cos = 1;

					this.edgeDihedralAngle.Add(e, cos);
				}

				double cost = len*Math.Pow(cos, 2);
				this.edgeCost.Add(e, cost);
			}

			this.avgEdgeLength /= this.meshEdges.Count;
		}
		
		#endregion


		#region load/save interfaces ...
		public void SaveSegmentation(StreamWriter sw)
		{
			if (this.faceRegionIndex == null)
			{
				Program.PrintText("can not save segmentation file --> faceregionindex = null");
				this.FloodFill_FaceBased();
				this.AssignPatchColors();
				this.ShowPatches = true;
			}
			for (int i = 0; i < mesh.FaceCount; ++i)
			{
				sw.WriteLine(this.faceRegionIndex[i].ToString());
			}
		}
		public void LoadSegmentation(StreamReader sr)
		{
			this.faceRegionIndex = new int[mesh.FaceCount];

			char[] delimiters = { ' ', '\t' };
			string s = "";

			int count = 0;
			while (sr.Peek() > -1)
			{
				s = sr.ReadLine();
				string[] tokens = s.Split(delimiters);
				int index = int.Parse(tokens[0]);

				this.faceRegionIndex[count++] = index;
			}

			this.facePatches.Clear(); count = 0;
			for (int i = 0; i < mesh.FaceCount; ++i)
			{
				if (count < faceRegionIndex[i])
				{
					count = faceRegionIndex[i];
				}
			}
			for (int i = 0; i <= count; ++i)
			{
				this.facePatches.Add(new List<int>());
			}
			for (int i = 0; i < mesh.FaceCount; ++i)
			{
				int index = faceRegionIndex[i];
				this.facePatches[index].Add(i);
			}


			bool[] assigned = new bool[mesh.VertexCount];
			this.components = new List<List<int>>();
			foreach (List<int> patch in this.facePatches)
			{
				List<int> componnet = new List<int>();
				foreach (int f in patch)
				{
					int b = f * 3;
					for (int i = 0; i < 3; ++i)
					{
						int c = mesh.FaceIndex[b + i];
						if (!assigned[c])
						{
							componnet.Add(c);
							assigned[c] = true;
						}
					}
				}
				this.components.Add(componnet);
			}

			this.AssignComponentsColors();

			this.AssignPatchColors();
			this.ShowPatches = true;
		}
		#endregion


		#region rendering
		private bool ShowCuts = true;

		public void Display()
		{
			DrawField();
			
			if (ShowCuts) { DrawIsolines(this.bestCuts, Color.Red, 2.0f); }
			if (ShowPatches) { DrawPatches(); }

		}

		private void DrawIsolines(List<Isoline> list, Color linecolor, float linewidth)
		{
			if (list == null || list.Count == 0) return;

			double[] pos = mesh.VertexPos;
			double[] nor = mesh.VertexNormal;

			GL.glPushAttrib(GL.GL_LINE_BIT | GL.GL_ENABLE_BIT);
			GL.glLineWidth(linewidth);
			GL.glEnable(GL.GL_LINE_SMOOTH);
			GL.glEnable(GL.GL_LIGHTING);
			GL.glEnable(GL.GL_NORMALIZE);
			GL.glPolygonMode(GL.GL_FRONT_AND_BACK, GL.GL_LINE);
			GL.glDisable(GL.GL_POLYGON_OFFSET_FILL);
			GL.glEnable(GL.GL_CULL_FACE);
			GL.glColor3ub(linecolor.R, linecolor.G, linecolor.B);
			GL.glBegin(GL.GL_LINES);
			foreach (Isoline line in list)
			{
				if (line.lineAsCut)
				{
					foreach (FaceRecord rec in line.faces)
					{
						Vector3d p = rec.pq.p, q = rec.pq.q, normal = rec.pq.n;
						GL.glNormal3d(normal.x, normal.y, normal.z);
						GL.glVertex3d(p.x, p.y, p.z);
						GL.glVertex3d(q.x, q.y, q.z);
					}
				}
				else
				{
					BoundaryCurve c = line.boundaryCurve;
					int n = c.vertices.Count;
					for (int i = 0; i < n; ++i)
					{
						int s = c.vertices[i]*3, t = c.vertices[(i + 1) % n]*3;
						
						GL.glNormal3d(nor[s],nor[s+1],nor[s+2]);
						GL.glVertex3d(pos[s],pos[s+1],pos[s+2]);
						GL.glVertex3d(pos[t],pos[t+1],pos[t+2]);
					}
				}
			}

			GL.glEnd();
			GL.glPopAttrib();
			GL.glEnable(GL.GL_POLYGON_OFFSET_FILL);
			GL.glEnable(GL.GL_CULL_FACE);
			GL.glDisable(GL.GL_LIGHTING);
			GL.glDisable(GL.GL_NORMALIZE);

		}
		private void DrawIsoline(Isoline line, Color linecolor)
		{
			GL.glPushAttrib(GL.GL_LINE_BIT | GL.GL_ENABLE_BIT);
			GL.glLineWidth(2.0f);
			GL.glEnable(GL.GL_LINE_SMOOTH);
			GL.glEnable(GL.GL_LIGHTING);
			GL.glEnable(GL.GL_NORMALIZE);
			GL.glPolygonMode(GL.GL_FRONT_AND_BACK, GL.GL_LINE);
			GL.glDisable(GL.GL_POLYGON_OFFSET_FILL);
			GL.glEnable(GL.GL_CULL_FACE);
			GL.glColor3ub(linecolor.R, linecolor.G, linecolor.B);
			GL.glBegin(GL.GL_LINES);
				foreach (FaceRecord rec in line.faces)
				{
					Vector3d p = rec.pq.p, q = rec.pq.q, normal = rec.pq.n;
					GL.glNormal3d(normal.x, normal.y, normal.z);
					GL.glVertex3d(p.x, p.y, p.z);
					GL.glVertex3d(q.x, q.y, q.z);
				}
			GL.glEnd();
			GL.glPopAttrib();
			GL.glEnable(GL.GL_POLYGON_OFFSET_FILL);
			GL.glEnable(GL.GL_CULL_FACE);
			GL.glDisable(GL.GL_LIGHTING);
			GL.glDisable(GL.GL_NORMALIZE);
		}
		private void DrawField()
		{
			if (this.vtxDisplayColor == null || !this.started) return;

			GL.glShadeModel(GL.GL_SMOOTH);
			GL.glPolygonMode(GL.GL_FRONT_AND_BACK, GL.GL_FILL);
			GL.glEnable(GL.GL_LIGHTING);
			GL.glEnable(GL.GL_NORMALIZE);
			GL.glEnableClientState(GL.GL_VERTEX_ARRAY);
			GL.glEnableClientState(GL.GL_COLOR_ARRAY);
			GL.glEnableClientState(GL.GL_NORMAL_ARRAY);
			fixed (double* vp = mesh.VertexPos, cp = this.vtxDisplayColor, np = mesh.VertexNormal)
			fixed (int* index = mesh.FaceIndex)
			{
				GL.glVertexPointer(3, GL.GL_DOUBLE, 0, vp);
				GL.glColorPointer(3, GL.GL_DOUBLE, 0, cp);
				GL.glNormalPointer(GL.GL_DOUBLE, 0, np);
				GL.glDrawElements(GL.GL_TRIANGLES, mesh.FaceCount * 3, GL.GL_UNSIGNED_INT, index);
			}
			GL.glDisableClientState(GL.GL_VERTEX_ARRAY);
			GL.glDisableClientState(GL.GL_COLOR_ARRAY);
			GL.glDisableClientState(GL.GL_NORMAL_ARRAY);
			GL.glDisable(GL.GL_LIGHTING);
		}
		private void DrawPatches()
		{
			GL.glShadeModel(GL.GL_FLAT);
			GL.glPolygonMode(GL.GL_FRONT_AND_BACK, GL.GL_FILL);
			GL.glEnable(GL.GL_LIGHTING);
			GL.glEnable(GL.GL_NORMALIZE);

			Mesh m = this.mesh;

			GL.glEnableClientState(GL.GL_VERTEX_ARRAY);
			fixed (double* vp = m.VertexPos)
			fixed (double* np = m.FaceNormal)
			{			
				GL.glVertexPointer(3, GL.GL_DOUBLE, 0, vp);
				GL.glBegin(GL.GL_TRIANGLES);
				for (int i = 0, j = 0; i < m.FaceCount; i++, j += 3)
				{
					GL.glColor3d(this.triDisplayColor[j], 
						this.triDisplayColor[j+1], this.triDisplayColor[j+2]);
					GL.glNormal3dv(np + j);
					GL.glArrayElement(m.FaceIndex[j]);
					GL.glArrayElement(m.FaceIndex[j + 1]);
					GL.glArrayElement(m.FaceIndex[j + 2]);
				}
				GL.glEnd();
			}
			GL.glDisableClientState(GL.GL_VERTEX_ARRAY);
			GL.glDisable(GL.GL_LIGHTING);
		}
		#endregion
	}
}
