﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using Emgu.CV;
using Emgu.CV.Structure;
using Emgu.CV.UI;
using MyGeometry;
using System.Drawing;


namespace i3_ImageManipulator
{
	public unsafe partial class ImageManipulator : IDisposable
	{
		// helper static functions for image manipulator
		public static double Distance2Line(Vector2d p, Vector2d u, Vector2d v)
		{
			Vector2d uv = (v - u).Normalize();
			Vector2d up = p - u;
			return (up - up.Dot(uv) * uv).Length();
		}
		public static double Distance2LineSegment(Vector2d p, Vector2d u, Vector2d v)
		{
			bool isProjectionIn = !((p - u).Dot(v - u) < 0 || (p - v).Dot(u - v) < 0);
			if (!isProjectionIn)
			{
				return Math.Min((p - u).Length(), (p - v).Length());
			}
			else
			{
				Vector2d uv = (v - u).Normalize();
				Vector2d up = p - u;
				return (up - up.Dot(uv) * uv).Length();
			}
		}
		public static double Distance2LineSegment(Vector3d p, Vector3d u, Vector3d v, out double r)
		{
			bool isProjectionIn = !((p - u).Dot(v - u) < 0 || (p - v).Dot(u - v) < 0);
			Vector3d uv = (v - u).Normalize();
			Vector3d up = p - u;
			double l = up.Dot(uv);
			r = l / (v - u).Length();
			if (!isProjectionIn)
			{
				return Math.Min(up.Length(), (p - v).Length());
			}
			else
			{
				return (up - l * uv).Length();
			}
		}
		public static double Distance2LineSegment(Vector2d p, Vector2d u, Vector2d v, out double projectionRatio)
		{
			Vector2d uv = (v - u).Normalize();
			Vector2d up = (p - u);
			double l = up.Dot(uv);
			projectionRatio = l / (v - u).Length();

			return (up - l * uv).Length();

		}
		public static double Point2LineAngleDistance(Vector2d p, Vector2d u, Vector2d v)
		{
			Vector2d c = (u + v) / 2;
			Vector2d cp = (p - c).Normalize();
			Vector2d uv = (v - u).Normalize();
			return 1.0 - Math.Abs(cp.Dot(uv));
		}
		public static Vector2d GetIntersectPoint(Vector2d u, Vector2d v, Vector2d p, Vector2d q)
		{
			Vector3d uu = new Vector3d(u, 1);
			Vector3d vv = new Vector3d(v, 1);
			Vector3d pp = new Vector3d(p, 1);
			Vector3d qq = new Vector3d(q, 1);

			Vector3d it = uu.Cross(vv).Cross(pp.Cross(qq));
			it.HomogenousNormalize();

			return it.ToVector2d();
		}
		private static Vector2d Point2Vecter2d(Point p)
		{
			return new Vector2d(p.X, p.Y);
		}
		private static Vector2d[] Point2Vecter2d(Point[] p)
		{
			int n = p.Length;
			Vector2d[] points = new Vector2d[n];
			for (int i = 0; i < n; ++i)
				points[i] = Point2Vecter2d(p[i]);
			return points;
		}
		private static Vector2d FindFootPrint(Vector2d p, Vector3d line)
		{
			// solve linear equations:
			// ax+by+x=0 && (-b,a).dot(x-e,y-f) = 0
			// => M = (a,b;-b,a), y = (-c,af-be).
			double a = line.x, b = line.y, c = line.z;
			double e = p.x, f = p.y;
			Matrix2d M = new Matrix2d(a, b, -b, a);
			Vector2d y = new Vector2d(-c, a * f - b * e);
			return M.Inverse() * y;
		}
		private static Bgr InterpolatePixelVectorBilinear(double sx, double sy, Image<Bgr, byte> src)
		{
			int minx = (int)sx;
			int miny = (int)sy;
			int maxx = (int)(sx + 1);
			int maxy = (int)(sy + 1);

			byte[, ,] data = src.Data;

			Vector3d c11 = new Vector3d(data[miny, minx, 0], data[miny, minx, 1], data[miny, minx, 2]);
			Vector3d c12 = new Vector3d(data[miny, maxx, 0], data[miny, maxx, 1], data[miny, maxx, 2]);
			Vector3d c21 = new Vector3d(data[maxy, minx, 0], data[maxy, minx, 1], data[maxy, minx, 2]);
			Vector3d c22 = new Vector3d(data[maxy, maxx, 0], data[maxy, maxx, 1], data[maxy, maxx, 2]);

			Vector3d c = (c11 * (maxx - sx) * (maxy - sy)
				+ c12 * (sx - minx) * (maxy - sy)
				+ c21 * (maxx - sx) * (sy - miny)
				+ c22 * (sx - minx) * (sy - miny)) / ((maxx - minx) * (maxy - miny));

			return new Bgr(c.x, c.y, c.z);
		}
		private static Bgr InterpolatePixelBilinear(double sx, double sy, Image<Bgr, byte> src)
		{
			int minx = (int)sx;
			int miny = (int)sy;
			int maxx = (int)(sx + 1);
			int maxy = (int)(sy + 1);

			double red = (src[miny, minx].Red * (maxx - sx) * (maxy - sy)
				+ src[miny, maxx].Red * (sx - minx) * (maxy - sy)
				+ src[maxy, minx].Red * (maxx - sx) * (sy - miny)
				+ src[maxy, maxx].Red * (sx - minx) * (sy - miny)) / ((maxx - minx) * (maxy - miny));
			double green = (src[miny, minx].Green * (maxx - sx) * (maxy - sy)
				+ src[miny, maxx].Green * (sx - minx) * (maxy - sy)
				+ src[maxy, minx].Green * (maxx - sx) * (sy - miny)
				+ src[maxy, maxx].Green * (sx - minx) * (sy - miny)) / ((maxx - minx) * (maxy - miny));
			double blue = (src[miny, minx].Blue * (maxx - sx) * (maxy - sy)
				+ src[miny, maxx].Blue * (sx - minx) * (maxy - sy)
				+ src[maxy, minx].Blue * (maxx - sx) * (sy - miny)
				+ src[maxy, maxx].Blue * (sx - minx) * (sy - miny)) / ((maxx - minx) * (maxy - miny));

			return new Bgr(blue, green, red);
		}
		private static Bgr InterpolatePixelBicubic(double sx, double sy, Image<Bgr, byte> src)
		{
			int minx = (int)sx;
			int miny = (int)sy;
			int maxx = (int)(sx + 1);
			int maxy = (int)(sy + 1);
			minx += 1;
			miny += 1;
			maxx += 1;
			maxy += 1;

			double[][] pointsred = new double[4][];
			pointsred[0] = new double[] { src[miny - 1, minx - 1].Red, src[miny - 1, minx].Red, src[miny - 1, maxx].Red, src[miny - 1, maxx + 1].Red };
			pointsred[1] = new double[] { src[miny, minx - 1].Red, src[miny, minx].Red, src[miny, maxx].Red, src[miny, maxx + 1].Red };
			pointsred[2] = new double[] { src[maxy, minx - 1].Red, src[maxy, minx].Red, src[maxy, maxx].Red, src[maxy, maxx + 1].Red };
			pointsred[3] = new double[] { src[maxy + 1, minx - 1].Red, src[maxy + 1, minx].Red, src[maxy + 1, maxx].Red, src[maxy + 1, maxx + 1].Red };
			double[][] pointsgreen = new double[4][];
			pointsgreen[0] = new double[] { src[miny - 1, minx - 1].Green, src[miny - 1, minx].Green, src[miny - 1, maxx].Green, src[miny - 1, maxx + 1].Green };
			pointsgreen[1] = new double[] { src[miny, minx - 1].Green, src[miny, minx].Green, src[miny, maxx].Green, src[miny, maxx + 1].Green };
			pointsgreen[2] = new double[] { src[maxy, minx - 1].Green, src[maxy, minx].Green, src[maxy, maxx].Green, src[maxy, maxx + 1].Green };
			pointsgreen[3] = new double[] { src[maxy + 1, minx - 1].Green, src[maxy + 1, minx].Green, src[maxy + 1, maxx].Green, src[maxy + 1, maxx + 1].Green };
			double[][] pointsblue = new double[4][];
			pointsblue[0] = new double[] { src[miny - 1, minx - 1].Blue, src[miny - 1, minx].Blue, src[miny - 1, maxx].Blue, src[miny - 1, maxx + 1].Blue };
			pointsblue[1] = new double[] { src[miny, minx - 1].Blue, src[miny, minx].Blue, src[miny, maxx].Blue, src[miny, maxx + 1].Blue };
			pointsblue[2] = new double[] { src[maxy, minx - 1].Blue, src[maxy, minx].Blue, src[maxy, maxx].Blue, src[maxy, maxx + 1].Blue };
			pointsblue[3] = new double[] { src[maxy + 1, minx - 1].Blue, src[maxy + 1, minx].Blue, src[maxy + 1, maxx].Blue, src[maxy + 1, maxx + 1].Blue };

			sx += 1;
			sy += 1;

			CachedBicubicInterpolator inter = new CachedBicubicInterpolator();
			inter.updateCoefficients(pointsred);
			double red = inter.getValue(sy - miny, sx - minx);
			inter.updateCoefficients(pointsgreen);
			double green = inter.getValue(sy - miny, sx - minx);
			inter.updateCoefficients(pointsblue);
			double blue = inter.getValue(sy - miny, sx - minx);

			return new Bgr(blue, green, red);
		}
		private static Vector3d Bgr2Vector3d(Bgr c)
		{
			return new Vector3d(c.Blue, c.Green, c.Red);
		}
		private static Bgr InterpolatePixelVectorBicubic(double sx, double sy, Bgr[,] src)
		{
			int minx = (int)sx;
			int miny = (int)sy;
			int maxx = (int)(sx + 1);
			int maxy = (int)(sy + 1);
			minx += 1;
			miny += 1;
			maxx += 1;
			maxy += 1;

			Vector3d[,] points = new Vector3d[4, 4] {
            { Bgr2Vector3d(src[miny - 1, minx - 1]), Bgr2Vector3d(src[miny - 1, minx]), Bgr2Vector3d(src[miny - 1, maxx]), Bgr2Vector3d(src[miny - 1, maxx + 1]) },
            { Bgr2Vector3d(src[miny, minx - 1]), Bgr2Vector3d(src[miny, minx]), Bgr2Vector3d(src[miny, maxx]), Bgr2Vector3d(src[miny, maxx + 1]) },
            { Bgr2Vector3d(src[maxy, minx - 1]), Bgr2Vector3d(src[maxy, minx]), Bgr2Vector3d(src[maxy, maxx]), Bgr2Vector3d(src[maxy, maxx + 1]) },
            { Bgr2Vector3d(src[maxy + 1, minx - 1]), Bgr2Vector3d(src[maxy + 1, minx]), Bgr2Vector3d(src[maxy + 1, maxx]), Bgr2Vector3d(src[maxy + 1, maxx + 1]) } };

			sx += 1;
			sy += 1;

			CachedVector3dBicubicInterpolator inter = new CachedVector3dBicubicInterpolator();
			inter.updateCoefficients(points);
			Vector3d color = inter.getValue(sy - miny, sx - minx);

			return new Bgr(color.x, color.y, color.z);
		}
		private static Quad GetPointsEnclosingQuad(List<Point2> points)
		{
			double maxx = -1, maxy = -1, minx = 1e8, miny = 1e8;
			foreach (Point2 p in points)
			{
				if (maxx < p.pos.x) maxx = p.pos.x;
				if (maxy < p.pos.y) maxy = p.pos.y;
				if (minx > p.pos.x) minx = p.pos.x;
				if (miny > p.pos.y) miny = p.pos.y;
			}
			Point2 p0 = new Point2(new Vector2d(minx, miny));
			Point2 p1 = new Point2(new Vector2d(maxx, miny));
			Point2 p2 = new Point2(new Vector2d(maxx, maxy));
			Point2 p3 = new Point2(new Vector2d(minx, maxy));
			return new Quad(p0, p1, p2, p3);
		}
		private static bool IsColinear(LineSegment l1, LineSegment l2)
		{
			Vector2d u = (l1.v.pos - l1.u.pos).Normalize();
			Vector2d v = (l2.v.pos - l2.u.pos).Normalize();
			double cos = Math.Abs(u.Dot(v));
			if (cos < 0.985) return false;
			Vector2d c = (l1.u.pos + l1.v.pos) / 2;
			double angledis = Point2LineAngleDistance(c, l2.u.pos, l2.v.pos);
			return angledis < 1e-3;
		}
		private static void FindVanishingIndex(Vector2d u, Vector2d v, Vector2d[] vps, out int index, out double dist)
		{
			// then find the cloest vp
			int n = vps.Length;
			double mindis = double.MaxValue; int minindex = -1;
			for (int i = 0; i < n; ++i)
			{
				Vector2d pt = vps[i];
				double dis = Point2LineAngleDistance(pt, u, v);
				if (dis < mindis)
				{
					mindis = dis;
					minindex = i;
				}
			}
			index = minindex;
			dist = mindis;
		}
		private static int FindVanishingIndex(Vector2d u, Vector2d v, Vector2d[] vps)
		{
			// then find the cloest vp
			int n = vps.Length;
			double mindis = double.MaxValue; int minindex = -1;
			for (int i = 0; i < n; ++i)
			{
				Vector2d pt = vps[i];
				double dis = Point2LineAngleDistance(pt, u, v);
				if (dis < mindis)
				{
					mindis = dis;
					minindex = i;
				}
			}
			return minindex;
		}
		private static Point Vector2Point(Vector2d v)
		{
			return new Point((int)v.x, (int)v.y);
		}
		public static void DeformProxy(PolyProxy proxy, Matrix4d T, int d)
		{
			if (proxy == null) return;

			Vector4d c = new Vector4d(proxy.center.oldpos, 0);
			foreach (Point3 p in proxy.points3)
			{
				Vector4d q = new Vector4d(p.oldpos, d);
				q = T * (q - c) + c;
				p.pos = q.XYZ();
			}
		}
		public static void DeformProxy(PolyProxy proxy, Matrix4d T, Vector3d o, int d)
		{
			if (proxy == null) return;

			Vector4d c = new Vector4d(o, 0);
			foreach (Point3 p in proxy.points3)
			{
				Vector4d q = new Vector4d(p.oldpos, d);
				q = T * (q - c) + c;
				p.pos = q.XYZ();
			}
		}
		public static void TruelyDeformProxy(PolyProxy proxy, Matrix4d T, Vector3d o, int d)
		{
			if (proxy == null) return;

			Vector4d c = new Vector4d(o, 0);
			foreach (Point3 p in proxy.points3)
			{
				Vector4d q = new Vector4d(p.pos, d);
				q = T * (q - c) + c;
				p.pos = q.XYZ();
			}
		}
		public static void DeformPolygon(Polygon plg, Matrix4d T, int d)
		{
			if (plg == null) return;
			Vector4d c = new Vector4d(plg.center.oldpos, 0);
			foreach (Point3 p in plg.points3)
			{
				Vector4d q = new Vector4d(p.oldpos, d);
				q = T * (q - c) + c;
				p.pos = q.XYZ();
			}
		}
		/// obtain reflectional maatrix, according to an arbitrary plane.
		public static Matrix4d ReflectionalMatrix(Vector3d plane_normal)
		{
			// create an coordinates sys, with plane_normal as x-axis
			Vector3d x = plane_normal;
			Vector3d y;
			if (x.x == 0 && x.y == 0)
				y = new Vector3d(1, 0, 0);
			else
				y = (new Vector3d(-x.y, x.x, 0)).Normalize();
			Vector3d z = x.Cross(y).Normalize();
			Matrix3d R = new Matrix3d(x, y, z).Transpose();
			Matrix3d InvR = R.Inverse();
			Matrix4d U = new Matrix4d(R);
			Matrix4d V = new Matrix4d(InvR);
			Matrix4d I = Matrix4d.IdentityMatrix();
			I[0, 0] = -1; // reflect matrix along yz plane
			return V * I * U;
		}
		public static Vector3d GetMirrorSymmetryPoint(Vector3d pt, Vector3d normal, Vector3d center)
		{
			Matrix4d R = ReflectionalMatrix(normal);
			return center + (R * new Vector4d(pt - center, 0)).XYZ();
		}
		public static Vector3d GetMirrorSymmetryVector(Vector3d vec, Vector3d normal)
		{
			Matrix4d R = ReflectionalMatrix(normal);
			return (R * new Vector4d(vec, 0)).XYZ();
		}
		public static PolyProxy CopyBox(PolyProxy box, Vector3d copydir, int distimes)
		{
			copydir = copydir.Normalize();
			Vector3d t = copydir * box.xlen;
			if (Math.Abs(copydir.Dot(box.frame.y)) > 0.8)
			{
				t = copydir * box.ylen;
			}
			else if (Math.Abs(copydir.Dot(box.frame.z)) > 0.8)
			{
				t = copydir * box.zlen;
			}
			PolyProxy newbox = box.Clone() as PolyProxy;
			Matrix4d T = Matrix4d.TranslationMatrix(t * distimes);
			DeformProxy(newbox, T, 1);
			foreach (Point3 p in newbox.points3)
			{
				p.UpdateOldPos();
			}
			newbox.Compute3DInfo();
			newbox.center.UpdateOldPos();
			return newbox;
		}
		public static Vector3d AnglesFromRotation(Matrix3d R)
		{
			//double angle_y =  Math.Asin(R[0,2]);        /* Calculate Y-axis angle */
			//double angle_x = 0, angle_z = 0;
			//double C  =  Math.Cos( angle_y );
			//if ( Math.Abs( C ) > 0.005 )             /* Gimball lock? */
			//{
			//    double trx = R[2, 2] / C;           /* No, so get X-axis angle */
			//    double ty = -R[1, 2] / C;
			//  angle_x  = Math.Atan2( ty, trx );
			//  trx = R[0, 0] / C;            /* Get Z-axis angle */
			//  ty = -R[0, 1] / C;
			//  angle_z  = Math.Atan2( ty, trx );
			//}
			//else                                 /* Gimball lock has occurred */
			//  {
			//  angle_x  = 0;                      /* Set X-axis angle to zero */
			//  double trx = R[1, 1];                 /* And calculate Z-axis angle */
			//  double ty = R[4, 0];
			//  angle_z  = Math.Atan2( ty, trx );
			//  }

			///* return only positive angles in [0,360] */
			//if (angle_x < 0) angle_x += 360;
			//if (angle_y < 0) angle_y += 360;
			//if (angle_z < 0) angle_z += 360;

			//return new Vector3d(angle_x, angle_y, angle_z);

			double alpha, alpha2, beta, beta2, gamma, gamma2;
			if (Math.Abs(R[2, 0]) != 1)
			{
				alpha = -Math.Asin(R[2, 0]);
				alpha2 = Math.PI - alpha;
				double cos1 = Math.Cos(alpha), cos2 = Math.Cos(alpha2);
				beta = Math.Atan2(R[2, 1] / cos1, R[2, 2] / cos1);
				beta2 = Math.Atan2(R[2, 1] / cos2, R[2, 2] / cos2);
				gamma = Math.Atan2(R[1, 0] / cos1, R[0, 0] / cos1);
				gamma2 = Math.Atan2(R[1, 0] / cos2, R[0, 0] / cos2);
			}
			else
			{
				gamma = 0;
				if (R[2, 0] == -1)
				{
					alpha = Math.PI / 2;
					beta = gamma + Math.Atan2(R[0, 1], R[0, 2]);
				}
				else
				{
					alpha = -Math.PI / 2;
					beta = -gamma + Math.Atan2(-R[0, 1], -R[0, 2]);
				}
			}
			return new Vector3d(beta, alpha, gamma);
		}
		public static Matrix3d Angles2RotationRhs(double a, double b, double r)
		{
			double cosa = Math.Cos(a), sina = Math.Sin(a);
			double cosb = Math.Cos(b), sinb = Math.Sin(b);
			double cosr = Math.Cos(r), sinr = Math.Sin(r);
			double r1 = cosb * cosr, r2 = sina * sinb * cosr - cosa * sinr, r3 = sina * sinr + cosa * sinb * cosr;
			double r4 = cosb * sinr, r5 = sina * sinb * sinr + cosa * cosr, r6 = -sina * cosr + cosa * sinb * sinr;
			double r7 = -sinb, r8 = sina * cosb, r9 = cosa * cosb;
			return new Matrix3d(new double[9] {
				r1,r2,r3,r4,r5,r6,r7,r8,r9});
		}
		public static Matrix3d Angles2RotationLhs(double a, double b, double r)
		{
			double c1 = Math.Cos(a), s1 = Math.Sin(a);
			double c2 = Math.Cos(b), s2 = Math.Sin(b);
			double c3 = Math.Cos(r), s3 = Math.Sin(r);
			double r1 = c2 * c3, r2 = -c2 * s3, r3 = s2;
			double r4 = c1*s3+c3*s1*s2, r5 = c1*c3-s1*s2*s3, r6 = -c2*s1;
			double r7 = s1 * s3 - c1 * c3 * s2, r8 = c3 * s1 + c1 * s2 * s3, r9 = c1 * c2;
			return new Matrix3d(new double[9] {
				r1,r2,r3,r4,r5,r6,r7,r8,r9});
		}
		public static Vector3d AnglesFromRotationLhs(Matrix3d R)
		{
			double alpha, beta, gamma;
			if (Math.Abs(R[0, 2]) != 1)
			{
				beta = Math.Asin(R[0, 2]);
				double cos = Math.Cos(beta);
				gamma = Math.Atan2(-R[0, 1] / cos, R[0, 0] / cos);
				alpha = Math.Atan2(-R[1, 2] / cos, R[2, 2] / cos);
			}
			else
			{
				gamma = 0;
				if (R[2, 0] == 1)
				{
					beta = Math.PI / 2;
					alpha = gamma + Math.Atan2(R[1, 0], R[1, 1]);
				}
				else
				{
					beta = -Math.PI / 2;
					alpha = -gamma + Math.Atan2(-R[1, 0], R[1, 1]);
				}
			}
			return new Vector3d(alpha, beta, gamma);
		}

		public static double Norm(double a, double b, double c, double d)
		{
			return Math.Sqrt(a * a + b * b + c * c + d * d);
		}
		public static Vector3d Rotation2Quaternion( Matrix3d a ) {
			Vector4d q = new Vector4d();
		  double trace = a[0,0] + a[1,1] + a[2,2]; // I removed + 1.0f; see discussion with Ethan
		  if( trace > 0 ) {// I changed M_EPSILON to 0
			 double s = 0.5 / Math.Sqrt(trace + 1.0);
			q.w = 0.25f / s;
			q.x = ( a[2,1] - a[1,2] ) * s;
			q.y = ( a[0,2] - a[2,0] ) * s;
			q.z = ( a[1,0] - a[0,1] ) * s;
		  } else {
			if ( a[0,0] > a[1,1] && a[0,0] > a[2,2] ) {
				double s = 2.0 * Math.Sqrt(1.0 + a[0, 0] - a[1, 1] - a[2, 2]);
			  q.w = (a[2,1] - a[1,2] ) / s;
			  q.x = 0.25f * s;
			  q.y = (a[0,1] + a[1,0] ) / s;
			  q.z = (a[0,2] + a[2,0] ) / s;
			} else if (a[1,1] > a[2,2]) {
				double s = 2.0 * Math.Sqrt(1.0 + a[1, 1] - a[0, 0] - a[2, 2]);
			  q.w = (a[0,2] - a[2,0] ) / s;
			  q.x = (a[0,1] + a[1,0] ) / s;
			  q.y = 0.25f * s;
			  q.z = (a[1,2] + a[2,1] ) / s;
			} else {
				double s = 2.0 * Math.Sqrt(1.0 + a[2, 2] - a[0, 0] - a[1, 1]);
			  q.w = (a[1,0] - a[0,1] ) / s;
			  q.x = (a[0,2] + a[2,0] ) / s;
			  q.y = (a[1,2] + a[2,1] ) / s;
			  q.z = 0.25f * s;
			}
		  }
		  return q.Normalize().XYZ();
		}
		public static Vector3d Rotation2Quaternion2(Matrix3d R)
		{
			double r11 = R[0, 0], r22 = R[1, 1], r33 = R[2, 2];
			double r12 = R[0, 1], r13 = R[0, 2], r23 = R[1, 2];
			double r21 = R[1, 0], r32 = R[2, 1], r31 = R[2, 0];
			double q0 = (r11 + r22 + r33 + 1.0f) / 4.0f;
			double q1 = (r11 - r22 - r33 + 1.0f) / 4.0f;
			double q2 = (-r11 + r22 - r33 + 1.0f) / 4.0f;
			double q3 = (-r11 - r22 + r33 + 1.0f) / 4.0f;
			if(q0 < 0.0f) q0 = 0.0f;
			if(q1 < 0.0f) q1 = 0.0f;
			if(q2 < 0.0f) q2 = 0.0f;
			if(q3 < 0.0f) q3 = 0.0f;
			q0 = Math.Sqrt(q0);
			q1 = Math.Sqrt(q1);
			q2 = Math.Sqrt(q2);
			q3 = Math.Sqrt(q3);
			if(q0 >= q1 && q0 >= q2 && q0 >= q3) {
				q0 *= +1.0f;
				q1 *= Math.Sign(r32 - r23);
				q2 *= Math.Sign(r13 - r31);
				q3 *= Math.Sign(r21 - r12);
			} else if(q1 >= q0 && q1 >= q2 && q1 >= q3) {
				q0 *= Math.Sign(r32 - r23);
				q1 *= +1.0f;
				q2 *= Math.Sign(r21 + r12);
				q3 *= Math.Sign(r13 + r31);
			} else if(q2 >= q0 && q2 >= q1 && q2 >= q3) {
				q0 *= Math.Sign(r13 - r31);
				q1 *= Math.Sign(r21 + r12);
				q2 *= +1.0f;
				q3 *= Math.Sign(r32 + r23);
			} else if(q3 >= q0 && q3 >= q1 && q3 >= q2) {
				q0 *= Math.Sign(r21 - r12);
				q1 *= Math.Sign(r31 + r13);
				q2 *= Math.Sign(r32 + r23);
				q3 *= +1.0f;
			} else {
				throw new ArgumentException("quaternion code error!");
			}
			double r = Norm(q0, q1, q2, q3);
			q0 /= r;
			q1 /= r;
			q2 /= r;
			q3 /= r;
			return new Vector3d(q0, q1, q2);
		}
		public static Matrix3d Quaternion2Rotation(Vector3d q)
		{
			double w = Math.Sqrt(1-q.SquareLength());
			if (double.IsNaN(w)) throw new ArgumentException("quaternion |.| > 1!");
		//	Vector4d quad = new Vector4d(q, w);
		//	return Trackball.QuatToMatrix3d(quad);

			double sqw = w*w;
			double sqx = q.x*q.x;
			double sqy = q.y*q.y;
			double sqz = q.z*q.z;

			// invs (inverse square length) is only required if quaternion is not already normalised
			double invs = 1 / (sqx + sqy + sqz + sqw);
			double m00 = ( sqx - sqy - sqz + sqw)*invs ; // since sqw + sqx + sqy + sqz =1/invs*invs
			double m11 = (-sqx + sqy - sqz + sqw) * invs;
			double m22 = (-sqx - sqy + sqz + sqw) * invs;
		    
			double tmp1 = q.x*q.y;
			double tmp2 = q.z*w;
			double m10 = 2.0 * (tmp1 + tmp2) * invs;
			double m01 = 2.0 * (tmp1 - tmp2) * invs;

			tmp1 = q.x * q.z;
			tmp2 = q.y * w;
			double m20 = 2.0 * (tmp1 - tmp2) * invs;
			double m02 = 2.0 * (tmp1 + tmp2) * invs;
			tmp1 = q.y*q.z;
			tmp2 = q.x*w;
			double m21 = 2.0 * (tmp1 + tmp2) * invs;
			double m12 = 2.0 * (tmp1 - tmp2) * invs;
			return new Matrix3d(new double[9] {
				m00,m01,m02,
				m10,m11,m22,
				m20,m21,m22
			});
		}
		public static double FindCCWRotAngle(Vector3d src, Vector3d tar, Vector3d axis)
		{
			// assume that all the three vectors are normalized!
			// rotate from src to tar around the axix in ccw
			double cos = src.Dot(tar);
			if (cos < -1) cos = -1;
			else if (cos > 1) cos = 1;
			double angle = Math.Acos(cos);
			if (src.Cross(tar).Dot(axis) < 0)
				return 2*Math.PI - angle;
			else
				return angle;
		}
		public static Vector3d RotMatrixToAngles(Matrix3d R, out int sign)
		{
			// in ZYZ convention
			Vector3d x2 = new Vector3d(R[0, 0], R[1, 0], R[2, 0]);
			Vector3d y2 = new Vector3d(R[0, 1], R[1, 1], R[2, 1]);
			Vector3d z2 = new Vector3d(R[0, 2], R[1, 2], R[2, 2]);
			sign = Math.Sign(x2.Cross(y2).Dot(z2));
			Vector3d x1 = new Vector3d(1, 0, 0);
			Vector3d y1 = new Vector3d(0, 1, 0);
			Vector3d z1 = new Vector3d(0, 0, sign); // left or right
			Vector3d N = z1.Cross(z2).Normalize();
			double alpha = FindCCWRotAngle(y1, N, z1); // Y->N  (Z)
			double beta = FindCCWRotAngle(z1, z2, N);  // Z->Z' (N)
			double gamma = FindCCWRotAngle(N, y2, z1); // N->Z  (Y')

			//Matrix3d R3 = AxisRotation(2, gamma*sign);
			//Matrix3d R2 = AxisRotation(1, beta);
			//Matrix3d R1 = AxisRotation(2, alpha*sign);
			//Vector3d yy = R1*R2*R3*y1;
			//yy.y = yy.y+0;

			//Vector3d zz = R1*R2 * R3 * z1;
			//zz.x = zz.x + 0;

			//Vector3d y3 = R1*R2*R3 * y1;
			//y3.y++;

			return new Vector3d(alpha, beta, gamma);
		}
		public static int GetRotMatrixConvention(Matrix3d R)
		{
			Vector3d x2 = new Vector3d(R[0, 0], R[1, 0], R[2, 0]);
			Vector3d y2 = new Vector3d(R[0, 1], R[1, 1], R[2, 1]);
			Vector3d z2 = new Vector3d(R[0, 2], R[1, 2], R[2, 2]);
			return Math.Sign(x2.Cross(y2).Dot(z2));
		}
		public static Matrix3d AnglesToRotMatrix(Vector3d angles, int sign)
		{
			Matrix3d R3 = AxisRotation(2, angles.z * sign);
			Matrix3d R2 = AxisRotation(1, angles.y);
			Matrix3d R1 = AxisRotation(2, angles.x * sign);
			return R1 * R2 * R3;

			//double a = angles.x, b = angles.y, r = angles.z;
			//double c1 = Math.Cos(a), s1 = Math.Sin(a);
			//double c2 = Math.Cos(b), s2 = Math.Sin(b);
			//double c3 = Math.Cos(r), s3 = Math.Sin(r);
			//if (sign < 0)
			//{
			//    double r1 = c1* c2 * c3 - s1*s3, r2 = -c3 * s1-c1*c2*s3, r3 = c1*s2;
			//    double r4 = c1 * s3 + c2*c3 * s1, r5 = c1 * c3 - c2*s1 * s3, r6 = s2 * s1;
			//    double r7 = -c3 * s2, r8 = s3 * s2, r9 = c2;
			//    return new Matrix3d(new double[9] {
			//        r1,r2,r3,r4,r5,r6,r7,r8,r9});
			//}
			//else
			//{
			//    double r1 = c1 * c2 * c3 - s1 * s3, r2 = c3 * s1 + c1 * c2 * s3, r3 = -c1 * s2;
			//    double r4 = -c1 * s3 - c2 * c3 * s1, r5 = c1 * c3 - c2 * s1 * s3, r6 = s2 * s1;
			//    double r7 = c3 * s2, r8 = s3 * s2, r9 = c2;
			//    return new Matrix3d(new double[9] {
			//        r1,r2,r3,r4,r5,r6,r7,r8,r9});
			//}
		}
		public static Matrix3d AxisRotation(int i, double angle)
		{
			Matrix3d R = Matrix3d.IdentityMatrix();
			double cos = Math.Cos(angle);
			double sin = Math.Sin(angle);
			switch (i)
			{
				case 0: //x
					R[1, 1] = R[2, 2] = cos;
					R[1, 2] = -sin;
					R[2, 1] = sin;
					break;
				case 1: // y
					R[0, 0] = R[2, 2] = cos;
					R[0, 2] = sin;
					R[2, 0] = -sin;
					break;
				case 2: // z
					R[1, 1] = R[0, 0] = cos;
					R[0, 1] = -sin;
					R[1, 0] = sin;
					break;
			}
			return R;
		}
		public static Vector2d FindLinesIntersectPoint(LineSegment l1, LineSegment l2)
		{
			Vector3d u1 = new Vector3d(l1.u.pos, 1), v1 = new Vector3d(l1.v.pos, 1);
			Vector3d u2 = new Vector3d(l2.u.pos, 1), v2 = new Vector3d(l2.v.pos, 1);
			Vector3d uv = u1.Cross(v1).Cross(u2.Cross(v2)).HomogenousNormalize();
			return uv.ToVector2d();
		}
		public static double[] GetBoxParameters(PolyProxy box)
		{
			Vector3d x = box.frame.x;
			double theta = FindCCWRotAngle(new Vector3d(1, 0, 0), x, new Vector3d(0, 0, 1));
			double s1 = box.xlen / 2;
			double s2 = box.ylen / 2;
			double s3 = box.zlen;
			double t1 = box.center.pos.x;
			double t2 = box.center.pos.y;
			return new double[6] {
				theta,s1,s2,s3,t1,t2
			};
		}
		public static Matrix4d GetBoxParameterMat(double[] parameters)
		{
			double theta = parameters[0];
			double s1 = parameters[1];
			double s2 = parameters[2];
			double s3 = parameters[3];
			double t1 = parameters[4];
			double t2 = parameters[5];
			Matrix4d R = Matrix4d.RotationMatrix(new Vector3d(0, 0, 1), theta);
			Matrix4d S = Matrix4d.IdentityMatrix();
			Matrix4d T = Matrix4d.TranslationMatrix(new Vector3d(t1, t2, 0));
			S[0, 0] = s1;
			S[1, 1] = s2;
			S[2, 2] = s3;
			return T * R * S;
		}
		public static Vector3d LeastSquaresLineFitting_old(Vector2d[] points, double[] weight)
		{
			int n = points.Length;
			double wx2 = 0, wx = 0, wy2 = 0, wy = 0, wxy = 0, w = 0;
			if (weight == null)
			{
				weight = new double[n];
				for (int i = 0; i < n; ++i)
					weight[i] = 1;
			}
			for (int i = 0; i < n; ++i)
			{
				double ww = weight[i];
				double x = points[i].x, y = points[i].y;
				wx2 += ww * x * x;
				wx += ww * x;
				wy2 += ww * y * y;
				wy += ww * y;
				wxy += ww * x * y;
				w += ww;
			}
			Matrix3d M = new Matrix3d(new double[9] {
				wx2,wxy,wx,
				wxy,wy2,wy,
				wx,wy,w}
			);

			//// cv eigendecompose
			//double[,] mat = new double[3, 3];
			//for (int i = 0; i < 3; ++i)
			//{
			//    for (int j = 0; j < 3; ++j)
			//    {
			//        mat[i, j] = M[i, j];
			//    }
			//}
			//Matrix<double> eigvec = new Matrix<double>(3, 3);
			//Matrix<double> eigval = new Matrix<double>(3, 1);
			//Matrix<double> cvmat = new Matrix<double>(mat);
			//CvInvoke.cvEigenVV(cvmat.Ptr, eigvec.Ptr, eigval.Ptr, 1e-30, 0, 0);
			//Vector3d v1 = new Vector3d(eigvec[2, 0], eigvec[2, 1], eigvec[2, 2]);
			//return v1;

			return M.SmallestEigenVector();
		}
		public static Vector3d LeastSquaresLineFitting(Vector2d[] points, double[] weight)
		{
			int n = points.Length;
			double wx2 = 0, wx = 0, wy2 = 0, wy = 0, wxy = 0, w = 0;
			if (weight == null)
			{
				weight = new double[n];
				for (int i = 0; i < n; ++i)
					weight[i] = 1;
			}
			for (int i = 0; i < n; ++i)
			{
				double ww = weight[i];
				double x = points[i].x, y = points[i].y;
				wx2 += ww * x * x;
				wx += ww * x;
				wy2 += ww * y * y;
				wy += ww * y;
				wxy += ww * x * y;
				w += ww;
			}
			Matrix3d M = new Matrix3d(new double[9] {
				wx2,wxy,wx,
				wxy,wy2,wy,
				wx,wy,w}
			);

			//// cv eigendecompose
			//double[,] mat = new double[3,3];
			//for (int i = 0; i < 3; ++i)
			//{
			// for (int j = 0; j < 3; ++j)
			// {
			// mat[i,j] = M[i,j];
			// }
			//}
			//Matrix<double> eigvec = new Matrix<double>(3,3);
			//Matrix<double> eigval = new Matrix<double>(3,1);
			//Matrix<double> cvmat = new Matrix<double>(mat);
			//CvInvoke.cvEigenVV(cvmat.Ptr, eigvec.Ptr, eigval.Ptr,1e-30, 0, 0);
			//Vector3d v1 = new Vector3d(eigvec[2,0],eigvec[2,1],eigvec[2,2]);
			//return v1;

			return M.SmallestEigenVector();
		}


		private static void FindAllSixCombs(List<int[]> allcombs, int n)
		{
			List<int> current = new List<int>();
			current.Add(0);
			FindAllSixCombs(allcombs, current, n);
		}
		private static void FindAllSixCombs(List<int[]> allcombs, List<int> current, int n)
		{
			if (current.Count == 6)
			{
				allcombs.Add(current.ToArray());
				return;
			}

			int last = -1;
			if (current.Count != 0)
				last = current[current.Count - 1];

			for (int i = last + 1; i < n; ++i)
			{
				int[] old = current.ToArray();
				current.Add(i);
				FindAllSixCombs(allcombs, current, n);
				current.Clear();
				current.AddRange(old);
			}
		}

		public static Vector3d[] GetBezierPoint(double t, Vector3d p0, Vector3d p1, Vector3d p2, Vector3d p3)
		{
			// B(t)=(1-t)^3P0+3(1-t)^2tP1+3(1-t)t^2P2+t^3P3
			// B'(t) = -3(1-t)^2P0+3[2(1-t)^2-2(1-t)*t]P1+3[2(1-t)t-t^2]+3t^2P3
			return new Vector3d[2] {
				(1-t)*(1-t)*(1-t)*p0 + 3*(1-t)*(1-t)*t*p1 + 3*(1-t)*t*t*p2 + t*t*t*p3,
				-3*(1-t)*(1-t)*p0 + 3*(2*(1-t)*(1-t)-2*(1-t)*t)*p1 + 3*(2*(1-t)*t-t*t)*p2 + 3*t*t*p3
			};
		}
		private static Vector2d TrapezoidCoords(Vector2d p, Vector2d[] poly)
		{
			Vector2d a = poly[0];
			Vector2d b = poly[1];
			Vector2d c = poly[2];
			Vector2d d = poly[3];

			// side (a, b) is parallel to side (d, c).
			Vector2d dir = (b - a).Normalize();
			Vector2d interpt1, interpt2;
			IntersectionHelper.LinesIntersection(p, p + dir, a, d, out interpt1);
			IntersectionHelper.LinesIntersection(p, p + dir, b, c, out interpt2);
			double d1 = (p - interpt1).Length();
			double d2 = (p - interpt2).Length();
			double signx = Math.Sign((p - interpt1).Dot(dir));
			double lambda;
			double h1 = Distance2LineSegment(p, a, b, out lambda);
			interpt1 = a + (b - a) * lambda;
			double h2 = Distance2Line(p, d, c);
			interpt2 = d + (c - d) * lambda;
			Vector2d dir2 = (interpt2 - interpt1).Normalize();
			double signy = Math.Sign((p - interpt1).Dot(dir2));

			return new Vector2d(signx * d1 / (d1 + d2), signy * h1 / (h1 + h2));
		}
		private static Vector2d TrapezoidPosition(Vector2d coord, Vector2d[] poly)
		{
			Vector2d a = poly[0];
			Vector2d b = poly[1];
			Vector2d c = poly[2];
			Vector2d d = poly[3];

			Vector2d dir = (b - a).Normalize();
			double lambda;
			Distance2LineSegment(c, a, b, out lambda);
			Vector2d interpt = a + (b - a) * lambda;
			Vector2d pp = interpt + (c - interpt) * coord.y;

			Vector2d interpt1, interpt2;
			IntersectionHelper.LinesIntersection(pp, pp + dir, a, d, out interpt1);
			IntersectionHelper.LinesIntersection(pp, pp + dir, b, c, out interpt2);

			return interpt1 + (interpt2 - interpt1) * coord.x;
		}
		private static double InterpolateBicubic(double sx, double sy, double[,] src)
		{
			int minx = (int)sx;
			int miny = (int)sy;
			int maxx = (int)(sx + 1);
			int maxy = (int)(sy + 1);
			minx += 1;
			miny += 1;
			maxx += 1;
			maxy += 1;
			sx += 1;
			sy += 1;

			double[][] points = new double[4][];
			points[0] = new double[] { src[miny - 1, minx - 1], src[miny - 1, minx], src[miny - 1, maxx], src[miny - 1, maxx + 1] };
			points[1] = new double[] { src[miny, minx - 1], src[miny, minx], src[miny, maxx], src[miny, maxx + 1] };
			points[2] = new double[] { src[maxy, minx - 1], src[maxy, minx], src[maxy, maxx], src[maxy, maxx + 1] };
			points[3] = new double[] { src[maxy + 1, minx - 1], src[maxy + 1, minx], src[maxy + 1, maxx], src[maxy + 1, maxx + 1] };

			CachedBicubicInterpolator inter = new CachedBicubicInterpolator();
			inter.updateCoefficients(points);
			double val = inter.getValue(sy - miny, sx - minx);

			return val;
		}
		public static double GetColorIntensity(Bgr c)
		{
			// 0.2989 * R + 0.5870 * G + 0.1140 * B
			return 0.2989 * c.Red + 0.5870 * c.Green + 0.1140 * c.Blue;
		}


		public static PointF Vector2PointF(Vector2d v)
		{
			return new PointF((float)v.x, (float)v.y);
		}
		public static Vector2d PointF2Vecter2d(PointF p)
		{
			return new Vector2d(p.X, p.Y);
		}
		public static Vector3d[] GetBoundingBoxPoints(List<PolyProxy> proxies)
		{
			if (proxies.Count < 1) return null;
			CoordinateFrame frame = proxies[0].frame;
			double minx = double.MaxValue, maxx = double.MinValue;
			double miny = double.MaxValue, maxy = double.MinValue;
			double minz = double.MaxValue, maxz = double.MinValue;
			foreach (PolyProxy box in proxies)
				foreach (Point3 p in box.points3)
				{
					Vector3d q = frame.GetPointLocalCoord(p.pos);
					if (minx > q.x) minx = q.x;
					if (maxx < q.x) maxx = q.x;
					if (miny > q.y) miny = q.y;
					if (maxy < q.y) maxy = q.y;
					if (minz > q.z) minz = q.z;
					if (maxz < q.z) maxz = q.z;
				}
			return new Vector3d[8] {
				frame.GetPointSpaceCoord(new Vector3d(maxx,miny,minz)),
				frame.GetPointSpaceCoord(new Vector3d(maxx,maxy,minz)),
				frame.GetPointSpaceCoord(new Vector3d(minx,maxy,minz)),
				frame.GetPointSpaceCoord(new Vector3d(minx,miny,minz)),
				frame.GetPointSpaceCoord(new Vector3d(maxx,miny,maxz)),
				frame.GetPointSpaceCoord(new Vector3d(maxx,maxy,maxz)),
				frame.GetPointSpaceCoord(new Vector3d(minx,maxy,maxz)),
				frame.GetPointSpaceCoord(new Vector3d(minx,miny,maxz))
			};
		}
		public static Vector3d[] GetBoundingBoxCenterAndScale(List<PolyProxy> proxies)
		{
			if (proxies.Count < 1) return null;
			CoordinateFrame frame = null;
			foreach (PolyProxy px in proxies)
			{
				if (px.frame != null)
				{
					frame = px.frame;
					break;
				}
			}
			if (frame == null) return new Vector3d[2] {new Vector3d(), new Vector3d()};
			double minx = double.MaxValue, maxx = double.MinValue;
			double miny = double.MaxValue, maxy = double.MinValue;
			double minz = double.MaxValue, maxz = double.MinValue;
			foreach (PolyProxy box in proxies)
				foreach (Point3 p in box.points3)
				{
					Vector3d q = frame.GetPointLocalCoord(p.pos);
					if (minx > q.x) minx = q.x;
					if (maxx < q.x) maxx = q.x;
					if (miny > q.y) miny = q.y;
					if (maxy < q.y) maxy = q.y;
					if (minz > q.z) minz = q.z;
					if (maxz < q.z) maxz = q.z;
				}
			Vector3d s = new Vector3d(maxx - minx, maxy - miny, maxz - minz);
			Vector3d[] v = new Vector3d[8] {
				frame.GetPointSpaceCoord(new Vector3d(maxx,miny,minz)),
				frame.GetPointSpaceCoord(new Vector3d(maxx,maxy,minz)),
				frame.GetPointSpaceCoord(new Vector3d(minx,maxy,minz)),
				frame.GetPointSpaceCoord(new Vector3d(minx,miny,minz)),
				frame.GetPointSpaceCoord(new Vector3d(maxx,miny,maxz)),
				frame.GetPointSpaceCoord(new Vector3d(maxx,maxy,maxz)),
				frame.GetPointSpaceCoord(new Vector3d(minx,maxy,maxz)),
				frame.GetPointSpaceCoord(new Vector3d(minx,miny,maxz))
			};
			Vector3d center = new Vector3d();
			for (int i = 0; i < 8; ++i)
			{
				center += v[i];
			}
			center /= 8;
			return new Vector3d[2] { center, s };
		}

	}
}
