﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using MyGeometry;

namespace i3_ImageManipulator
{
    public class ShadowNonLinearOptimizer
    {
        /// <summary>
        /// h: proxy height.
        /// bc: center of proxy's bottom polygon.
        /// sc: center of shadow polygon (the rectangle area).
        /// </summary>
        private double[] h;
        private Vector2d[] bc, sc;
        private Vector3d[] lps;

        private void function_grad(double[] xarr, ref double func, double[] grad, object obj)
        {
            // this callback calculates the function value
            double factor = 1;
            double x = xarr[0];
            double y = xarr[1];
            double z = xarr[2];

            func = 0;
            for (int i = 0; i < h.Length; ++i)
            {
                double xpart = ((bc[i].x - (x - bc[i].x) * h[i] / (z - h[i])) - sc[i].x);
                double ypart = ((bc[i].y - (y - bc[i].y) * h[i] / (z - h[i])) - sc[i].y);
                func += (xpart * xpart + ypart * ypart) * factor;
            }

            // now gradient
            for (int i = 0; i < h.Length; ++i)
            {
                grad[0] += (-2 * h[i] * (bc[i].x - sc[i].x - h[i] * (x - bc[i].x) / (z - h[i])) / (z - h[i])) * factor;
                grad[1] += (-2 * h[i] * (bc[i].y - sc[i].y - h[i] * (y - bc[i].y) / (z - h[i])) / (z - h[i])) * factor;
                grad[2] += (2 * h[i] * (x - bc[i].x) * (bc[i].x - sc[i].x - h[i] * (x - bc[i].x) / (z - h[i])) / ((z - h[i]) * (z - h[i]))) * factor;
                grad[2] += (2 * h[i] * (y - bc[i].y) * (bc[i].y - sc[i].y - h[i] * (y - bc[i].y) / (z - h[i])) / ((z - h[i]) * (z - h[i]))) * factor;
            }
        }

        public void function1_fvec(double[] xarr, double[] fi, object obj)
        {
            double factor = 1;
            double x = xarr[0];
            double y = xarr[1];
            double z = xarr[2];

            for (int i = 0; i < h.Length; ++i)
            {
                double xpart = ((bc[i].x - (x - bc[i].x) * h[i] / (z - h[i])) - sc[i].x);
                double ypart = ((bc[i].y - (y - bc[i].y) * h[i] / (z - h[i])) - sc[i].y);
                fi[i * 2] = xpart * factor;
                fi[i * 2 + 1] = ypart * factor;
            }
        }

        public void function1_jac(double[] xarr, double[] fi, double[,] jac, object obj)
        {
            double factor = 1;
            double x = xarr[0];
            double y = xarr[1];
            double z = xarr[2];

            for (int i = 0; i < h.Length; ++i)
            {
                double xpart = ((bc[i].x - (x - bc[i].x) * h[i] / (z - h[i])) - sc[i].x);
                double ypart = ((bc[i].y - (y - bc[i].y) * h[i] / (z - h[i])) - sc[i].y);
                
                fi[i * 2] = xpart * factor;
                jac[i * 2, 0] = -h[i] / (z - h[i]);
                jac[i * 2, 1] = 0;
                jac[i * 2, 2] = h[i] * (x - bc[i].x) / ((z - h[i]) * (z - h[i]));

                fi[i * 2 + 1] = ypart * factor;
                jac[i * 2 + 1, 0] = 0;
                jac[i * 2 + 1, 1] = -h[i] / (z - h[i]);
                jac[i * 2 + 1, 2] = h[i] * (y - bc[i].y) / ((z - h[i]) * (z - h[i]));
            }
        }

        public Vector3d Optimize(ShadowOptimizationData data)
        {
            this.h = data.heights;
            this.bc = data.bottomcenters;
            this.sc = data.shadowcenters;
            this.lps = data.lightpositions;

            double mincost = double.MaxValue;
            Vector3d minlp = new Vector3d();
            for (int i = 0; i < this.lps.Length; ++i)
            {
                //
                // This example demonstrates minimization of a nonlinear function
                // with nonlinear conjugate gradient method.
                //
                Vector3d lp = lps[i];
                double[] x = new double[] { lp.x, lp.y, lp.z };
                double epsg = 0.0000000001;
                double epsf = 0;
                double epsx = 0;
                int maxits = 0;

                //alglib.mincgstate state;
                //alglib.mincgreport rep;

                //alglib.mincgcreate(x, out state);
                //alglib.mincgsetcond(state, epsg, epsf, epsx, maxits);
                //alglib.mincgoptimize(state, function_grad, null, null);
                //alglib.mincgresults(state, out x, out rep);

                //alglib.minlbfgsstate state;
                //alglib.minlbfgsreport rep;

                //alglib.minlbfgscreate(1, x, out state);
                //alglib.minlbfgssetcond(state, epsg, epsf, epsx, maxits);
                //alglib.minlbfgsoptimize(state, function_grad, null, null);
                //alglib.minlbfgsresults(state, out x, out rep);

                alglib.minlmstate state;
                alglib.minlmreport rep;

                //alglib.minlmcreatev(this.h.Length * 2, x, 0.0001, out state);
                //alglib.minlmsetcond(state, epsg, epsf, epsx, maxits);
                //alglib.minlmoptimize(state, function1_fvec, null, null);
                //alglib.minlmresults(state, out x, out rep);

                alglib.minlmcreatevj(this.h.Length * 2, x, out state);
                alglib.minlmsetcond(state, epsg, epsf, epsx, maxits);
                alglib.minlmoptimize(state, function1_fvec, function1_jac, null, null);
                alglib.minlmresults(state, out x, out rep);

                double[] xtemp = (double[])x.Clone();
                double[] fitemp = new double[this.h.Length * 2];
                function1_fvec(xtemp, fitemp, null);
                double cost = fitemp.Sum(val => val * val);
                if (cost < mincost)
                {
                    mincost = cost;
                    minlp = new Vector3d(x[0], x[1], x[2]);
                }

                Program.OutputText(string.Format("{0}", rep.terminationtype), true); // EXPECTED: 4
                Program.OutputText(string.Format("{0}", alglib.ap.format(x, 2)), true); // EXPECTED: [-3,+3]
            }

            return minlp;
        }
    }
}
