﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Drawing;
using System.Windows.Forms;
using Tao.OpenGl;
using Tao.Platform.Windows;
using MyGeometry;
using System.IO;

using System.Runtime.InteropServices;

namespace i3_ImageManipulator
{
    public unsafe class ImageView : SimpleOpenGlControl
    {
        [DllImport("opengl32", EntryPoint = "wglUseFontBitmaps", CallingConvention = CallingConvention.Winapi)]
        public static extern bool wglUseFontBitmaps(
        IntPtr hDC,
        [MarshalAs(UnmanagedType.U4)] UInt32 first,
        [MarshalAs(UnmanagedType.U4)] UInt32 count,
        [MarshalAs(UnmanagedType.U4)] UInt32 listBase
        );

        [DllImport("GDI32.DLL", EntryPoint = "SelectObject",
        CallingConvention = CallingConvention.Winapi)]
        public static extern IntPtr SelectObject(
        [In] IntPtr hDC,
        [In] IntPtr font
        );


        public interface IMeshDisplay
        {
            void Display();
            void SetData();
        }
		public enum EnumOperationMode { 
			Viewing = 0,
			Selection,
			ControllerAnisotropicScaling, 
			RigidTransformation,
			ControllerDragging,
			RigidTranslation
		};
        public delegate void OnPickingHandler(object sender, int index);


        // private fields
		private ImageManipulator imgManipulator = null;
        public ImageManipulator ImgManipulator {
			get { return this.imgManipulator; }
			set {
				if (this.imgManipulator != null)
					this.imgManipulator.CurrentTransformation = this.currTransformation;
				this.imgManipulator = value;
				if (this.imgManipulator != null)
					this.currTransformation = this.imgManipulator.CurrentTransformation;
				this.Refresh();
			}
		}
        private EnumOperationMode currentMode = EnumOperationMode.Viewing;
		private MutualRelation currentrelation = MutualRelation.ONOBJECT;
		private Trackball arcball = new Trackball(100, 100);
		private Trackball manipulatingball = new Trackball(200, 200);
		private i3ObjectSwitcher switcher = new i3ObjectSwitcher();
		public bool enablesuggestion = false;
		public bool fixedrotation = false;
		public bool enablerealtime = false;
		public bool isselectingface = false;
		public bool isrigiddeformation = false;
		public void SetRelation(MutualRelation relation)
		{
			this.currentrelation = relation;
		}

        // font
        private bool initFont = false;
        public UInt32 fontBase = 0;
        private void BuildFont(PaintEventArgs pe)
        {
            IntPtr dc = pe.Graphics.GetHdc();
            IntPtr oldFontH = IntPtr.Zero;
            System.Drawing.Font font =
                new Font(
                "Arial",
                10F,
                System.Drawing.FontStyle.Bold,
                System.Drawing.GraphicsUnit.Point,
                ((System.Byte)(0)));

            fontBase = (uint)Gl.glGenLists(128);

            IntPtr fontH = font.ToHfont();
            oldFontH = SelectObject(dc, fontH);

            bool ret = wglUseFontBitmaps(
                dc,
                0,
                128,
                fontBase);

            SelectObject(dc, oldFontH);						// Selects The Font We Want

            pe.Graphics.ReleaseHdc(dc);

            if (!ret) throw new Exception();
        }
        private void ReleaseFont()
        {
            Gl.glDeleteLists(fontBase, 255);
        }


        // for viewing
        private double scaleRatio;
        public Vector2d currMousePosition = new Vector2d();
        private Vector2d mouseDownPosition = new Vector2d();
		private Vector3d currtransaxis = new Vector3d();


		private bool isMouseDown = false;
		public bool isCalibrateVanishingPoints = false;

		private TransformStyle currtransformstyle = TransformStyle.None;
        public Matrix4d currTransformation = Matrix4d.IdentityMatrix();
		public Matrix4d GetTransMatrix()
		{
			Matrix4d R = this.arcball.GetMatrix();
			return R * this.currTransformation;
		}
		public void Close()
		{
			if (this.imgManipulator != null)
			{
				this.currTransformation = Matrix4d.IdentityMatrix();
				this.imgManipulator.Dispose();
				this.imgManipulator = null;
				this.Refresh();
				GC.Collect();
			}
		}
		public void SaveTransformation(string file)
		{
			StreamWriter sw = new StreamWriter(file);
			for (int i = 0; i < 4; ++i)
			{
				for (int j = 0; j < 4; ++j)
				{
					sw.Write(this.currTransformation[i, j] + " ");
				}
			}
			sw.Write("\n");
			sw.Close();
		}
		public void LoadTransformation(string file)
		{
			StreamReader sr = new StreamReader(file);
			char[] delimiters = { ' ', '\t' };
			string s = ""; string[] tokens;
				
			s = sr.ReadLine(); // line 
			tokens = s.Split(delimiters);
			double[] trans = new double[16];
			for (int i = 0; i < 16; ++i)
			{
				trans[i] = double.Parse(tokens[i]);
			}

			Matrix4d T = new Matrix4d(trans);
			this.currTransformation = T;

			sr.Close();
		}
		public void AlignRotation(ref Matrix4d T, Vector3d axis, TransformStyle style)
		{
			switch (style)
			{
				case TransformStyle.Rotation:
					{
						double rotAngle; Vector3d rotaxis;
						Matrix4d.FindRotAxisAngle(T, out rotaxis, out rotAngle);
						T = Matrix4d.RotationMatrix(axis, rotAngle);
					}
					break;
			}
		}
		public void Copy()
		{
			this.switcher.CopyFrom(this.imgManipulator);
		}
		public void Paste()
		{
			this.switcher.PasteTo(this.imgManipulator,
				this.currMousePosition);
		}

        // for 2d elements drawing
		public Vector3d ComputeScreenProjectionPoint(Vector3d spacepos)
		{
			Gl.glPushMatrix();
			Matrix4d mat = this.arcball.GetMatrix() * currTransformation;
			Gl.glMultMatrixd(mat.Transpose().ToArray());
			OpenGLProjector glProjector = new OpenGLProjector();
			Gl.glPopMatrix();

			return glProjector.Project(spacepos);
		}
		

        // public properties
        public EnumOperationMode CurrentMode
        {
            get { return currentMode; }
            set { currentMode = value; }
        }

        public ImageView()
        {
            InitializeComponent();

            this.AutoMakeCurrent = true;
			this.InitializeContexts();
			this.InitOpenGL();
        }
        ~ImageView()
        {
            ReleaseFont();
        }
        private void InitializeComponent()
        {
            this.SuspendLayout();
            // 
            // ImageView
            // 
            this.Name = "ImageView";
            this.Size = new System.Drawing.Size(436, 271);
            this.ResumeLayout(false);

        }

        protected override void OnSizeChanged(EventArgs e)
        {
            base.OnSizeChanged(e);

            Size s = this.Size;
            if (s.Width == 0 || s.Height == 0) return;
        }
        protected override void OnMouseDown(System.Windows.Forms.MouseEventArgs e)
        {
            base.OnMouseDown(e);


            this.currMousePosition = new Vector2d(e.X, e.Y);
            this.mouseDownPosition = currMousePosition;
            this.isMouseDown = true;

            switch (this.currentMode)
            {
                case EnumOperationMode.Viewing:
                    break;
                
				case EnumOperationMode.Selection:
					if (e.Button == MouseButtons.Left && this.imgManipulator != null)
					{
						this.imgManipulator.GetSelectionStartPoint(mouseDownPosition);
					}
					break;
				case EnumOperationMode.ControllerDragging:
					if (this.imgManipulator != null)
					{
						this.imgManipulator.FixSelect();
						if (e.Button == MouseButtons.Middle)
						{
							this.currtransformstyle = TransformStyle.Scaling;
							this.manipulatingball.Click(mouseDownPosition, Trackball.MotionType.Scale);
							break;
						}
						else if (e.Button == MouseButtons.Left)
						{
							this.imgManipulator.InitialDragging(mouseDownPosition);
						}
						else
						{
							this.imgManipulator.RelaseFixSelect();
						}

					}
					break;
				case EnumOperationMode.RigidTransformation:
					if (this.imgManipulator != null)
					{
						this.imgManipulator.InitEdit();

						if (e.Button == MouseButtons.Middle)
						{
							this.currtransformstyle = TransformStyle.Scaling;
							this.manipulatingball.Click(mouseDownPosition, Trackball.MotionType.Scale);
							break;
						}
						else
						{
							if (this.isrigiddeformation)
							{
								this.imgManipulator.SelectMetaphorStartingPoint(mouseDownPosition);
								this.imgManipulator.InitialDragging(mouseDownPosition);
							}
							else if (this.imgManipulator.CurrentSelectedFace != null)
							{
								this.imgManipulator.SelectTranslateMetaphor(mouseDownPosition);
								this.imgManipulator.SelectMetaphorStartingPoint(mouseDownPosition);
								this.currtransaxis = this.imgManipulator.CurrentSelectedFace.normal;
							}
							
						}
						this.Refresh();
					}
					break;
				case EnumOperationMode.RigidTranslation:
					if (this.imgManipulator != null&& this.imgManipulator.CurrentSelectedFace != null)
					{
						this.imgManipulator.InitEdit();
	
						if (e.Button == MouseButtons.Left)
						{
							this.imgManipulator.FixSelect();
							this.imgManipulator.InitEdit();
							this.imgManipulator.SelectTranslateMetaphor(mouseDownPosition);
							this.currtransaxis = this.imgManipulator.CurrentSelectedFace.normal;
						}
						else if (e.Button == MouseButtons.Right)
						{
							this.imgManipulator.RelaseFixSelect();
						}
						else if (e.Button == MouseButtons.Middle)
						{
							this.currtransformstyle = TransformStyle.Scaling;
							this.manipulatingball.Click(mouseDownPosition, Trackball.MotionType.Scale);
							break;
						}
						this.Refresh();
					}
					break;
				case EnumOperationMode.ControllerAnisotropicScaling:
					if (this.imgManipulator != null && this.imgManipulator.CurrentSelectedFace!=null)
					{
						if (e.Button == MouseButtons.Left)
						{
					//		this.imgManipulator.FixSelect();
							this.imgManipulator.InitEdit();
					//		this.imgManipulator.SelectFace(mouseDownPosition,false);
							this.imgManipulator.SelectTranslateMetaphor(mouseDownPosition);
							this.currtransaxis = this.imgManipulator.CurrentSelectedFace.normal;
						}
						else if (e.Button == MouseButtons.Right)
						{
							this.imgManipulator.RelaseFixSelect();
						}
					}
					this.Refresh();
					break;
            }
        }
        protected override void OnMouseMove(System.Windows.Forms.MouseEventArgs e)
        {
            base.OnMouseMove(e);

            this.currMousePosition = new Vector2d(e.X, e.Y);
            switch (this.currentMode)
            {
                case EnumOperationMode.Viewing:
					
					Program.SetStatus("(" + e.X + ", " + e.Y + ")");
                    break;

                case EnumOperationMode.Selection:
				//	Program.SetStatus("(" + e.X + ", " + e.Y + ")");
					if (this.imgManipulator != null)
					{
						this.imgManipulator.GetSelectionEndPoint(currMousePosition); 
						this.Refresh();
					}
                    break;
				case EnumOperationMode.RigidTranslation:
					if (this.imgManipulator != null)
					{
						if (isMouseDown)
						{
							if (e.Button == MouseButtons.Middle)
							{
								this.manipulatingball.Drag(currMousePosition);
								Matrix4d S = this.manipulatingball.GetMatrix();
								this.imgManipulator.DeformSelectedProxy(S, this.currtransformstyle, 0);
							}
							else
							{
								if (this.isrigiddeformation)
									this.imgManipulator.SelectMetaphorEndingPoint(currMousePosition);
								else
								{
									this.imgManipulator.SelectMetaphorTranslatePoint(mouseDownPosition, currMousePosition);
									this.imgManipulator.TranslateSelectedProxy();
								}
							}
						}
						else
						{
							this.imgManipulator.Select(currMousePosition, false);
						}

						this.Refresh();
					}
					break;
				case EnumOperationMode.ControllerDragging:
					if (this.imgManipulator != null)
					{
						if (isMouseDown)
						{
							if (e.Button == MouseButtons.Middle)
							{
								this.manipulatingball.Drag(currMousePosition);
								Matrix4d S = this.manipulatingball.GetMatrix();
								this.imgManipulator.DeformSelectedProxy(S, this.currtransformstyle, 0);
							}
							else
								this.imgManipulator.DragSelected(currMousePosition);
						}
						else
						{
							this.imgManipulator.Select(currMousePosition, true);
					//		this.imgManipulator.SelectMetaphorRotationPoint(currMousePosition);
						}
						this.Refresh();
					}
					
					break;
				
				
				case EnumOperationMode.ControllerAnisotropicScaling:
					if (this.imgManipulator != null)
					{
						if (this.isMouseDown)
						{
							this.imgManipulator.SelectMetaphorTranslatePoint(mouseDownPosition, currMousePosition);
							if (this.isrigiddeformation)
							{
								this.imgManipulator.TranslateSelectedProxy();
							}
							else
								this.imgManipulator.TranslateSelectedFace();
						}
						else
						{
				//			this.imgManipulator.Select(currMousePosition, false);
							this.imgManipulator.SelectTranslateMetaphor(currMousePosition);
						}
					}
					this.Refresh();
					break;
				
				
            }
        }
        protected override void OnMouseUp(System.Windows.Forms.MouseEventArgs e)
        {
            base.OnMouseUp(e);

            this.currMousePosition = new Vector2d(e.X, e.Y);
            this.isMouseDown = false;

			if (e.Button == MouseButtons.Middle && this.imgManipulator != null)
			{
				this.imgManipulator.CopyImgPos();
			}

            switch (this.currentMode)
            {
                case EnumOperationMode.Viewing:
					//if (currMousePosition.Equals(mouseDownPosition)) break;
					//if (this.imgManipulator != null && this.imgManipulator.has3dobjects)
					//{
					//    Matrix4d m = arcball.GetMatrix();
					//    this.currTransformation = m * this.currTransformation;
					//    this.arcball.End();
					//    this.Refresh();
					//}
                    break;

                case EnumOperationMode.Selection:
					if (e.Button == MouseButtons.Left && this.imgManipulator != null)
					{
						if (this.isselectingface)
							this.imgManipulator.SelectBoxFaceByRect();
						else
						{
							if ((Control.ModifierKeys & Keys.Shift) == Keys.Shift)
							{
								this.imgManipulator.SelectBoxesByRect();
							}
							else if ((Control.ModifierKeys & Keys.Control)== Keys.Control)
							{
								this.imgManipulator.DeselectBoxesByRect();
							}
							else if (!this.imgManipulator.SelectBoxByRect((Control.ModifierKeys & Keys.Control) == Keys.Control))
							{
								//				this.imgManipulator.SelectObjPolygonByRect();
							}
						}
						this.imgManipulator.ReleaseSelectionPoints();
						this.Refresh();
					}
                    break;

                
				case EnumOperationMode.ControllerDragging:
					if (this.imgManipulator != null)
					{
						if (e.Button == MouseButtons.Middle) this.manipulatingball.End();

						this.imgManipulator.DeformUpdateSelected(null);

						this.imgManipulator.DeformUpdate();
						this.imgManipulator.DoTxtureMapping();

						if ((currMousePosition - mouseDownPosition).Length() > 1)
						{
							this.imgManipulator.RelaseFixSelect();
						}

						this.Refresh();
					}
					break;
				
				case EnumOperationMode.ControllerAnisotropicScaling:
					if (this.imgManipulator != null)
					{
						this.imgManipulator.DeformUpdateSelected(null);
						this.imgManipulator.Propagate();
						this.imgManipulator.DeformUpdate();
						this.imgManipulator.DoTxtureMapping();
						
						//if ((currMousePosition - mouseDownPosition).Length() > 1)
						//{
						//    this.imgManipulator.RelaseFixSelect();
						//}

						this.Refresh();
					}
					break;
				case EnumOperationMode.RigidTranslation:
					if (this.imgManipulator != null)
					{
						this.imgManipulator.DeformUpdateSelected(null);
						this.imgManipulator.DeformUpdate();
						this.imgManipulator.DoTxtureMapping();

						if ((currMousePosition - mouseDownPosition).Length() > 1)
						{
							this.imgManipulator.RelaseFixSelect();
						}

						this.Refresh();
					}
					break;
            }
        }
		protected override void OnMouseWheel(MouseEventArgs e)
		{
			base.OnMouseWheel(e);
			
			if (imgManipulator == null) return;
			if (e.Delta > 0)
			{
				imgManipulator.EnlargeSacle();
			}
			else
			{
				imgManipulator.ReduceSacle();
			}
			this.Refresh();

		}
        protected override void OnPaint(PaintEventArgs e)
        {
            if (!initFont)
            {
                BuildFont(e);
                initFont = true;
            }

            Color c = SystemColors.Control;
            if (this.DesignMode == true)
            {
                base.OnPaint(e);
                return;
            }

			this.MakeCurrent();
		//	this.InitMatrix();

			Gl.glClearColor(1.0f, 1.0f, 1.0f, 0.0f);
			Gl.glClear(Gl.GL_COLOR_BUFFER_BIT | Gl.GL_DEPTH_BUFFER_BIT);
			Gl.glLoadIdentity();

			if (this.imgManipulator != null)
			{
				this.imgManipulator.RenderSense();
			}
			this.SwapBuffers();
        }


		public void SetBallBounds(int w, int h)
		{
			this.scaleRatio = (w > h) ? h : w;
			this.arcball.SetBounds(w * 1.0, h * 1.0);
			this.manipulatingball.SetBounds(w * 1.0, h * 1.0);
		}
        private void InitOpenGL()
        {
            Gl.glDisable(Gl.GL_BLEND);
            Gl.glDisable(Gl.GL_DEPTH_TEST);
            Gl.glDisable(Gl.GL_CULL_FACE);
            Gl.glDisable(Gl.GL_LIGHTING);

            Gl.glColorMaterial(Gl.GL_FRONT, Gl.GL_AMBIENT_AND_DIFFUSE);
            Gl.glEnable(Gl.GL_COLOR_MATERIAL);

            Gl.glClearColor(1.0f, 1.0f, 1.0f, 1.0f);
        }
        private void InitMatrix()
        {
            double w = Size.Width;
            double h = Size.Height;

	//		w = this.glviewportw;
	//		h = this.glviewporth;

	//		Gl.glViewport(0, 0, (int)w, (int)h);

			//Gl.glMatrixMode(Gl.GL_PROJECTION);
			//Gl.glLoadIdentity();

			//Glu.gluOrtho2D(0, w, h, 0);

			//Gl.glMatrixMode(Gl.GL_MODELVIEW);
			//Gl.glLoadIdentity();
        }



        private void DrawSelectionRect()
        {
            Gl.glMatrixMode(Gl.GL_PROJECTION);
            Gl.glPushMatrix();
            Gl.glLoadIdentity();
            Glu.gluOrtho2D(0, this.Width, 0, this.Height);
            Gl.glMatrixMode(Gl.GL_MODELVIEW);
            Gl.glPushMatrix();
            Gl.glLoadIdentity();

            Gl.glPolygonMode(Gl.GL_FRONT_AND_BACK, Gl.GL_LINE);
            Gl.glDisable(Gl.GL_CULL_FACE);
            Gl.glDisable(Gl.GL_DEPTH_TEST);
            Gl.glColor3f(0.0f, 0.0f, 0.0f);
            Gl.glRectd(mouseDownPosition.x, this.Height - mouseDownPosition.y, 
				currMousePosition.x, this.Height - currMousePosition.y);
            Gl.glEnable(Gl.GL_CULL_FACE);
            Gl.glEnable(Gl.GL_DEPTH_TEST);

            Gl.glMatrixMode(Gl.GL_PROJECTION);
            Gl.glPopMatrix();
            Gl.glMatrixMode(Gl.GL_MODELVIEW);
            Gl.glPopMatrix();
        }

        private Color strokeColor = Color.Coral;
        public void SetMainStrokeColor(Color c)
        {
            strokeColor = c;
        }
		public void DrawStroke(List<Vector2d> curve, Color c)
		{
			if (curve == null || curve.Count < 1) return;

			Gl.glColor3ub(c.R, c.G, c.B);
			Gl.glLineWidth(2.5f);
			Gl.glEnable(Gl.GL_LINE_SMOOTH);
			Gl.glBegin(Gl.GL_LINES);
			int n = curve.Count;
			int k = n - 1;
			for (int i = 0; i < k; ++i)
			{
				Gl.glVertex2d(curve[i].x, curve[i].y);
				Gl.glVertex2d(curve[(i + 1) % n].x, curve[(i + 1) % n].y);
			}
			Gl.glEnd();

			Gl.glPointSize(2.0f);
			Gl.glBegin(Gl.GL_POINTS);
			foreach (Vector2d u in curve)
			{
				Gl.glVertex2d(u.x, u.y);
			}
			Gl.glEnd();
		}
    }
}