﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Drawing;
using System.Windows.Forms;
using Tao.OpenGl;
using Tao.Platform.Windows;
using MyGeometry;
using System.IO;

using System.Runtime.InteropServices;

namespace i3_ImageManipulator
{
    public unsafe class ImageView : SimpleOpenGlControl
    {
        [DllImport("opengl32", EntryPoint = "wglUseFontBitmaps", CallingConvention = CallingConvention.Winapi)]
        public static extern bool wglUseFontBitmaps(
        IntPtr hDC,
        [MarshalAs(UnmanagedType.U4)] UInt32 first,
        [MarshalAs(UnmanagedType.U4)] UInt32 count,
        [MarshalAs(UnmanagedType.U4)] UInt32 listBase
        );

        [DllImport("GDI32.DLL", EntryPoint = "SelectObject",
        CallingConvention = CallingConvention.Winapi)]
        public static extern IntPtr SelectObject(
        [In] IntPtr hDC,
        [In] IntPtr font
        );


        public interface IMeshDisplay
        {
            void Display();
            void SetData();
        }
		public enum EnumOperationMode { 
			Viewing = 0,
			Selection,
			AssignHexagon,
			AdjustHexagon
		};
        public delegate void OnPickingHandler(object sender, int index);


        // private fields
		private ImageManipulator imgManipulator = null;
        public ImageManipulator ImgManipulator {
			get { return this.imgManipulator; }
			set {
				if (this.imgManipulator != null)
					this.imgManipulator.CurrentTransformation = this.currTransformation;
				this.imgManipulator = value;
				if (this.imgManipulator != null)
					this.currTransformation = this.imgManipulator.CurrentTransformation;
				this.Refresh();
			}
		}
        private EnumOperationMode currentMode = EnumOperationMode.Viewing;
		private MutualRelation currentrelation = MutualRelation.ONOBJECT;
		private Trackball arcball = new Trackball(100, 100);
		private Trackball manipulatingball = new Trackball(200, 200);
		public bool enablesuggestion = false;
		public bool fixedrotation = false;
		public bool enablerealtime = false;
		public bool isselectingface = false;
		public bool isrigiddeformation = false;
		private bool isMouseDown = false;
		public void SetRelation(MutualRelation relation)
		{
			this.currentrelation = relation;
		}

        // font
        private bool initFont = false;
        public UInt32 fontBase = 0;
        private void BuildFont(PaintEventArgs pe)
        {
            IntPtr dc = pe.Graphics.GetHdc();
            IntPtr oldFontH = IntPtr.Zero;
            System.Drawing.Font font =
                new Font(
                "Arial",
                10F,
                System.Drawing.FontStyle.Bold,
                System.Drawing.GraphicsUnit.Point,
                ((System.Byte)(0)));

            fontBase = (uint)Gl.glGenLists(128);

            IntPtr fontH = font.ToHfont();
            oldFontH = SelectObject(dc, fontH);

            bool ret = wglUseFontBitmaps(
                dc,
                0,
                128,
                fontBase);

            SelectObject(dc, oldFontH);						// Selects The Font We Want

            pe.Graphics.ReleaseHdc(dc);

            if (!ret) throw new Exception();
        }
        private void ReleaseFont()
        {
            Gl.glDeleteLists(fontBase, 255);
        }


        // for viewing
        private double scaleRatio;
        public Vector2d currMousePosition = new Vector2d();
		private Vector2d prevMousePosition = new Vector2d();
        private Vector2d mouseDownPosition = new Vector2d();

		public bool isCalibrateVanishingPoints = false;

        public Matrix4d currTransformation = Matrix4d.IdentityMatrix();
		public Matrix4d GetTransMatrix()
		{
			Matrix4d R = this.arcball.GetMatrix();
			return R * this.currTransformation;
		}
		public void Close()
		{
			if (this.imgManipulator != null)
			{
				this.currTransformation = Matrix4d.IdentityMatrix();
				this.imgManipulator.Dispose();
				this.imgManipulator = null;
				this.Refresh();
				GC.Collect();
			}
		}
		public void SaveTransformation(string file)
		{
			StreamWriter sw = new StreamWriter(file);
			for (int i = 0; i < 4; ++i)
			{
				for (int j = 0; j < 4; ++j)
				{
					sw.Write(this.currTransformation[i, j] + " ");
				}
			}
			sw.Write("\n");
			sw.Close();
		}
		public void LoadTransformation(string file)
		{
			StreamReader sr = new StreamReader(file);
			char[] delimiters = { ' ', '\t' };
			string s = ""; string[] tokens;
				
			s = sr.ReadLine(); // line 
			tokens = s.Split(delimiters);
			double[] trans = new double[16];
			for (int i = 0; i < 16; ++i)
			{
				trans[i] = double.Parse(tokens[i]);
			}

			Matrix4d T = new Matrix4d(trans);
			this.currTransformation = T;

			sr.Close();
		}
		public void AlignRotation(ref Matrix4d T, Vector3d axis, TransformStyle style)
		{
			switch (style)
			{
				case TransformStyle.Rotation:
					{
						double rotAngle; Vector3d rotaxis;
						Matrix4d.FindRotAxisAngle(T, out rotaxis, out rotAngle);
						T = Matrix4d.RotationMatrix(axis, rotAngle);
					}
					break;
			}
		}


        // for 2d elements drawing
		public Vector3d ComputeScreenProjectionPoint(Vector3d spacepos)
		{
			Gl.glPushMatrix();
			Matrix4d mat = this.arcball.GetMatrix() * currTransformation;
			Gl.glMultMatrixd(mat.Transpose().ToArray());
			OpenGLProjector glProjector = new OpenGLProjector();
			Gl.glPopMatrix();

			return glProjector.Project(spacepos);
		}
		

        // public properties
        public EnumOperationMode CurrentMode
        {
            get { return currentMode; }
            set { currentMode = value; }
        }

        public ImageView()
        {
            InitializeComponent();

            this.AutoMakeCurrent = true;
			this.InitializeContexts();
			this.InitOpenGL();
        }
        ~ImageView()
        {
            ReleaseFont();
        }
        private void InitializeComponent()
        {
            this.SuspendLayout();
            // 
            // ImageView
            // 
            this.Name = "ImageView";
            this.Size = new System.Drawing.Size(436, 271);
            this.ResumeLayout(false);

        }

        protected override void OnSizeChanged(EventArgs e)
        {
            base.OnSizeChanged(e);

            Size s = this.Size;
            if (s.Width == 0 || s.Height == 0) return;
        }
        protected override void OnMouseDown(System.Windows.Forms.MouseEventArgs e)
        {
            base.OnMouseDown(e);

            this.currMousePosition = new Vector2d(e.X, e.Y);
            this.mouseDownPosition = currMousePosition;
			this.isMouseDown = true;
            switch (this.currentMode)
            {
				case EnumOperationMode.Viewing:
					if (this.imgManipulator != null)
						switch (e.Button)
						{
							case MouseButtons.Left: arcball.Click(currMousePosition, Trackball.MotionType.Rotation); break;
							//				case MouseButtons.Middle: arcball.Click(currMousePosition / this.scaleRatio, Trackball.MotionType.Pan); break;
							case MouseButtons.Right: arcball.Click(currMousePosition, Trackball.MotionType.Scale); break;
						}
					break;
            }
        }
        protected override void OnMouseMove(System.Windows.Forms.MouseEventArgs e)
        {
            base.OnMouseMove(e);

            this.currMousePosition = new Vector2d(e.X, e.Y);
            switch (this.currentMode)
            {
                case EnumOperationMode.Viewing:
					if (isMouseDown && this.imgManipulator != null)
					{
						switch (e.Button)
						{
							case MouseButtons.Left: arcball.Drag(currMousePosition); break;
							//		case MouseButtons.Middle: arcball.Drag(currMousePosition / this.scaleRatio); break;
							case MouseButtons.Right: arcball.Drag(currMousePosition); break;
						}
						this.Refresh();
					}
					Program.SetStatus("(" + e.X + ", " + e.Y + ")");
                    break;
				case EnumOperationMode.AssignHexagon:
					if (this.imgManipulator != null)
					{
						this.imgManipulator.CreateHexagonPoint(currMousePosition);
					}
					this.Refresh();
					break;
				case EnumOperationMode.AdjustHexagon:
					if (this.imgManipulator != null)
					{
						if (this.isMouseDown)
						{
							Vector2d off = this.currMousePosition - this.mouseDownPosition;
							this.imgManipulator.AdjustCalibratingPoint(off);
						}
						else
						{
							this.imgManipulator.SelectSixPoints(currMousePosition);
						}
						this.Refresh();
					}

					break;
				
            }
        }
        protected override void OnMouseUp(System.Windows.Forms.MouseEventArgs e)
        {
            base.OnMouseUp(e);

            this.currMousePosition = new Vector2d(e.X, e.Y);
			this.isMouseDown = false;

			if (e.Button == MouseButtons.Middle && this.imgManipulator != null)
			{
				this.imgManipulator.CopyImgPos();
			}

            switch (this.currentMode)
            {
				case EnumOperationMode.Viewing:
					if (currMousePosition.Equals(mouseDownPosition)) break;
					if (this.imgManipulator != null)
					{
						Matrix4d m = arcball.GetMatrix();
						this.currTransformation = m * this.currTransformation;
						this.arcball.End();
						this.Refresh();
					}
					break;
				case EnumOperationMode.AssignHexagon:
					if (e.Button == MouseButtons.Left && this.imgManipulator != null)
					{
						prevMousePosition.x = currMousePosition.x;
						prevMousePosition.y = currMousePosition.y;
						this.imgManipulator.AddHexagonPoint();
						this.Refresh();
					}
					else if (e.Button == MouseButtons.Right)
					{
						this.imgManipulator.ReleaseSelection();
						this.Refresh();
					}
					break;
				case EnumOperationMode.AdjustHexagon:
					if (this.imgManipulator != null && e.Button != MouseButtons.Middle)
					{
						this.imgManipulator.CalibrateUpdate();
						this.Refresh();
					}
					break;
            }
        }
		protected override void OnMouseWheel(MouseEventArgs e)
		{
			base.OnMouseWheel(e);
			
			if (imgManipulator == null) return;
			if (e.Delta > 0)
			{
				imgManipulator.EnlargeSacle();
			}
			else
			{
				imgManipulator.ReduceSacle();
			}
			this.Refresh();

		}
        protected override void OnPaint(PaintEventArgs e)
        {
            if (!initFont)
            {
                BuildFont(e);
                initFont = true;
            }

            Color c = SystemColors.Control;
            if (this.DesignMode == true)
            {
                base.OnPaint(e);
                return;
            }

			this.MakeCurrent();
		//	this.InitMatrix();

			Gl.glClearColor(1.0f, 1.0f, 1.0f, 0.0f);
			Gl.glClear(Gl.GL_COLOR_BUFFER_BIT | Gl.GL_DEPTH_BUFFER_BIT);
			Gl.glLoadIdentity();

			if (this.imgManipulator != null)
			{
				this.imgManipulator.RenderSense();
			}
			this.SwapBuffers();
        }

		public void SetBallBounds(int w, int h)
		{
			this.scaleRatio = (w > h) ? h : w;
			this.arcball.SetBounds(w * 1.0, h * 1.0);
			this.manipulatingball.SetBounds(w * 1.0, h * 1.0);
		}
        private void InitOpenGL()
        {
            Gl.glDisable(Gl.GL_BLEND);
            Gl.glDisable(Gl.GL_DEPTH_TEST);
            Gl.glDisable(Gl.GL_CULL_FACE);
            Gl.glDisable(Gl.GL_LIGHTING);

            Gl.glColorMaterial(Gl.GL_FRONT, Gl.GL_AMBIENT_AND_DIFFUSE);
            Gl.glEnable(Gl.GL_COLOR_MATERIAL);

            Gl.glClearColor(1.0f, 1.0f, 1.0f, 1.0f);
        }
        private void InitMatrix()
        {
            double w = Size.Width;
            double h = Size.Height;

	//		w = this.glviewportw;
	//		h = this.glviewporth;

	//		Gl.glViewport(0, 0, (int)w, (int)h);

			//Gl.glMatrixMode(Gl.GL_PROJECTION);
			//Gl.glLoadIdentity();

			//Glu.gluOrtho2D(0, w, h, 0);

			//Gl.glMatrixMode(Gl.GL_MODELVIEW);
			//Gl.glLoadIdentity();
        }

        private void DrawSelectionRect()
        {
            Gl.glMatrixMode(Gl.GL_PROJECTION);
            Gl.glPushMatrix();
            Gl.glLoadIdentity();
            Glu.gluOrtho2D(0, this.Width, 0, this.Height);
            Gl.glMatrixMode(Gl.GL_MODELVIEW);
            Gl.glPushMatrix();
            Gl.glLoadIdentity();

            Gl.glPolygonMode(Gl.GL_FRONT_AND_BACK, Gl.GL_LINE);
            Gl.glDisable(Gl.GL_CULL_FACE);
            Gl.glDisable(Gl.GL_DEPTH_TEST);
            Gl.glColor3f(0.0f, 0.0f, 0.0f);
            Gl.glRectd(mouseDownPosition.x, this.Height - mouseDownPosition.y, 
				currMousePosition.x, this.Height - currMousePosition.y);
            Gl.glEnable(Gl.GL_CULL_FACE);
            Gl.glEnable(Gl.GL_DEPTH_TEST);

            Gl.glMatrixMode(Gl.GL_PROJECTION);
            Gl.glPopMatrix();
            Gl.glMatrixMode(Gl.GL_MODELVIEW);
            Gl.glPopMatrix();
        }

        private Color strokeColor = Color.Coral;
        public void SetMainStrokeColor(Color c)
        {
            strokeColor = c;
        }
		public void DrawStroke(List<Vector2d> curve, Color c)
		{
			if (curve == null || curve.Count < 1) return;

			Gl.glColor3ub(c.R, c.G, c.B);
			Gl.glLineWidth(2.5f);
			Gl.glEnable(Gl.GL_LINE_SMOOTH);
			Gl.glBegin(Gl.GL_LINES);
			int n = curve.Count;
			int k = n - 1;
			for (int i = 0; i < k; ++i)
			{
				Gl.glVertex2d(curve[i].x, curve[i].y);
				Gl.glVertex2d(curve[(i + 1) % n].x, curve[(i + 1) % n].y);
			}
			Gl.glEnd();

			Gl.glPointSize(2.0f);
			Gl.glBegin(Gl.GL_POINTS);
			foreach (Vector2d u in curve)
			{
				Gl.glVertex2d(u.x, u.y);
			}
			Gl.glEnd();
		}
    }
}