﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Linq;
using System.Collections;	// -- hashtable 
using System.ComponentModel;
using Tao.OpenGl;
using Tao.Platform.Windows;
using MyGeometry;
using System.Drawing;
using System.IO;
using System.Diagnostics;

using System.Drawing.Imaging;

using Emgu.CV;
using Emgu.CV.Structure;
using Emgu.CV.UI;


namespace i3_ImageManipulator
{
    [TypeConverterAttribute(typeof(DeformerConverter))]
    public unsafe partial class ImageManipulator : IDisposable
    {
		// variables, members
		public byte TabIndex = 0;
		public ImageView parentview = null;
		public CameraCalibrator cameracalibrator = null;
		public Color BoxSelectColor { get; set; }
		public Color ProxyColor { get; set; }
		public bool renderLayerImage = true;
		public bool RenderLayerImage {
			get { return renderLayerImage; }
			set {
				renderLayerImage = value; 
				if (parentview != null) 
					parentview.Refresh(); 
			} 
		}
		public bool RenderBoxWithTexutreMap { get; set; }
		public bool RenderSolidProxies { get; set; }
		public bool RenderLineProxies { get; set; }
		public bool TextureAllPolygons { get; set; }
		public float FlatProxyLineWidth { get; set; }
		public float TranslateScaleFactor { get; set; }
		public float CalibratingNodeSize { get; set; }
		private double proxyrendersize = 0.015;
		public bool finalized = false;
		private Color[] colorMall = null;
		public Color[] ColorMall
		{
			get { return colorMall; }
			set { colorMall = value; }
		}
        private Vector2d imgpos = new Vector2d(0, 0);
		private Vector2d oldimgpos = new Vector2d(0, 0);
		private Vector2d[] vp2d = null;
		public string imagename;


		private Point2 selectedpoint = null;
		private List<Point2> currhexagonpoints = new List<Point2>();
		private int calibrateindex = -1;


        private int imgwidth = 0;
        private int imgheight = 0;
        private int imgchannel = 4;
		private int glviewportw = 800;
		private int glviewporth = 600;
		private double[] glmodelviewmatrix = null;
		private double[] glprojectionmatrix = null;
		private Matrix4d roomtrans = null, roomtrans_inv = null;
		private Vector3d lightpos = new Vector3d(5, 5, 5);
		private List<Polygon> hexagons = null;
		private List<Polygon> allpolygons = null;
		private List<PolyProxy> polyproxies = new List<PolyProxy>();
		private List<Hull> alignedhulls = new List<Hull>();
        private uint[] layertxtid = new uint[1];
		private double sacleRatio = 1.0;
		private bool has3dobjects = false;
		public Matrix4d CurrentTransformation = Matrix4d.IdentityMatrix();


		// opencv utilities
		public Image<Bgr, byte> Image { get { return this.image; } }
		private Image<Bgr, byte> image = null;


		// constructor
		public ImageManipulator()
		{
			this.InitialRandomColors(36);
			BoxSelectColor = Color.Lime;
			ProxyColor = Color.Yellow;
			this.CalibratingNodeSize = 7.0f;
			this.FlatProxyLineWidth = 1.0f;
			this.TranslateScaleFactor = 10;
			this.RenderLineProxies = true;
			this.RenderSolidProxies = false;
		}
		

		// mem functions
		public void EnlargeSacle()
		{
			sacleRatio /= 0.9;
		}
		public void ReduceSacle()
		{
			sacleRatio *= 0.9;
			if (sacleRatio < 0.1094189891315125)
			{
				sacleRatio = 0.1094189891315124;
			}
		}
		public Vector2d ImgPos
		{
			get { return imgpos; }
			set { imgpos = value; }
		}
		public void CopyImgPos()
		{
			this.oldimgpos = new Vector2d(imgpos.x, imgpos.y);
		}


		// initial
		private void SetGLViewPort(int w, int h)
		{
			this.glviewportw = w;
			this.glviewporth = h;
		}		
		private void InitialRandomColors(int K)
		{
			Random rand = new Random();
			this.colorMall = new Color[K];
			for (int i = 0; i < K; ++i)
			{
				this.colorMall[i] = Color.FromArgb(
					rand.Next(0, 255),
					rand.Next(0, 255),
					rand.Next(0, 255)
				);
			}
			this.colorMall[3] = Color.FromArgb(197, 27, 125);
			this.colorMall[0] = Color.FromArgb(233, 163, 201);
			this.colorMall[4] = Color.FromArgb(241, 60, 45);
			this.colorMall[6] = Color.FromArgb(253, 224, 239);
			this.colorMall[2] = Color.FromArgb(230, 245, 208);
			this.colorMall[1] = Color.FromArgb(161, 215, 106);
			this.colorMall[5] = Color.FromArgb(80, 127, 218);
		}
		private void SetTranslationCenter()
		{
			Vector3d transcenter = new Vector3d();
			foreach (PolyProxy box in this.polyproxies)
			{
				transcenter += box.center.pos;
			}
			transcenter /= this.polyproxies.Count;
			this.roomtrans = Matrix4d.TranslationMatrix(transcenter);
			this.roomtrans_inv = Matrix4d.TranslationMatrix(new Vector3d() - transcenter);

			// set the light position
			Vector3d maxCoord = new Vector3d(double.MinValue, double.MinValue, double.MinValue);
		//	Vector3d minCoord = new Vector3d(double.MaxValue, double.MaxValue, double.MaxValue);
			foreach (PolyProxy box in this.polyproxies)
			{
				foreach (Point3 v in box.points3)
				{
					maxCoord = Vector3d.Max(maxCoord, v.pos);
			//		minCoord = Vector3d.Min(minCoord, v.pos);
				}
			}
			this.lightpos = 1.5 * maxCoord;
		}
		private Matrix3d ComputePolygonHomography(Polygon face)
		{
			if (face.localframe == null) face.ComputeLocalFrame();
			int n = face.points3.Count;
			Vector2d[] src = new Vector2d[n];
			Vector2d[] tar = new Vector2d[n];
			for (int i = 0; i < n; ++i)
			{
				src[i] = face.points[i].pos;
				tar[i] = face.localframe.GetPointLocalCoord(face.points3[i].pos).ToVector2d();
			}
			return CameraCalibrator.ComputeHomographyMatrixCV(src, tar);
		}
		private Vector3d ProjectScreenPoint2PolyFace(Vector2d pt, Polygon face)
		{
			if (face.H == null)
			{
				face.H = ComputePolygonHomography(face);
			}
			Matrix3d H = face.H;
			Vector3d planepoint = H * new Vector3d(pt, 1);
			planepoint.HomogenousNormalize();
			planepoint.z = 0;
			return (face.localframe.R_inv * planepoint + face.localframe.o);
		}
		private void GetActualGLPoint(ref Vector2d p)
		{
			p -= this.imgpos;
			p /= this.sacleRatio;
		}
		private void GetActualGLVector(ref Vector2d p)
		{
			p /= this.sacleRatio;
		}	
		private void GetProxyLineSegmentsP3(PolyProxy proxy)
		{
			foreach (Polygon plg in proxy.polygons)
			{
				plg.ComputeLinesegmentsP3();
			}
		}
		private void GetAllPolygons()
		{
			List<Polygon> plys = new List<Polygon>();
			foreach (PolyProxy px in this.polyproxies)
			{
				plys.AddRange(px.polygons);
			}
			int index = 0;
			foreach (Polygon plg in plys)
			{
				plg.globalindex = index++;
			}
			this.allpolygons = plys;
		}
		private void AssignProxyIndex()
		{
			int boxindex = 0;
			foreach (PolyProxy box in this.polyproxies)
			{
				box.index = boxindex;
				if (box.segindex == -1)
					box.segindex = (boxindex + 1);
				boxindex++;
			}
		}
		private void ComputeProxyRenderSize()
		{
			if (this.cameracalibrator == null) return;
			Vector3d c = this.cameracalibrator.camera_R.Inverse() * this.cameracalibrator.camera_t;
			double l = c.Length();
			this.proxyrendersize = l * 1e-3;
		}


		/* algorithms */
		private Vector3d GLUProject(Vector3d p)
		{
			int[] viewport = new int[4];// {0,0,this.imgwidth,this.imgheight};
			double x, y, z;
			Gl.glGetIntegerv(Gl.GL_VIEWPORT, viewport);
			Glu.gluProject(p.x, p.y, p.z, this.glmodelviewmatrix, this.glprojectionmatrix, 
				viewport, out x, out y, out z);
			return new Vector3d(x, y, z);
		}
		private Matrix3d GetHomography(Shape2D src, Shape2D tar)
		{
			if (src.points.Count != tar.points.Count) return null;
			int n = src.points.Count;
			Vector2d[] srcv2 = new Vector2d[n];
			Vector2d[] tarv2 = new Vector2d[n];
			for (int i = 0; i < n; ++i)
			{
				srcv2[i] = src.points[i].pos;
				tarv2[i] = tar.points[i].pos;
			}
			return CameraCalibrator.ComputeHomographyMatrixCV(tarv2, srcv2);
		}
		
		
		// fitting
		public void FitCuboids2Hexagons()
		{
			if (this.hexagons == null || this.hexagons.Count < 1) return;

			this.polyproxies.Clear();

			// fit
			int k = this.hexagons.Count;
			foreach (Polygon hull in this.hexagons)
			{
				PolyProxy proxy = this.FitHexagonBox(hull);

				if (proxy == null) throw new ArgumentException("@FitCuboids2Hulls, failed to fit a proxy!");

				proxy.index = hull.index;
				this.polyproxies.Add(proxy);
			}

			// compute 3d info
			foreach (PolyProxy box in this.polyproxies)
				this.FinalizeProxy(box);
		}
		private PolyProxy FitHexagonBox(Polygon hex)
		{
			CameraCalibrator cam = this.cameracalibrator;
			Vector2d z = hex.points[0].pos;
			Vector2d u = hex.points[1].pos;
			Vector2d v = hex.points[2].pos;
			Vector2d w = hex.points[3].pos;
			Vector3d line = new Vector3d(u, 1).Cross(new Vector3d(v, 1));
			Vector3d line1 = new Vector3d(z, 1).Cross(new Vector3d(this.vp2d[2], 1));
			u = line.Cross(line1).HomogenousNormalize().ToVector2d();
			Vector3d u3 = cam.ComputePointGround3dPos(u);
			Vector3d v3 = cam.ComputePointGround3dPos(v);
			Vector3d w3 = cam.ComputePointGround3dPos(w);
			Vector3d uv = (v3 - u3).Normalize();
			Vector3d vw = w3 - v3;
			v3 = v3 + vw.Dot(uv) * uv;
			Vector3d t3 = u3 + w3 - v3;
			double h = cam.Compute3DCoordinate(z, u).z;
			Vector3d h3 = new Vector3d(0, 0, h);
			Point3[] pts3 = new Point3[8] {
				new Point3(u3),
				new Point3(v3),
				new Point3(w3),
				new Point3(t3),
				new Point3(u3+h3),
				new Point3(v3+h3),
				new Point3(w3+h3),
				new Point3(t3+h3)
			};
			Point2[] pts2 = new Point2[8] {
				new Point2(cam.ComputePointImgPos(pts3[0].pos)),
				new Point2(cam.ComputePointImgPos(pts3[1].pos)),
				new Point2(cam.ComputePointImgPos(pts3[2].pos)),
				new Point2(cam.ComputePointImgPos(pts3[3].pos)),
				new Point2(cam.ComputePointImgPos(pts3[4].pos)),
				new Point2(cam.ComputePointImgPos(pts3[5].pos)),
				new Point2(cam.ComputePointImgPos(pts3[6].pos)),
				new Point2(cam.ComputePointImgPos(pts3[7].pos))
			};
			PolyProxy box = new PolyProxy(pts2, pts3);
			box.MakeBox();

			return box;
		}


		// interface feedback
		public void SetCalibrateIndex()
		{
			if (this.hexagons == null) return;
			int index = 0;
			foreach (Polygon plg in this.hexagons)
			{
				if (plg.points.Contains(this.selectedpoint))
				{
					this.calibrateindex = index;
				}
				index++;
			}
		}
		public void AddHexagonPoint()
		{
			if (this.selectedpoint != null)
			{
				this.currhexagonpoints.Add(this.selectedpoint);
				if (this.currhexagonpoints.Count == 6)
				{
					if (this.hexagons == null) this.hexagons = new List<Polygon>();

					Polygon hex = new Polygon(this.currhexagonpoints.ToArray());
					this.hexagons.Add(hex);

					this.currhexagonpoints.Clear();
				}
			}
		}
		public void CalibrateUpdate()
		{
			if (this.selectedpoint != null)
				this.selectedpoint.UpdateOldPos();
		}
		public void SelectSixPoints(Vector2d mousepos)
		{
			if (this.hexagons == null) return;

			this.GetActualGLPoint(ref mousepos);
			double min = double.MaxValue; Point2 p = null;
			foreach (Polygon plg in this.hexagons)
			{
				if (plg.points == null) return;
				foreach (Point2 pt in plg.points)
				{
					double d = (pt.pos - mousepos).Length();
					if (d < min)
					{
						min = d;
						p = pt;
					}
				}
				if (min < 50)
					this.selectedpoint = p;
				else
					this.selectedpoint = null;
			}
		}
		public void CreateHexagonPoint(Vector2d mousepos)
		{
			this.GetActualGLPoint(ref mousepos);
			this.selectedpoint = new Point2(mousepos);
		}
		public void AdjustCalibratingPoint(Vector2d off)
		{
			off /= this.sacleRatio;
			if (this.selectedpoint != null)
				this.selectedpoint.Translate(off);
		}
		public void ReleaseSelection()
		{
			this.selectedpoint = null;
			this.has3dobjects = true;
		}
		private void FinalizeProxy(PolyProxy proxy)
		{
			this.GetProxyLineSegmentsP3(proxy);
		}

		// rendering entities
		public void RenderSense()
		{
			this.Render2dScene();
			this.Render3dScene();
			this.HighLightSelection();
		}
        public void Render2dScene()
        {	
			Gl.glPushMatrix();

			Gl.glViewport(0, this.parentview.Size.Height-this.glviewporth, this.glviewportw, this.glviewporth);

			Gl.glMatrixMode(Gl.GL_PROJECTION);
			Gl.glLoadIdentity();

			Glu.gluOrtho2D(0, this.glviewportw, this.glviewporth, 0);

			Gl.glMatrixMode(Gl.GL_MODELVIEW);
			Gl.glLoadIdentity();

            Gl.glPushMatrix();

			Gl.glTranslated(imgpos.x, imgpos.y, 0.0);
			Gl.glScaled(this.sacleRatio, this.sacleRatio, this.sacleRatio);
			Gl.glColor3f(1.0f, 1.0f, 1.0f);

			if (this.renderLayerImage)
				this.RenderVanishingLines();
			
			#region  Render Image
			if (renderLayerImage)
			{
				Gl.glEnable(Gl.GL_TEXTURE_2D);
				Gl.glTexEnvi(Gl.GL_TEXTURE_ENV, Gl.GL_TEXTURE_ENV_MODE, (int)Gl.GL_REPLACE);
				Gl.glBindTexture(Gl.GL_TEXTURE_2D, layertxtid[0]);
				Gl.glBegin(Gl.GL_QUADS);
				Gl.glTexCoord2d(0, 0);
				Gl.glVertex2d(0, 0);
				Gl.glTexCoord2d(1, 0);
				Gl.glVertex2d(this.imgwidth, 0);
				Gl.glTexCoord2d(1, 1);
				Gl.glVertex2d(this.imgwidth, this.imgheight);
				Gl.glTexCoord2d(0, 1);
				Gl.glVertex2d(0, this.imgheight);
				Gl.glEnd();
				Gl.glDisable(Gl.GL_TEXTURE_2D);
			}
			#endregion

	//		this.HighlightElements2D();

			Gl.glPopMatrix();

			Gl.glPopMatrix();

        }
		public void Render3dScene()
		{
			if (this.glprojectionmatrix == null) return;

			Gl.glViewport(0, this.parentview.Size.Height - this.glviewporth, this.glviewportw, this.glviewporth);

			// -------------------------------------------------------------------------
			// draw 3d sense
			Matrix4d m = this.parentview.GetTransMatrix();
			m = this.roomtrans * m *this.roomtrans_inv;


			Gl.glPushMatrix();
			Gl.glMatrixMode(Gl.GL_PROJECTION);
			Gl.glLoadIdentity();
			Gl.glLoadMatrixd(this.glprojectionmatrix);

			Gl.glMatrixMode(Gl.GL_MODELVIEW);
			Gl.glLoadIdentity();
			Gl.glLoadMatrixd(this.glmodelviewmatrix);


			this.InitLighting();

			Gl.glPushMatrix();
			Gl.glMultMatrixd((m.Transpose()).ToArray());


			if (this.RenderBoxWithTexutreMap)
				this.RenderTextured3DScene();

			this.RenderPolyProxies3D(this.polyproxies, this.ProxyColor);// Color.FromArgb(230, 219, 24));
				
			this.HighlightElements3D();

			Gl.glPopMatrix();
			this.DisableLighting();
			Gl.glPopMatrix();
		}

		// render elements
		private void RenderTextured3DScene()
		{
			Gl.glShadeModel(Gl.GL_SMOOTH);
			
			Gl.glPolygonMode(Gl.GL_FRONT_AND_BACK, Gl.GL_FILL);
			Gl.glEnable(Gl.GL_LIGHTING);
			Gl.glEnable(Gl.GL_NORMALIZE);

		//	Gl.glDisable(Gl.GL_CULL_FACE);
			Gl.glEnable(Gl.GL_DEPTH_TEST);

			Gl.glEnable(Gl.GL_BLEND);
			Gl.glBlendFunc(Gl.GL_SRC_ALPHA, Gl.GL_ONE_MINUS_SRC_ALPHA);
			
			Gl.glEnable(Gl.GL_ALPHA_TEST);
			Gl.glAlphaFunc(Gl.GL_GREATER, 0.4f);

			foreach (PolyProxy b in this.polyproxies)
			{
				foreach (Polygon poly in b.polygons)
				{
					if (!this.TextureAllPolygons && poly.textureindex == -1) continue;
					if (poly.textid == null) continue;

					uint texid = usegltexturemapping ? this.layertxtid[0] : poly.textid[0];

					Gl.glEnable(Gl.GL_TEXTURE_2D);
					Gl.glTexEnvi(Gl.GL_TEXTURE_ENV, Gl.GL_TEXTURE_ENV_MODE, (int)Gl.GL_REPLACE);
					Gl.glBindTexture(Gl.GL_TEXTURE_2D, texid);

					Gl.glColor3ub(255, 255, 255);
					Gl.glBegin(Gl.GL_POLYGON);

					Vector3d p, n;
					Vector2d t;
					t = poly.assignedtextures[0];
					if (usegltexturemapping)
						Gl.glTexCoord2d(poly.points[0].texcoord.x, poly.points[0].texcoord.y);
					else
						Gl.glTexCoord2d(t.x, t.y);
					p = poly.points3[0].pos;
					n = poly.points3[0].normal;
					Gl.glNormal3d(n.x, n.y, n.z);
					Gl.glVertex3d(p.x, p.y, p.z);


					t = poly.assignedtextures[1];
					if (usegltexturemapping)
						Gl.glTexCoord2d(poly.points[1].texcoord.x, poly.points[1].texcoord.y);
					else
						Gl.glTexCoord2d(t.x, t.y);
					p = poly.points3[1].pos;
					n = poly.points3[1].normal;
					Gl.glNormal3d(n.x, n.y, n.z);
					Gl.glVertex3d(p.x, p.y, p.z);


					t = poly.assignedtextures[2];
					if (usegltexturemapping)
						Gl.glTexCoord2d(poly.points[2].texcoord.x, poly.points[2].texcoord.y);
					else
						Gl.glTexCoord2d(t.x, t.y);
					p = poly.points3[2].pos;
					n = poly.points3[2].normal;
					Gl.glNormal3d(n.x, n.y, n.z);
					Gl.glVertex3d(p.x, p.y, p.z);


					t = poly.assignedtextures[3];
					if (usegltexturemapping)
						Gl.glTexCoord2d(poly.points[3].texcoord.x, poly.points[3].texcoord.y);
					else
						Gl.glTexCoord2d(t.x, t.y);
					p = poly.points3[3].pos;
					n = poly.points3[3].normal;
					Gl.glNormal3d(n.x, n.y, n.z);
					Gl.glVertex3d(p.x, p.y, p.z);

					Gl.glEnd();
					Gl.glDisable(Gl.GL_TEXTURE_2D);

				}
			}

			Gl.glDisable(Gl.GL_ALPHA_TEST);

			Gl.glDisable(Gl.GL_BLEND);
			Gl.glDisable(Gl.GL_DEPTH_TEST);
		//	Gl.glEnable(Gl.GL_CULL_FACE);

			Gl.glDisable(Gl.GL_LIGHTING);
			Gl.glDisable(Gl.GL_NORMALIZE);
		}
		private void RenderLineSegment(LineSegment s, int colorid)
		{
			Gl.glEnable(Gl.GL_LINE_SMOOTH);
			switch (colorid)
			{
				case 0:
					Gl.glColor3ub(255, 0, 0);
					break;
				case 1:
					Gl.glColor3ub(0, 255, 0);
					break;
				case 2:
					Gl.glColor3ub(0, 0, 255);
					break;
				case 3:
					Gl.glColor3ub(255, 255, 255);
					break;
				default:
					Gl.glColor3ub(255, 20, 147);
					break;
			}
			Vector2d u = s.u.pos, v = s.v.pos;
			double L = (u - v).Length();
			int intervals = (int)(L / 10);
			int k = intervals;
			double step = 1.0/k;
			Gl.glLineWidth(1.0f);
			Gl.glBegin(Gl.GL_LINES);
			for (int i = 0; i < k; i+=2)
			{
				double r1 = step * i, r2 = step*(i+1);
				Vector2d p = u * (1 - r1) + v * r1;
				Vector2d q = u * (1 - r2) + v * r2;
				Gl.glVertex2d(p.x, p.y);
				Gl.glVertex2d(q.x, q.y);
			}
			Gl.glEnd();
			Gl.glColor3ub(255, 255, 255);
			Gl.glPointSize(2.0f);
			Gl.glBegin(Gl.GL_POINTS);
			Gl.glVertex2d(s.u.pos.x, s.u.pos.y);
			Gl.glVertex2d(s.v.pos.x, s.v.pos.y);
			Gl.glEnd();
		}
		private void RenderPoint(Point2 p, float size, Color c)
		{
			Gl.glColor3ub(c.R,c.G,c.B);
			Gl.glPointSize(size);
			Gl.glBegin(Gl.GL_POINTS);
			Gl.glVertex2d(p.pos.x, p.pos.y);
			Gl.glEnd();
			Gl.glPointSize(1.0f);
		}
		private void RenderCalibrateNode(Point2 p, float size, Color c)
		{
			double x = p.pos.x, y = p.pos.y;
			Gl.glColor3ub(c.R, c.G, c.B);
			Gl.glLineWidth(1.0f);
			Gl.glBegin(Gl.GL_LINES);
			Gl.glVertex2d(x - size, y - size);
			Gl.glVertex2d(x + size, y - size);
			Gl.glVertex2d(x + size, y - size);
			Gl.glVertex2d(x + size, y + size);
			Gl.glVertex2d(x + size, y + size);
			Gl.glVertex2d(x - size, y + size);
			Gl.glVertex2d(x - size, y + size);
			Gl.glVertex2d(x - size, y - size);
			Gl.glEnd();

			Gl.glLineWidth(2.0f);
			Gl.glBegin(Gl.GL_LINES);
			Gl.glVertex2d(x - size, y - size);
			Gl.glVertex2d(x + size, y + size);
			
			Gl.glVertex2d(x - size, y + size);
			Gl.glVertex2d(x + size, y - size);
			Gl.glEnd();
			Gl.glPointSize(1.0f);
		}
		private void HighlightElements2D()
		{
			foreach (Point2 pt in this.currhexagonpoints)
			{
				this.RenderCalibrateNode(pt, CalibratingNodeSize, Color.Blue);
			}
			if (parentview.CurrentMode == ImageView.EnumOperationMode.AssignHexagon ||
				parentview.CurrentMode == ImageView.EnumOperationMode.AdjustHexagon)
			{
				if (this.hexagons != null)
				{
					this.RenderAllHexagons();
				}
			}
			if (this.selectedpoint != null)
			{
				this.RenderCalibrateNode(this.selectedpoint, CalibratingNodeSize, Color.Red);
			}
		}
		private void HighlightElements3D()
		{
		}
		private void HighLightSelection()
		{
			Gl.glPushMatrix();

			Gl.glViewport(0, this.parentview.Size.Height - this.glviewporth, this.glviewportw, this.glviewporth);

			Gl.glMatrixMode(Gl.GL_PROJECTION);
			Gl.glLoadIdentity();

			Glu.gluOrtho2D(0, this.glviewportw, this.glviewporth, 0);

			Gl.glMatrixMode(Gl.GL_MODELVIEW);
			Gl.glLoadIdentity();

			Gl.glPushMatrix();

			Gl.glTranslated(imgpos.x, imgpos.y, 0.0);
			Gl.glScaled(this.sacleRatio, this.sacleRatio, this.sacleRatio);
			Gl.glColor3f(1.0f, 1.0f, 1.0f);

			this.HighlightElements2D();

			Gl.glPopMatrix();

			Gl.glPopMatrix();
		}

		// for tests
		public static List<Vector3d> userballs = new List<Vector3d>();
		private void RenderCalibratingSegment(Point2 s, Point2 t, Color cc)
		{
			Gl.glDisable(Gl.GL_LIGHTING);
			Gl.glEnable(Gl.GL_LINE_SMOOTH);
			Gl.glHint(Gl.GL_LINE_SMOOTH_HINT, Gl.GL_NICEST);
			Vector2d u = s.pos, v = t.pos;
			double L = (u - v).Length();
			int intervals = (int)(L / 10);
			int k = intervals;
			double step = 1.0 / k;
			Color[] c = new Color[2] {
				Color.WhiteSmoke,
				Color.Black
			};
			Gl.glLineWidth(3.0f);
			Gl.glBegin(Gl.GL_LINES);
			for (int i = 0; i < k - 1; ++i)
			{
				Color clr = c[i % 2];
				Gl.glColor3ub(clr.R, clr.G, clr.B);
				double r1 = i * step, r2 = (i + 1) * step;
				Vector2d pos1 = s.pos * (1 - r1) + r1 * t.pos;
				Vector2d pos2 = s.pos * (1 - r2) + r2 * t.pos;
				Gl.glVertex2d(pos1.x, pos1.y);
				Gl.glVertex2d(pos2.x, pos2.y);
			}
			Gl.glEnd();
			Gl.glLineWidth(1.0f);

			this.RenderCalibrateNode(s, this.CalibratingNodeSize, cc);
			this.RenderCalibrateNode(t, this.CalibratingNodeSize, cc);

			Gl.glDisable(Gl.GL_LIGHTING);
		}
		private void RenderSolidSegment(LineSegment s, int colorid)
		{
			Gl.glEnable(Gl.GL_LINE_SMOOTH);
			Gl.glHint(Gl.GL_LINE_SMOOTH_HINT, Gl.GL_NICEST);
			switch (colorid)
			{
				case 0:
					Gl.glColor3ub(255, 0, 0);
					break;
				case 1:
					Gl.glColor3ub(0, 255, 0);
					break;
				case 2:
					Gl.glColor3ub(0, 0, 255);
					break;
				case 3:
					Gl.glColor3ub(255, 255, 0);
					break;
				default:
					Gl.glColor3ub(255, 20, 147);
					break;
			}
			Vector2d u = s.u.pos, v = s.v.pos;
			double L = (u - v).Length();
			int intervals = (int)(L / 10);
			int k = intervals;
			double step = 1.0/k;
			Gl.glLineWidth(2.0f);
			Gl.glBegin(Gl.GL_LINES);
			Gl.glVertex2d(s.u.pos.x, s.u.pos.y);
			Gl.glVertex2d(s.v.pos.x, s.v.pos.y);
			Gl.glEnd();
			//Gl.glColor3ub(255, 164, 255);
			//Gl.glPointSize(5.0f);
			//Gl.glBegin(Gl.GL_POINTS);
			//Gl.glVertex2d(s.u.pos.x, s.u.pos.y);
			//Gl.glVertex2d(s.v.pos.x, s.v.pos.y);
			//Gl.glEnd();
		}
		private void RenderVanishingLines()
		{
			if (this.vp2d == null) return;
			int n = this.vp2d.Length;
			Gl.glColor3ub(0, 255, 0);
			Gl.glLineWidth(2.0f);
			Gl.glBegin(Gl.GL_LINES);
			for (int i = 0; i < n; ++i)
			{
				Gl.glVertex2d(this.vp2d[i].x, this.vp2d[i].y);
				Gl.glVertex2d(this.vp2d[(i+1)%n].x, this.vp2d[(i+1)%n].y);
			}
			Gl.glEnd();
			Gl.glLineWidth(1.0f);
			Gl.glPointSize(8.0f);
			Gl.glBegin(Gl.GL_POINTS);
			Gl.glColor3ub(255, 0, 0);
			Gl.glVertex2d(this.vp2d[0].x, this.vp2d[0].y);
			Gl.glColor3ub(0, 255, 0);
			Gl.glVertex2d(this.vp2d[1].x, this.vp2d[1].y);
			Gl.glColor3ub(0, 0, 255);
			Gl.glVertex2d(this.vp2d[2].x, this.vp2d[2].y);
			Gl.glEnd();
			Gl.glPointSize(1.0f);
		}
		private void RenderShape3D(Vector3d[] points, Color c)
		{
			if (points == null) return;

			//Gl.glEnable(Gl.GL_BLEND);
			//Gl.glBlendFunc(Gl.GL_SRC_ALPHA, Gl.GL_ONE_MINUS_SRC_ALPHA);
			//Gl.glEnable(Gl.GL_LINE_SMOOTH);
			//Color temp = Color.FromArgb(60, c);
			//Gl.glColor4ub(temp.R, temp.G, temp.B, temp.A);
			//Gl.glBegin(Gl.GL_POLYGON);
			//foreach (Vector3d p in points)
			//{
			//    Gl.glVertex3d(p.x, p.y,p.z);
			//}
			//Gl.glEnd();
			//Gl.glDisable(Gl.GL_BLEND);

			Gl.glDisable(Gl.GL_LIGHTING);
			int k = points.Length;
			Gl.glLineWidth(2.0f);
			Gl.glColor3ub(c.R, c.G, c.B);
			Gl.glBegin(Gl.GL_LINES);
			for (int j = 0; j < k; ++j)
			{
				Vector3d u = points[j], v = points[(j + 1) % k];
				Gl.glVertex3d(u.x, u.y,u.z);
				Gl.glVertex3d(v.x, v.y,v.z);
			}
			Gl.glEnd();
			Gl.glLineWidth(1.0f);
			Gl.glEnable(Gl.GL_LIGHTING);
		}
		private void RenderShape2D(Shape2D shape, float linewidth, Color c, bool outlineonly)
		{
			if (shape == null) return;
			Gl.glDisable(Gl.GL_LIGHTING);
			if (!outlineonly)
			{
				Gl.glEnable(Gl.GL_BLEND);
				Gl.glBlendFunc(Gl.GL_SRC_ALPHA, Gl.GL_ONE_MINUS_SRC_ALPHA);
				Gl.glEnable(Gl.GL_LINE_SMOOTH);
				Color temp = Color.FromArgb(60, c);
				Gl.glColor4ub(temp.R, temp.G, temp.B, temp.A);
				switch (shape.shaptype)
				{
					case ShapeType2D.QUAD:
						Gl.glBegin(Gl.GL_QUADS);
						foreach (Point2 p in shape.points)
						{
							Gl.glVertex2d(p.pos.x, p.pos.y);
						}
						Gl.glEnd();
						break;
					case ShapeType2D.POLYGON:
						Gl.glBegin(Gl.GL_POLYGON);
						foreach (Point2 p in shape.points)
						{
							Gl.glVertex2d(p.pos.x, p.pos.y);
						}
						Gl.glEnd();
						break;
				}
				Gl.glDisable(Gl.GL_BLEND);
			}
			
			List<Point2> pts = shape.points;
			int k = pts.Count;
			Gl.glColor3ub(c.R, c.G, c.B);
			Gl.glLineWidth(linewidth);
			Gl.glBegin(Gl.GL_LINES);
			for (int j = 0; j < k; ++j)
			{
				Vector2d u = pts[j].pos, v = pts[(j + 1) % k].pos;
				Gl.glVertex2d(u.x, u.y);
				Gl.glVertex2d(v.x, v.y);
			}
			Gl.glEnd();
			Gl.glLineWidth(1.0f);
			Gl.glEnable(Gl.GL_LIGHTING);
		}
		private void RenderPolygon3D(Polygon plg, Color c, bool outlineonly)
		{
			if (plg == null) return;
			Gl.glDisable(Gl.GL_LIGHTING);
			if (!outlineonly)
			{
				Gl.glEnable(Gl.GL_BLEND);
				Gl.glBlendFunc(Gl.GL_SRC_ALPHA, Gl.GL_ONE_MINUS_SRC_ALPHA);
				Gl.glEnable(Gl.GL_LINE_SMOOTH);
				Color temp = Color.FromArgb(60, c);
				Gl.glColor4ub(temp.R, temp.G, temp.B, temp.A);

				Gl.glBegin(Gl.GL_POLYGON);
				foreach (Point3 p in plg.points3)
				{
					Gl.glVertex3d(p.pos.x, p.pos.y, p.pos.z);
				}
				Gl.glEnd();
				Gl.glDisable(Gl.GL_BLEND);
			}

			List<Point3> pts = plg.points3;
			int k = pts.Count;
			Gl.glBegin(Gl.GL_LINES);
			Gl.glColor3ub(c.R, c.G, c.B);
			for (int j = 0; j < k; ++j)
			{
				Vector3d u = pts[j].pos, v = pts[(j + 1) % k].pos;
				Gl.glVertex3d(u.x, u.y, u.z);
				Gl.glVertex3d(v.x, v.y, v.z);
			}
			Gl.glEnd();
			Gl.glEnable(Gl.GL_LIGHTING);
		}
		private void RenderPolyProxies3D(List<PolyProxy> proxies, Color c)
		{
			if (proxies.Count < 1) return; // int index = 0;
			if (this.RenderLineProxies)
			{
				Color cc = this.colorMall[5];
				foreach (PolyProxy proxy in proxies)
				{
					this.RenderPolyProxy3D(proxy, cc, FlatProxyLineWidth);
				}
			}
			if (this.RenderSolidProxies)
			{
				Gl.glEnable(Gl.GL_LIGHTING);
				foreach (PolyProxy proxy in proxies)
					this.RenderProxySolidOutLine(proxy, c);

				foreach (PolyProxy proxy in proxies)
					this.RenderPolyProxy3D_Solid(proxy, c, 20, false);

				foreach (PolyProxy proxy in proxies)
				    this.RenderPolyProxy3D_Solid(proxy, c, 10, true);

				foreach (PolyProxy proxy in proxies)
					this.RenderProxySolidOutLine(proxy, c);
				Gl.glDisable(Gl.GL_LIGHTING);
			}
		}
		private void RenderPolyProxy3D(PolyProxy proxy, Color c, float width)
		{
			Gl.glDisable(Gl.GL_LIGHTING);
			Gl.glColor3ub(c.R, c.G, c.B);
			Gl.glEnable(Gl.GL_LINE_SMOOTH);
			Gl.glLineWidth(width);
			Gl.glBegin(Gl.GL_LINES);
			foreach (Polygon poly in proxy.polygons)
			{
				for (int i = 0; i < poly.points3.Count; ++i)
				{
					Point3 p = poly.points3[i];
					Point3 q = poly.points3[(i + 1) % poly.points3.Count];
					Gl.glVertex3d(p.pos.x, p.pos.y, p.pos.z);
					Gl.glVertex3d(q.pos.x, q.pos.y, q.pos.z);
				}
			}
			Gl.glEnd();
			Gl.glLineWidth(1.0f);

			//Gl.glEnable(Gl.GL_BLEND);
			//Gl.glBlendFunc(Gl.GL_SRC_ALPHA, Gl.GL_ONE_MINUS_SRC_ALPHA);
			//Color temp = Color.FromArgb(alpha, c);
			//Gl.glColor4ub(temp.R, temp.G, temp.B, temp.A);
			//for (int i = proxy.polygons.Count - 1; i >= 0; --i)
			//{
			//    Gl.glBegin(Gl.GL_POLYGON);
			//    Polygon poly = proxy.polygons[i];
			//    foreach (Point3 pt in poly.points3)
			//    {
			//        Gl.glVertex3d(pt.pos.x, pt.pos.y, pt.pos.z);
			//    }
			//    Gl.glEnd();
			//}
			//Gl.glDisable(Gl.GL_BLEND);
			Gl.glEnable(Gl.GL_LIGHTING);
		}
		private void InitLighting()
		{
			Gl.glEnable(Gl.GL_LIGHTING);
			Gl.glEnable(Gl.GL_NORMALIZE);

			float diffuse = 1.0f;
			float specular = 0.1f;
			int shinness = 128;

			float[] LightDiffuse = { diffuse, diffuse, diffuse, 1.0f };
			float[] LightSpecular = { specular, specular, specular, 1f };
			float[] LightPosition = { (float)lightpos.x, (float)lightpos.y, (float)lightpos.z, 1f };
			Gl.glLightfv(Gl.GL_LIGHT0, Gl.GL_DIFFUSE, LightDiffuse);
			Gl.glLightfv(Gl.GL_LIGHT0, Gl.GL_SPECULAR, LightSpecular);
			Gl.glLightfv(Gl.GL_LIGHT0, Gl.GL_POSITION, LightPosition);
			Gl.glEnable(Gl.GL_LIGHT0);


			Gl.glEnable(Gl.GL_COLOR_MATERIAL);
			Gl.glColorMaterial(Gl.GL_FRONT, Gl.GL_AMBIENT_AND_DIFFUSE);
			//Gl.glColorMaterial(Gl.GL_FRONT, Gl.GL_SPECULAR);
			//Gl.glColorMaterial(Gl.GL_FRONT, Gl.GL_DIFFUSE);
			float[] mdiffuse = { 0.9f, 0.9f, 0.9f, 1f };
			float[] mspecular = { 0.9f, 0.9f, 0.99f, 1f };
			float[] mambient = { 0.9f, 0.9f, 0.99f, 1f };
			Gl.glMaterialfv(Gl.GL_FRONT, Gl.GL_SPECULAR, mspecular);
			Gl.glMaterialfv(Gl.GL_FRONT, Gl.GL_DIFFUSE, mdiffuse);
			Gl.glMaterialfv(Gl.GL_FRONT, Gl.GL_AMBIENT, mambient);
			Gl.glMaterialf(Gl.GL_FRONT, Gl.GL_SHININESS, shinness);

		}
		private void DisableLighting()
		{
			Gl.glDisable(Gl.GL_LIGHTING);
			Gl.glDisable(Gl.GL_NORMALIZE);
			Gl.glDisable(Gl.GL_COLOR_MATERIAL);
		}
		private void RenderProxySolidOutLine(PolyProxy proxy, Color c)
		{
			Gl.glShadeModel(Gl.GL_SMOOTH);

			Gl.glClearDepth(1.0f);							// Depth buffer setup             
			Gl.glEnable(Gl.GL_DEPTH_TEST);					// Enables Depth Testing             
			Gl.glDepthFunc(Gl.GL_LEQUAL);					// The Type Of Depth Test To Do     
			Gl.glHint(Gl.GL_PERSPECTIVE_CORRECTION_HINT, Gl.GL_NICEST);     /* Really Nice Perspective Calculations */
			Gl.glEnable(Gl.GL_CULL_FACE);
			Gl.glPolygonOffset(1f, 1f);

			Gl.glPushAttrib(Gl.GL_POLYGON_BIT | Gl.GL_LIGHTING_BIT);
			Gl.glPolygonMode(Gl.GL_FRONT_AND_BACK, Gl.GL_FILL);

			Color curvecolor = Color.FromArgb(c.R, c.G, c.B);
			Gl.glColor3ub(curvecolor.R, curvecolor.G, curvecolor.B);
			Glu.GLUquadric quad = Glu.gluNewQuadric();
			double radius = this.proxyrendersize;// (this.roomlayout.box.points3[2].pos - this.roomlayout.box.points3[0].pos).Length() * 0.0015;
			foreach (Point3 p in proxy.points3)
			{
				Gl.glPushMatrix();
				Gl.glTranslated(p.pos.x, p.pos.y, p.pos.z);
				Glu.gluSphere(quad, radius, 30, 20);
				Gl.glPopMatrix();
			}

			int count = 0;
			foreach (Polygon poly in proxy.polygons)
			{
				for (int i = 0; i < poly.points3.Count; ++i)
				{
					Point3 p = poly.points3[i];
					Point3 q = poly.points3[(i + 1) % poly.points3.Count];

					Vector3d pq = q.pos - p.pos;
					double h = pq.Length();
					double angle;
					angle = Math.Acos(pq.z / h) / Math.PI * 180;

					Gl.glPushMatrix();
					Gl.glTranslated(p.pos.x, p.pos.y, p.pos.z);
					Gl.glRotated(angle, -pq.y, pq.x, 0);
					Glu.gluCylinder(quad, radius, radius, h, 30, 20);
					Gl.glPopMatrix();
				}
				if (count++ >= 1)
					break;
			}
			if (proxy.polygons.Count > 1)
			{
				Polygon poly1 = proxy.polygons[0], poly2 = proxy.polygons[1];
				for (int i = 0; i < poly1.points.Count; ++i)
				{
					Point3 p = poly1.points3[i];
					Point3 q = poly2.points3[i];

					Vector3d pq = q.pos - p.pos;
					double h = pq.Length();
					double angle;
					angle = Math.Acos(pq.z / h) / Math.PI * 180;

					Gl.glPushMatrix();
					Gl.glTranslated(p.pos.x, p.pos.y, p.pos.z);
					Gl.glRotated(angle, -pq.y, pq.x, 0);
					Glu.gluCylinder(quad, radius, radius, h, 30, 20);
					Gl.glPopMatrix();
				}
			}
			Glu.gluDeleteQuadric(quad);
			Gl.glPopAttrib();
			Gl.glDisable(Gl.GL_CULL_FACE);
			Gl.glDisable(Gl.GL_DEPTH_TEST);
		}
		private void RenderPolyProxy3D_Solid(PolyProxy proxy, Color c, byte alpha, bool withdepthtest)
		{
			Color temp = Color.FromArgb(alpha, Color.Blue);
			Gl.glShadeModel(Gl.GL_SMOOTH);

			Gl.glClearDepth(1.0f);							// Depth buffer setup 
			if (!withdepthtest)
			{
				Gl.glDisable(Gl.GL_DEPTH_TEST);
			}
			else
			{
				Gl.glEnable(Gl.GL_DEPTH_TEST);					// Enables Depth Testing             
				Gl.glDepthFunc(Gl.GL_LEQUAL);					// The Type Of Depth Test To Do     
				temp = Color.FromArgb(0, 0, 0, 0);
			}
			Gl.glHint(Gl.GL_PERSPECTIVE_CORRECTION_HINT, Gl.GL_NICEST);     /* Really Nice Perspective Calculations */
			Gl.glDisable(Gl.GL_CULL_FACE);

			Gl.glEnable(Gl.GL_BLEND);
			Gl.glBlendFunc(Gl.GL_SRC_ALPHA, Gl.GL_ONE_MINUS_SRC_ALPHA);
			Gl.glPolygonMode(Gl.GL_FRONT_AND_BACK, Gl.GL_FILL);
			Gl.glColor4ub(temp.R, temp.G, temp.B, temp.A);
			for (int i = proxy.polygons.Count - 1; i >= 0; --i)
			{
				Polygon poly = proxy.polygons[i];
				Gl.glBegin(Gl.GL_POLYGON);
				Gl.glNormal3d(poly.normal.x, poly.normal.y, poly.normal.z);
				foreach (Point3 pt in poly.points3)
				{
					Gl.glVertex3d(pt.pos.x, pt.pos.y, pt.pos.z);
				}
				Gl.glEnd();
			}
			Gl.glDisable(Gl.GL_BLEND);

			Gl.glEnable(Gl.GL_CULL_FACE);
			Gl.glEnable(Gl.GL_DEPTH_TEST);
			Gl.glDepthFunc(Gl.GL_LEQUAL);
		}
		
		private void RenderTorus(double majorRadius, double minorRadius, int numMajor, int numMinor)
		{
			Vector3d vNormal = new Vector3d();
			double majorStep = 2.0f * Math.PI / numMajor;
			double minorStep = 2.0f * Math.PI / numMinor;
			int i, j;

			for (i = 0; i < numMajor; i++)
			{
				double a0 = i * majorStep;
				double a1 = a0 + majorStep;
				float x0 = (float)Math.Cos(a0);
				float y0 = (float)Math.Sin(a0);
				float x1 = (float)Math.Cos(a1);
				float y1 = (float)Math.Sin(a1);

				Gl.glBegin(Gl.GL_TRIANGLE_STRIP);
				for (j = 0; j <= numMinor; j++)
				{
					double b = j * minorStep;
					double c = Math.Cos(b);
					double r = minorRadius * c + majorRadius;
					double z = minorRadius * Math.Sin(b);

					//First point
					Gl.glTexCoord2f((float)i / (float)(numMajor), (float)(j) / (float)(numMinor));
					vNormal[0] = x0 * c;
					vNormal[1] = y0 * c;
					vNormal[2] = z / minorRadius;
					vNormal = vNormal.Normalize();
					Gl.glNormal3d(vNormal.x, vNormal.y, vNormal.z);
					Gl.glVertex3d(x0 * r, y0 * r, z);

					Gl.glTexCoord2f((float)(i + 1) / (float)(numMajor), (float)(j) / (float)(numMinor));
					vNormal[0] = x1 * c;
					vNormal[1] = y1 * c;
					vNormal[2] = z / minorRadius;
					vNormal = vNormal.Normalize();
					Gl.glNormal3d(vNormal.x, vNormal.y, vNormal.z);
					Gl.glVertex3d(x1 * r, y1 * r, z);
				}
				Gl.glEnd();
			}
		}
		private void RenderSolidSphere(Vector3d p, double radius, Color c)
		{
			Glu.GLUquadric quad = Glu.gluNewQuadric();
			Gl.glColor3ub(c.R, c.G, c.B);
			Gl.glPushMatrix();
			Gl.glTranslated(p.x, p.y, p.z);
			Glu.gluSphere(quad, radius, 30, 30);
			Gl.glPopMatrix();
			Glu.gluDeleteQuadric(quad);
		}
		private void RenderLines(List<LineSegment> clusters)
		{
			Gl.glDisable(Gl.GL_LIGHTING);
			Gl.glLineWidth(this.FlatProxyLineWidth);
			Gl.glBegin(Gl.GL_LINES);
			foreach (LineSegment line in clusters)
			{
				Color c = Color.Yellow;
				switch (line.vanishingindex)
				{
					case 0:
						c = this.colorMall[0];
						break;
					case 1:
						c = this.colorMall[1];
						break;
					case 2:
						c = this.colorMall[2];
						break;
					case 3:
						c = this.colorMall[3];
						break;
				}
				Gl.glVertex2d(line.u.pos.x, line.u.pos.y);
				Gl.glVertex2d(line.v.pos.x, line.v.pos.y);
			}
			Gl.glEnd();
			Gl.glLineWidth(2.0f);
			Gl.glEnable(Gl.GL_LIGHTING);
		}
		private void RenderPoint(Vector2d p, Color c, float size)
		{
			Gl.glPointSize(size);
			Gl.glColor3ub(c.R, c.G, c.B);
			Gl.glBegin(Gl.GL_POINTS);
			Gl.glVertex2d(p.x,p.y);
			Gl.glEnd();
			Gl.glPointSize(1.0f);
		}
		private void RenderObjHexagons()
		{
			if (this.hexagons == null) return;
			int j = 0;
			foreach (Polygon plg in this.hexagons)
			{
				RenderShape2D(plg, 1.0f, this.colorMall[j++ % this.colorMall.Length], false);
			//	RenderShape2D(plg,this.colorMall[j++%this.colorMall.Length]);
			}
		}
		private void RenderShape2D(Shape2D shape, Color c)
		{
			Gl.glColor3ub(c.R, c.G, c.B);
			Gl.glLineWidth(2.0f);
			Gl.glBegin(Gl.GL_LINES);
			foreach (LineSegment line in shape.linesegments)
			{
				Vector2d p = line.u.pos, q = line.v.pos;
				Gl.glVertex2d(p.x, p.y);
				Gl.glVertex2d(q.x, q.y);
			}
			Gl.glEnd();
			Gl.glLineWidth(1.0f);
		}
		private void RenderAllHexagons()
		{
			int j = 0;
			foreach (Polygon plg in this.hexagons)
			{
				Color c = this.colorMall[j++ % this.colorMall.Length];
				RenderHexagon(plg, c);
			}
		}
		private void RenderHexagon(Polygon plg, Color c)
		{
			if (plg.points == null) return;
			//foreach (Point2 pt in plg.points)
			//{
			//    this.RenderCalibrateNode(pt, CalibratingNodeSize, c);
			//}
			int nn = plg.points.Count;
			for (int i = 0; i < nn; ++i)
			{
				Point2 pt1 = plg.points[i], pt2 = plg.points[(i + 1) % nn];
				this.RenderCalibratingSegment(pt1, pt2, c);
			}
			this.RenderCalibrateNode(plg.points[0], this.CalibratingNodeSize, Color.Red);
		}

		private byte[] Image2Texture(Image<Bgr, byte> image)
		{
			int w = image.Width;
			int h = image.Height;
			byte[] txture = new byte[(w + 1) * (h + 1) * this.imgchannel];
			for (int i = 0; i < w; i++)
			{
				for (int j = 0; j < h; j++)
				{
					int shift = (j * w + i) * imgchannel;
					Bgr bgr = image[j, i];
					txture[shift + 0] = (byte)(bgr.Red);
					txture[shift + 1] = (byte)(bgr.Green);
					txture[shift + 2] = (byte)(bgr.Blue);
				}
			}
			return txture;
		}
		private byte[] Image2Texture(Image<Bgra, byte> image)
		{
			int w = image.Width;
			int h = image.Height;
			byte[] txture = new byte[(w + 1) * (h + 1) * (this.imgchannel+1)];
			for (int i = 0; i < w; i++)
			{
				for (int j = 0; j < h; j++)
				{
					int shift = (j * w + i) * imgchannel;
					Bgra bgr = image[j, i];
					txture[shift + 3] = (byte)(bgr.Alpha);
					txture[shift + 0] = (byte)(bgr.Red);
					txture[shift + 1] = (byte)(bgr.Green);
					txture[shift + 2] = (byte)(bgr.Blue);
					
				}
			}
			return txture;
		}

		#region IDisposable Members
        public void Dispose()
        {
        }
        #endregion

		#region texture mapping
		private bool usegltexturemapping = false;
		public bool UseGLTextureMapping
		{
			get { return usegltexturemapping; }
			set { 
				usegltexturemapping = value;
				this.parentview.Refresh();
			}
		}
		public int currenttexid = 0;
		Matrix<double> cvMat = new Matrix<double>(3, 3);
		private void CreateGLTxture(Image<Bgr,byte> img, uint[] txtid) // create gltexture
		{
			byte[] txture = this.Image2Texture(img);
			// -- create texture --
			Gl.glGenTextures(1, txtid);	// Create The Texture

			// Typical Texture Generation Using Data From The Bitmap
			Gl.glBindTexture(Gl.GL_TEXTURE_2D, txtid[0]);
			Gl.glTexImage2D(Gl.GL_TEXTURE_2D, 0, 4, (int)img.Width, (int)img.Height, 0, Gl.GL_RGBA, Gl.GL_UNSIGNED_BYTE, txture);
			Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_MIN_FILTER, Gl.GL_LINEAR);
			Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_MAG_FILTER, Gl.GL_LINEAR);

			Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_WRAP_S, Gl.GL_CLAMP_TO_EDGE);
			Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_WRAP_T, Gl.GL_CLAMP_TO_EDGE);
			Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_WRAP_R, Gl.GL_CLAMP_TO_EDGE);
			
		}
		private void CreateGLTxture(Image<Bgra, byte> img, uint[] txtid) // create gltexture
		{
			byte[] txture = this.Image2Texture(img);
			Gl.glGenTextures(1, txtid);	// Create The Texture

			// Typical Texture Generation Using Data From The Bitmap
			Gl.glBindTexture(Gl.GL_TEXTURE_2D, txtid[0]);
			Gl.glTexImage2D(Gl.GL_TEXTURE_2D, 0, 4, (int)img.Width, (int)img.Height, 0, Gl.GL_RGBA, Gl.GL_UNSIGNED_BYTE, txture);
			Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_MIN_FILTER, Gl.GL_LINEAR);
			Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_MAG_FILTER, Gl.GL_LINEAR);

			Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_WRAP_S, Gl.GL_CLAMP_TO_EDGE);
			Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_WRAP_T, Gl.GL_CLAMP_TO_EDGE);
			Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_WRAP_R, Gl.GL_CLAMP_TO_EDGE);

		}
		#endregion
		
	}
}
