﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Collections;
using System.ComponentModel;
using MyGeometry;
using System.Drawing;
using System.IO;


using Emgu.CV;
using Emgu.CV.Structure;
using Emgu.CV.UI;


namespace i3_ImageManipulator
{
	public unsafe partial class ImageManipulator : IDisposable
    {
		// ---------------------------------------------------------------------------------------
		// This part of code is used to jointly optimize the 3d scene
		// to the image objects (hexagons).
		// This parallelism-based regression strategy is iterated, each time when a new 
		// configuration of vps is obtained, the box hulls will be updated accordingly, and 
		// an objects-to-hulls fitting error is computed based on the new configuration, and 
		// compared to the previous error. The iteration stops when convergence is reached.
		// ---------------------------------------------------------------------------------------
		private int initialboxindex = 0;
		
		
		// initialize the scene and the cameras from the hexagons.
		// operates in three steps: 
		// first from each hexagon, computes its optimized 3d scene; 
		// second, the scene from the one hexagon which gives 
		// the least fitting error is selected as the initial scene.
		// third, based on the initial scene, we jointly optimize the cameras 
		// and the 3d scene while conforming to the relations.
		public void InitialCalibrate()
		{
			// 1. calibrate from each hexagon for each frame
			int khexagons = this.hexagons.Count;

			
			double minfiterror = 1e8;
			bool successful = false;
			List<PolyProxy> bestscene = new List<PolyProxy>(); CameraCalibrator bestcamera = new CameraCalibrator();
			for (int i = 0; i < khexagons; ++i)
			{
				// calibrate using hexagon i
				if (this.HexagonCalibrate(this.hexagons[i], this.hexagons))
				{
					successful = true;
					double error = this.ComputeSceneFittingError(this.cameracalibrator, this.polyproxies, this.hexagons);
					if (error < minfiterror)
					{
						minfiterror = error;

						this.CopyScene(bestscene, this.polyproxies);
						this.CopyCamera(bestcamera, this.cameracalibrator);

						// optimize the scene
						double fiterror = this.JointOptimize(this.hexagons);
						if (fiterror < minfiterror && minfiterror - fiterror > 5)
						{
							minfiterror = fiterror;
								
							this.CopyScene(bestscene, this.polyproxies);
							this.CopyCamera(bestcamera, this.cameracalibrator);

						}
					}
				}
			}

			if (!successful)
			{
				Program.OutputText("@InitialCalibrate, no hexagon is calibratable!", true);
				return;
			}


			Program.OutputText("best proxy fitting error: " + minfiterror, true);

			this.CopyScene(this.polyproxies, bestscene);
			this.CopyCamera(this.cameracalibrator, bestcamera);
			this.cameracalibrator.GetGLMatrices(out this.glprojectionmatrix, out this.glmodelviewmatrix,
				this.glviewportw, this.glviewporth, 0.01, 100);
			this.vp2d = this.cameracalibrator.GetVPs();


			// set gl view
			this.SetTranslationCenter();
			this.ComputeProxyRenderSize();
		}


		private bool HexagonCalibrate(Polygon plg, List<Polygon> plgs)
		{
			// find the projection matrix
			Vector2d[] imgpts = new Vector2d[6] {
	            plg.points[0].pos,
	            plg.points[1].pos,
	            plg.points[2].pos,
	            plg.points[3].pos,
	            plg.points[4].pos,
	            plg.points[5].pos
	        };
			Vector3d[] spacepoints = new Vector3d[6] { // simplify to one case
				new Vector3d(1,-1,1),
				new Vector3d(1,-1,0),
				new Vector3d(1,1,0),
				new Vector3d(-1,1,0),
				new Vector3d(-1,1,1),
				new Vector3d(-1,-1,1)
			};

			double[,] mat = CameraCalibrator.ComputeProjectionMatrixP2(imgpts, spacepoints);

			// get initial guess from mat M$
			Vector3d m1 = new Vector3d(mat[0, 0], mat[1, 0], mat[2, 0]); // first column
			Vector3d m2 = new Vector3d(mat[0, 1], mat[1, 1], mat[2, 1]); // second column
			Vector3d m3 = new Vector3d(mat[0, 2], mat[1, 2], mat[2, 2]); // third column

			// solve directly
			double u = this.imgwidth / 2, v = this.imgheight / 2;
			double a1 = m1[0], b1 = m1[1], c1 = m1[2];
			double a2 = m2[0], b2 = m2[1], c2 = m2[2];
			double a3 = m3[0], b3 = m3[1], c3 = m3[2];
			Vector3d b = new Vector3d(-(a1 * a2 + b1 * b2), -(a1 * a3 + b1 * b3), -(a3 * a2 + b3 * b2));
			Matrix3d Q = new Matrix3d(
				new Vector3d(c1 * c2, c1 * c3, c3 * c2),
				new Vector3d(c1 * a2 + a1 * c2, c1 * a3 + a1 * c3, c3 * a2 + a3 * c2),
				new Vector3d(c1 * b2 + b1 * c2, c1 * b3 + b1 * c3, c3 * b2 + b3 * c2)
			);
			Vector3d output = Q.Inverse() * b;
			u = -output[1];
			v = -output[2];
			double f2 = output[0] - u * u - v * v;

			if (f2 < 0) return false;

			double f = Math.Sqrt(Math.Abs(f2));

			// output error
			double aa = a1 * a2 + b1 * b2 + c1 * c2 * (u * u + v * v + f * f) + (c1 * a2 + a1 * c2) * (-u) + (c1 * b2 + b1 * c2) * (-v);
			double bb = a1 * a3 + b1 * b3 + c1 * c3 * (u * u + v * v + f * f) + (c1 * a3 + a1 * c3) * (-u) + (c1 * b3 + b1 * c3) * (-v);
			double cc = a3 * a2 + b3 * b2 + c3 * c2 * (u * u + v * v + f * f) + (c3 * a2 + a3 * c2) * (-u) + (c3 * b2 + b3 * c2) * (-v);
			double ee = aa * aa + bb * bb + cc * cc;

			Program.OutputText("- direct solver error: " + ee, true);


			// compute W
			Matrix3d W = Matrix3d.IdentityMatrix();
			W[2, 2] = u * u + v * v;
			W[0, 2] = W[2, 0] = -u;
			W[1, 2] = W[2, 1] = -v;
			W[2, 2] += f * f;
			W *= (1 / f / f);

			double lambda = Math.Sqrt(m3.Dot(W * m3));
			double l1 = Math.Sqrt(m1.Dot(W * m1)) / lambda;
			double l2 = Math.Sqrt(m2.Dot(W * m2)) / lambda;

			Matrix<double> InvK = new Matrix<double>(3, 3);
			InvK.SetZero();
			InvK[0, 0] = InvK[1, 1] = 1.0 / f;
			InvK[0, 2] = -u / f;
			InvK[1, 2] = -v / f;
			InvK[2, 2] = 1.0;
			Matrix<double> M = new Matrix<double>(mat);
			Matrix<double> F = InvK.Mul(M);
			Vector3d r1 = new Vector3d(F[0, 0], F[1, 0], F[2, 0]).Normalize();
			Vector3d r2 = new Vector3d(F[0, 1], F[1, 1], F[2, 1]).Normalize();
			Vector3d r3 = new Vector3d(F[0, 2], F[1, 2], F[2, 2]).Normalize();
			Vector3d t = new Vector3d(F[0, 3], F[1, 3], F[2, 3]) * (1.0 / lambda);


			Matrix3d K = new Matrix3d();
			K[0, 0] = K[1, 1] = f;
			K[0, 2] = u;
			K[1, 2] = v;
			K[2, 2] = 1.0;
			Matrix3d R = new Matrix3d(r1, r2, r3);
			if (this.cameracalibrator == null) this.cameracalibrator = new CameraCalibrator();
			this.cameracalibrator.camera_K = K;
			this.cameracalibrator.camera_R = R;
			this.cameracalibrator.camera_t = t;
			this.cameracalibrator.ComputeParameters();

			// get the box
			Vector3d[] canonicalpoints = new Vector3d[8] {
				new Vector3d(1, -1, 0),
				new Vector3d(1, 1, 0),
				new Vector3d(-1, 1, 0),
				new Vector3d(-1, -1, 0),
				new Vector3d(1, -1, 1),
				new Vector3d(1, 1, 1),
				new Vector3d(-1, 1, 1),
				new Vector3d(-1, -1, 1)
			};

			Matrix4d S = Matrix4d.ScalingMatrix(l1, l2, 1);
			Point3[] pts3 = new Point3[8];
			Point2[] pts2 = new Point2[8];
			for (int i = 0; i < 8; ++i)
			{
				Vector3d pos = (S * new Vector4d(canonicalpoints[i], 0)).XYZ();
				pts3[i] = new Point3(pos);
				pts2[i] = new Point2(this.cameracalibrator.ComputePointImgPos(pos));
			}
			PolyProxy vbox = new PolyProxy(pts2, pts3);
			vbox.MakeBox();
			this.FinalizeProxy(vbox);

			this.cameracalibrator.GetGLMatrices(out this.glprojectionmatrix, out this.glmodelviewmatrix,
				this.glviewportw, this.glviewporth, 0.01, 100);
			this.vp2d = this.cameracalibrator.GetVPs();

			this.polyproxies.Clear();
			this.polyproxies.Add(vbox);

			// fit others
			PolyProxy initialbox = this.polyproxies[0];
			this.FitCuboids2Hexagons();

			// re-arrange the boxes, include the initial one
			double mindis = double.MaxValue; PolyProxy vbox2 = null;
			foreach (PolyProxy box in this.polyproxies)
			{
				double d = (box.center.pos - initialbox.center.pos).Length();
				if (d < mindis)
				{
					mindis = d;
					vbox2 = box;
				}
			}
			this.initialboxindex = this.polyproxies.IndexOf(vbox2);
			this.polyproxies[initialboxindex] = initialbox;

			this.RenderSolidProxies = true;
			this.has3dobjects = this.polyproxies.Count > 0;
			return true;
		}
		private void Calibrate(Matrix3d camera_K, Matrix3d camera_R, Vector3d camera_t)
		{
			this.cameracalibrator.camera_R = camera_R;
			this.cameracalibrator.camera_K = camera_K;
			this.cameracalibrator.camera_t = camera_t;
			this.cameracalibrator.ComputeParameters();
			this.cameracalibrator.GetGLMatrices(out this.glprojectionmatrix, out this.glmodelviewmatrix,
				this.glviewportw, this.glviewporth, 0.01, 100);
			this.vp2d = this.cameracalibrator.GetVPs();
		}
		private void CopyCamera(CameraCalibrator cam1, CameraCalibrator cam2)
		{
			cam1.camera_K = cam2.camera_K;
			cam1.camera_R = cam2.camera_R;
			cam1.camera_t = cam2.camera_t;
			cam1.ComputeParameters();
		}
		private void CopyScene(List<PolyProxy> s1, List<PolyProxy> s2)
		{
			s1.Clear();
			s1.AddRange(s2);
		}
		private void Optimize(List<Polygon> boxhulls, int initialindex)
		{
			// initialindex >= 0 means we optimize with set of boxes, given an initial box
			// which has less parameters (l1,l2) than the others (l1,l2,l3,theta,cx,cy)
			// find 3d-2d (new hull points) correspondence	
		
			//1. exclude occuluded proxies from optimization
			List<Polygon> vhulls = boxhulls;
			List<PolyProxy> vproxies = this.polyproxies;


			// 2. now optimize
			Vector3d[] hexagoncanonicalpoints = new Vector3d[6] { // simplify to one case
				new Vector3d(1,-1,1),
				new Vector3d(1,-1,0),
				new Vector3d(1,1,0),
				new Vector3d(-1,1,0),
				new Vector3d(-1,1,1),
				new Vector3d(-1,-1,1)
			};
			List<List<Vector2d>> imgpts = new List<List<Vector2d>>();
			List<List<Vector3d>> spacepts = new List<List<Vector3d>>();
			List<Vector2d> tmppts = new List<Vector2d>(); int index = 0;
			foreach (Polygon hull in vhulls)
			{
				PolyProxy box = vproxies[index];
				int n = hull.points.Count;
				List<Vector3d> spoints = new List<Vector3d>();
				List<Vector2d> ipoints = new List<Vector2d>();
				// special care for the initial box
				for (int i = 0; i < n; ++i)
				{
					Vector2d pt = hull.points[i].pos;
					ipoints.Add(pt);
				}
				spoints.AddRange(hexagoncanonicalpoints);
				imgpts.Add(ipoints);
				spacepts.Add(spoints);
				index++;
			}

			// get initial value for the projection matrix
			List<double> initial = new List<double>();
			initial.Add(this.cameracalibrator.camera_K[0, 0]);
			initial.Add(this.cameracalibrator.camera_K[0, 2]);
			initial.Add(this.cameracalibrator.camera_K[1, 2]);

			// camera rotation angles, a, b, r
			int sign = 1;
			RotMatrixToAngles(this.cameracalibrator.camera_R, out sign);
			Matrix3d RR = new Matrix3d();
			for (int i = 0; i < 3; ++i)
			{
				for (int j = 0; j < 3; ++j)
				{
					RR[i, j] = this.cameracalibrator.camera_R[i, j];
				}
			}
			if (sign < 0)
			{
				RR[0, 2] = -RR[0, 2];
				RR[1, 2] = -RR[1, 2];
				RR[2, 2] = -RR[2, 2];
			}
			Vector3d angles = AnglesFromRotation(RR);

			initial.Add(angles.x);
			initial.Add(angles.y);
			initial.Add(angles.z);
			// translation
			initial.Add(this.cameracalibrator.camera_t[0]);
			initial.Add(this.cameracalibrator.camera_t[1]);
			initial.Add(this.cameracalibrator.camera_t[2]);


			// boxes parameters
			List<List<int>> proxyParaIndex = new List<List<int>>();
			int jindex = 9; int cnt = 0;
			foreach (PolyProxy box in vproxies)
			{
				double[] parameters = GetBoxParameters(box);
				Matrix4d Mat = GetBoxParameterMat(parameters);
				if (initialindex == cnt)
				{
					// 2 paras
					List<int> indices = new List<int>();
					indices.Add(jindex++);
					indices.Add(jindex++);
					proxyParaIndex.Add(indices);

					initial.AddRange(new double[2] { parameters[1], parameters[2] });
				}
				else
				{
					// 6 paras
					List<int> indices = new List<int>();
					for (int k = 0; k < 6; ++k)
						indices.Add(jindex++);
					proxyParaIndex.Add(indices);

					initial.AddRange(parameters);
				}
				cnt++;
			}

			//-----------------------------------------------------------------------------
			// add semantic constrains here, such as symmetry, repetitions, coplanar, etc.
			//-----------------------------------------------------------------------------
			List<Pair<int, int>> sconstraints = new List<Pair<int, int>>();
			if (true)
			{
				int n = vproxies.Count;
				for (int i = 0; i < n - 1; ++i)
				{
					PolyProxy p1 = vproxies[i];
					List<int> indices1 = proxyParaIndex[i];
					for (int j = i + 1; j < n; ++j)
					{
						PolyProxy p2 = vproxies[j];
						List<int> indices2 = proxyParaIndex[j];
						if (this.IsApproxiRep(p1,p2))
						{
							sconstraints.Add(new Pair<int, int>(indices1[0], indices2[0]));
							sconstraints.Add(new Pair<int, int>(indices1[1], indices2[1]));

							int i2 = indices1.Count == 2 ? -1 : indices1[3];
							int j2 = indices2.Count == 2 ? -1 : indices2[3];
							sconstraints.Add(new Pair<int, int>(i2, j2));

							Program.OutputText("Repetition detected: (" + i + " " + j + ")", true);
						}
						else if (this.IsApproxiCoplanar(p1, p2))
						{
							int i2 = indices1.Count == 2 ? -1 : indices1[3];
							int j2 = indices2.Count == 2 ? -1 : indices2[3];
							
							sconstraints.Add(new Pair<int, int>(i2,j2));

							Program.OutputText("Coplanar detected: (" + i + " " + j + ")", true);
						}
					}
				}
			}

		//	sconstraints.Clear();

			//-----------------------------------------------------------------------------



			// optimize
			FittingJointOptimizer optimizer = new FittingJointOptimizer();
			optimizer.Init(initial.ToArray(), spacepts, imgpts, sconstraints, sign, true);
			double[] p = null;
			if (initialindex < 0)
				p = optimizer.LMOptimize_Single();
			else
				p = optimizer.LMOptimize_Joint(initialindex);


			// re-build
			Vector3d[] canonicalpoints = new Vector3d[8] {
				new Vector3d(1, -1, 0),
				new Vector3d(1, 1, 0),
				new Vector3d(-1, 1, 0),
				new Vector3d(-1, -1, 0),
				new Vector3d(1, -1, 1),
				new Vector3d(1, 1, 1),
				new Vector3d(-1, 1, 1),
				new Vector3d(-1, -1, 1)
			};

			Matrix3d K = new Matrix3d();
			K[0, 0] = K[1, 1] = p[0];
			K[0, 2] = p[1];
			K[1, 2] = p[2];
			K[2, 2] = 1.0;
			Matrix3d R = Angles2RotationRhs(p[3], p[4], p[5]);
			if (sign < 0)
			{
				R[0, 2] = -R[0, 2];
				R[1, 2] = -R[1, 2];
				R[2, 2] = -R[2, 2];
			}
			Vector3d t = new Vector3d(p[6], p[7], p[8]);
			this.Calibrate(K, R, t);

			// find new box 3d positions from the optimization
			index = 0; Bgr gr = new Bgr(Color.Red); int s = 9;
			List<PolyProxy> optimizedproxies = new List<PolyProxy>();
			int nn = vproxies.Count;
			for (int i = 0; i < nn; ++i)
			{
				List<Vector2d> ipoints = imgpts[index];
				Matrix4d MM = null;
				if (i == initialindex)
				{
					MM = GetBoxParameterMat(new double[6] {
						0, p[s], p[s+1], 1,0,0
					});
					s += 2;
				}
				else
				{
					MM = GetBoxParameterMat(new double[6] {
						p[s], p[s+1], p[s+2],p[s+3],p[s+4],p[s+5]
					});
					s += 6;
				}
				List<Point3> pts3 = new List<Point3>();
				List<Point2> pts2 = new List<Point2>();
				foreach (Vector3d pt in canonicalpoints)
				{
					Vector4d p4 = new Vector4d(pt.x, pt.y, pt.z, 1);
					Vector3d p3 = (MM * p4).XYZ();
					Vector2d p2 = this.cameracalibrator.ComputePointImgPos(p3);
					pts3.Add(new Point3(p3));
					pts2.Add(new Point2(p2));
				}
				PolyProxy vbox = new PolyProxy(pts2, pts3);
				vbox.MakeBox();
				this.FinalizeProxy(vbox);
				optimizedproxies.Add(vbox);
			}

			this.polyproxies = optimizedproxies;

		}
		private double JointOptimize(List<Polygon> hexes)
		{
			// this function jointly align the fitting results by segmentation boundaries
			// given the initial vps, compute fitting hulls, then improve the fitting hulls
			// optimize the camera parameters according to the improved fitting hulls
			if (this.polyproxies.Count < 2)
			{
				Program.OutputText("@FittingJointOptimize, this.polyproxies.Count < 2, not using joint optimizer",true);
				return 1e8;
			}

			// optimize
			int itr = 0, maxiter = 1;
			double currerror = this.ComputeSceneFittingError(this.cameracalibrator, this.polyproxies, hexes);
			double preverror;
			Program.OutputText(" - projection error before optimization: " + currerror, true);
			do
			{
				preverror = currerror;

				// -----------------------------------------------------------------------------
				this.Optimize(hexes, this.initialboxindex);
				// -----------------------------------------------------------------------------

				currerror = this.ComputeSceneFittingError(this.cameracalibrator, this.polyproxies, hexes);

			} while (preverror > currerror && ++itr < maxiter);
			if (preverror < currerror)
			{
				Program.OutputText("@JointOptimize, preverror < currerror!, quit optimization",true);
				return 1e8;
			}
			Program.OutputText(" - projection error after optimization: " + currerror,true);
			this.AssignProxyIndex();
			return currerror;
		}


		private bool IsApproxiRep(PolyProxy p1, PolyProxy p2)
		{
			double ethr = 0.3;
			double e1 = Math.Abs(p1.xlen - p2.xlen) / (p1.xlen + p2.xlen);
			double e2 = Math.Abs(p1.ylen - p2.ylen) / (p1.ylen + p2.ylen);
			double e3 = Math.Abs(p1.zlen - p2.zlen) / (p1.zlen + p2.zlen);
			return (e1 < ethr && e2 < ethr && e3 < ethr);
		}
		private bool IsApproxiCoplanar(PolyProxy p1, PolyProxy p2)
		{
			double ratio = Math.Abs(p1.zlen - p2.zlen) / (p1.zlen + p2.zlen);
			return ratio < 0.1;
		}


		private double ComputeSceneFittingError(CameraCalibrator cam, List<PolyProxy> proxies, List<Polygon> hexes)
		{
			double error = 0; int index = 0;
			foreach (PolyProxy box in proxies)
			{
				int count = 0;
				foreach (int j in box.hexagonindice)
				{
					Vector2d pos = cam.ComputePointImgPos(box.points3[j].pos);
					error += (pos - hexes[index].points[count++].pos).Length();
				}
				index++;
			}
			return error;
		}
		private double ComputeHullObjectDistance(Shape2D hull, Point2[] objconvexhull)
		{
			double object2hulldis = 0; int num = 0;
			foreach (Point2 pt in objconvexhull)
			{
				double min = double.MaxValue;
				foreach (LineSegment line in hull.linesegments)
				{
					double dis = Distance2LineSegment(pt.pos, line.u.pos, line.v.pos);
					if (min > dis)
					{
						min = dis;
					}
				}
				object2hulldis += min;
				num++;
			}
			object2hulldis /= num;

			double hull2objectdis = 0;
			foreach (Point2 point in hull.points)
			{
				double mindis = double.MaxValue;
				foreach (Point2 pt in objconvexhull)
				{
					double d = (pt.pos - point.pos).Length();
					if (mindis > d)
					{
						mindis = d;
					}
				}
				hull2objectdis += mindis;
			}
			hull2objectdis /= hull.points.Count;
			return hull2objectdis + object2hulldis;
		}

	}
}
