﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using System.Drawing;
using Emgu.CV;
using Emgu.Util;
using MyGeometry;

//using MyCholmodSolver;

using System.IO;
using Emgu.CV.Structure;

namespace i3_ImageManipulator
{
	public enum MutualRelation { PARALLEL, COPLANAR, CONTAINER, ONOBJECT, REPETITION, SYMMETRY, JOINT, HEIGHTPRESERVE, NONE };
	public enum TransformStyle { Translation = 0, Rotation, Scaling, None }
	public enum ShapeType2D { TRIANGLE, QUAD, POLYGON, CIRCLLE, NONE };
	public enum ProxyConstrutionMode { AXISALIGNED, SPECIFYHEIGHT, OTHERS, NONE }
	public enum ProxyType { BOX, QUAD, JOINT, CYLINDER, CONE };
	public struct Transform
	{
		public Matrix4d T;
		public TransformStyle style;
		public Transform(Matrix4d t, TransformStyle s)
		{
			this.T = t;
			this.style = s;
		}
	}
	public class CoordinateFrame
	{
		public Vector3d o, x, y, z;
		public CoordinateFrame(Vector3d o_, Vector3d x_, Vector3d y_, Vector3d z_)
		{
			this.o = o_;
			this.x = x_;
			this.y = y_;
			this.z = z_;
			this.R_inv = new Matrix3d(x_, y_, z_);
		}
		public Matrix3d R_inv = null;
		public Vector3d GetPointLocalCoord(Vector3d p)
		{
			Vector3d po = (p - this.o);
			return new Vector3d(po.Dot(this.x), po.Dot(this.y), po.Dot(this.z));
		}
		public Vector3d GetPointSpaceCoord(Vector3d p)
		{
			return this.R_inv * p + this.o;
		}
		public Vector3d PointAtCoord(Vector2d coord)
		{
			return this.R_inv * new Vector3d(coord, 0) + this.o;
		}

	}
	public class Point2 : ICloneable
	{
		public bool occluded = false;
		public int index = -1;
		public Vector2d pos, oldpos;
		public Vector2d texcoord;
		public Point2(Vector2d pt)
		{
			this.oldpos = this.pos = pt;
			this.texcoord = this.pos;
		}
		public Point2()
		{
			this.oldpos = this.pos = new Vector2d();
			this.texcoord = this.pos;
		}
		public void Translate(Vector2d off)
		{
			this.pos = this.oldpos + off;
		}
		public void UpdateOldPos()
		{
			this.oldpos = this.pos;
		}
		public object Clone()
		{
			Point2 another = new Point2(this.pos);
			another.index = this.index;
			another.texcoord = this.texcoord;
			another.occluded = this.occluded;
			return another;
		}
	}
	public class Point3 : ICloneable
	{
		public int index = -1;
		public List<Point3> symmpoints = null;
		public Matrix3d R = null; // local frame encoding for manipulation, R does not change unless rotated
		public Vector3d pos, oldpos;
		public Vector3d normal;
		public Point3(Vector3d pt)
		{
			this.oldpos = this.pos = pt;
		}
		public Point3()
		{
			this.pos = new Vector3d();
		}
		public void UpdateOldPos()
		{
			this.oldpos = this.pos;
		}
		public object Clone()
		{
			Point3 another = new Point3(this.pos);
			if (this.R != null)
				another.R = new Matrix3d(this.R);
			another.index = this.index;
			another.normal = this.normal;
			return another;
		}
	}
	public class LineSegment : IComparable<LineSegment>
	{
		public Point2 u, v;
		public Point3 u3 = null, v3 = null;
		public int vanishingindex = -1;			// which vanishing direction it corresponds
		public double length = 0;
		public double fiterror = 0;
		public LineSegment(Point2 _u, Point2 _v)
		{
			this.u = _u;
			this.v = _v;
			this.vanishingindex = -1;
			this.length = this.GetLength();
		}
		public LineSegment(Point2 _u, Point2 _v, int _i)
		{
			this.u = _u;
			this.v = _v;
			this.vanishingindex = _i;
			this.length = this.GetLength();
		}
		public void Translate(Vector2d off)
		{
			this.u.Translate(off);
			this.v.Translate(off);
		}
		public double GetLength()
		{
			return (this.v.pos - this.u.pos).Length();
		}
		#region IComparable<LineSegment> Members
		public int CompareTo(LineSegment other)
		{
			return Math.Sign(length - other.length);
		}

		#endregion
	}
	public class Shape2D
	{
		public Vector2d center2;
		public Vector3d normal;
		public double radius2 = 0;
		public int textureindex = -1;
		public bool visible = true;
		public ShapeType2D shaptype = ShapeType2D.NONE;
		public List<Point2> points = null;
		public List<int> boxptindice = null;
		public List<LineSegment> linesegments = new List<LineSegment>();
		public Shape2D()
		{
		}
		public void AddPoint(Point2 pt)
		{
			if (this.points == null)
			{
				this.points = new List<Point2>();
				this.linesegments = new List<LineSegment>();
			}
			this.points.Add(pt);
			int n = this.points.Count;
			if (n < 2) return;
			this.linesegments.Add(
				new LineSegment(this.points[n - 2], this.points[n - 1])
			);
		}
		public void Close()
		{
			this.linesegments.Add(
				new LineSegment(this.points[this.points.Count - 1], this.points[0])
			);
			this.ComputeCenter();
		}
		public void ComputeCenter()
		{
			Vector2d c = new Vector2d();
			foreach (Point2 p in this.points)
			{
				c += p.pos;
			}
			this.center2 = c / this.points.Count;
		}
		public void ComputeRadius()
		{
			Vector2d c = new Vector2d();
			foreach (Point2 p in this.points)
			{
				c += p.pos;
			}
			this.center2 = c / this.points.Count;
			double sum = 0;
			foreach (Point2 p in this.points)
			{
				sum += (p.pos - center2).Length();
			}
			this.radius2 = sum / this.points.Count;
		}

		public static bool PointInPoly(Vector2d p, Shape2D shape)
		{
			return PointInPoly(p, shape.points);
		}
		public static bool PointInPoly(Vector2d p, List<Point2> points)
		{
			bool c = false;
			int n = points.Count;
			for (int i = 0, j = n - 1; i < n; j = i++)
			{
				if (((points[i].pos.y > p.y) != (points[j].pos.y > p.y)) &&
					(p.x < (points[j].pos.x - points[i].pos.x) * (p.y - points[i].pos.y) / (points[j].pos.y - points[i].pos.y) + points[i].pos.x))
					c = !c;
			}
			return c;
		}
		public static bool PointInPoly(Vector2d p, Point2[] points)
		{
			bool c = false;
			int n = points.Length;
			for (int i = 0, j = n - 1; i < n; j = i++)
			{
				if (((points[i].pos.y > p.y) != (points[j].pos.y > p.y)) &&
					(p.x < (points[j].pos.x - points[i].pos.x) * (p.y - points[i].pos.y) / (points[j].pos.y - points[i].pos.y) + points[i].pos.x))
					c = !c;
			}
			return c;
		}
		public static bool PointInPoly(Vector2d p, Vector2d[] shape)
		{
			int n = shape.Length;
			Vector2d o = shape[0];
			for (int i = 1; i < n - 1; ++i)
			{
				if (PointInTriangle(p, o, shape[i], shape[i + 1]))
					return true;
			}
			return false;
		}
		//public static bool PointInPoly(Vector2d p, Shape2D shape)
		//{
		//    int n = shape.points.Count;
		//    Vector2d o = shape.points[0].pos;
		//    for (int i = 1; i < n-1; ++i)
		//    {
		//        if (PointInTriangle(p,o,shape.points[i].pos,shape.points[i+1].pos))
		//            return true;
		//    }
		//    return false;
		//}
		public static bool PointInTriangle(Vector2d p, Vector2d a, Vector2d b, Vector2d c)
		{
			if (SameSide(p, a, b, c) && SameSide(p, b, a, c) && SameSide(p, c, a, b))
				return true;
			return false;
		}
		public static bool SameSide(Vector2d p1, Vector2d p2, Vector2d a, Vector2d b)
		{
			Vector3d A = new Vector3d(a);
			Vector3d B = new Vector3d(b);
			Vector3d P = new Vector3d(p1);
			Vector3d Q = new Vector3d(p2);
			Vector3d E = B - A;
			Vector3d cp1 = E.Cross(P - A);
			Vector3d cp2 = E.Cross(Q - A);
			if (cp1.Dot(cp2) >= 0)
				return true;
			return false;
		}
		public static double Point2PolyBoundaryDist(Vector2d p, Shape2D shape, out int segindex, out double lambda)
		{
			double mindist = double.MaxValue;
			segindex = -1;
			lambda = 0;
			for (int i = 0; i < shape.points.Count; ++i)
			{
				int next = (i + 1) % shape.points.Count;
				double templambda;
				double dist = ImageManipulator.Distance2LineSegment(p, shape.points[i].pos, shape.points[next].pos);
				ImageManipulator.Distance2LineSegment(p,shape.points[i].pos, shape.points[next].pos, out templambda);
				if (dist < mindist)
				{
					mindist = dist;
					segindex = i;
					lambda = templambda;
				}
			}
			return mindist;
		}
		public static double Point2PolyBoundaryDist(Vector2d p, List<Point2> points, out int segindex)
		{
			double mindist = double.MaxValue;
			segindex = -1;
			for (int i = 0; i < points.Count; ++i)
			{
				int next = (i + 1) % points.Count;
				double dist = ImageManipulator.Distance2LineSegment(p, points[i].pos, points[next].pos);
				if (dist < mindist)
				{
					mindist = dist;
					segindex = i;
				}
			}
			return mindist;
		}
		public static double Point2PolyBoundaryDist(Vector2d p, Vector2d[] shape, out int segindex, out double lambda)
		{
			double mindist = double.MaxValue;
			segindex = -1;
			lambda = 0;
			for (int i = 0; i < shape.Length; ++i)
			{
				int next = (i + 1) % shape.Length;
				double templambda;
				double dist = ImageManipulator.Distance2LineSegment(p, shape[i], shape[next]);
				ImageManipulator.Distance2LineSegment(p, shape[i], shape[next], out templambda);
				if (dist < mindist)
				{
					mindist = dist;
					segindex = i;
					lambda = templambda;
				}
			}
			return mindist;
		}
		public static double Point2PolyBoundaryDist(Vector2d p, Vector2d[] shape)
		{
			double mindist = double.MaxValue;
			for (int i = 0; i < shape.Length; ++i)
			{
				int next = (i + 1) % shape.Length;
				double dist = ImageManipulator.Distance2LineSegment(p, shape[i], shape[next]);
				mindist = Math.Min(mindist, dist);
			}
			return mindist;
		}
		public static double Point2PolyBoundaryDist2(Vector2d p, Shape2D shape, out int segindex, out double lambda)
		{
			double mindist = double.MaxValue;
			segindex = -1;
			lambda = 0;
			for (int i = 0; i < shape.points.Count; ++i)
			{
				int next = (i + 1) % shape.points.Count;
				double templambda;
				double dist = ImageManipulator.Distance2LineSegment(p, shape.points[i].pos, shape.points[next].pos, out templambda);
				if (dist < mindist)
				{
					mindist = dist;
					segindex = i;
					lambda = templambda;
				}
			}
			return mindist;
		}
	}
	public class Quad : Shape2D
	{
		public int vanishingindex = -1;
		public Point3[] points3 = new Point3[4];
		public Vector3d center;
		public Quad()
		{
			this.shaptype = ShapeType2D.QUAD;
		}
		public Quad(Vector2d a, Vector2d b, Vector2d c, Vector2d d)
		{
			this.shaptype = ShapeType2D.QUAD;
			this.points = new List<Point2>();
			this.points.Add(new Point2(a));
			this.points.Add(new Point2(b));
			this.points.Add(new Point2(c));
			this.points.Add(new Point2(d));
		}
		public Quad(Point2 a, Point2 b, Point2 c, Point2 d)
		{
			this.shaptype = ShapeType2D.QUAD;
			this.points = new List<Point2>();
			this.points.Add(a);
			this.points.Add(b);
			this.points.Add(c);
			this.points.Add(d);
		}
		public Quad(Point2[] pts)
		{
			this.shaptype = ShapeType2D.QUAD;
			this.points = new List<Point2>();
			this.points.AddRange(pts);
		}
	}
	public class Polygon : Shape2D, ICloneable
	{
		public Polygon opposite = null;
		public Polygon assignedpolygon = null;
		public Vector2d[] assignedtextures = new Vector2d[4] {
			new Vector2d(0,0),
			new Vector2d(1,0),
			new Vector2d(1,1),
			new Vector2d(0,1)
		};
		public uint[] textid = null;
		public Image<Bgra, byte> texture = null;

		public static double propagationpointdensity = 0;
		public CoordinateFrame localframe = null;
		public Matrix3d H = null; // homography matrix
		public Point3 center;
		public int prevattachindex = -1, attachindex = -1;
		public double depth = 0;
		public double sliceratio = 0; // for decomposition of a box
		public int sliceindex = -1; // for decomposition of a box
		public int index = -1;
		public int globalindex = -1;
		public int occludcount = 0;
		public PolyProxy hostproxy = null;
		public List<Polygon> mirrorsymmetries = null;
		public List<Symmetry> symmelements = null;
		public List<int> groupindex = new List<int>();
		public bool visited = false;
		public List<Point3> points3 = new List<Point3>();
		public List<Point3> propagatepoints3 = null; // use for propagation, much denser than the original points
		public List<int> propagateKs = null; // use for propagation
		public Polygon()
		{
			this.shaptype = ShapeType2D.POLYGON;
		}
		public Polygon(List<Point2> pts)
		{
			this.shaptype = ShapeType2D.POLYGON;
			this.points = pts;
			this.linesegments = new List<LineSegment>();
			int n = pts.Count;
			for (int i = 0; i < n; ++i)
			{
				this.linesegments.Add(
					new LineSegment(pts[i], pts[(i + 1) % n])
				);
			}
			this.ComputeCenter();
		}
		public Polygon(Point2[] pts)
		{
			this.shaptype = ShapeType2D.POLYGON;
			this.points = new List<Point2>();
			this.points.AddRange(pts);
			this.linesegments = new List<LineSegment>();
			int n = pts.Length;
			for (int i = 0; i < n; ++i)
			{
				this.linesegments.Add(
					new LineSegment(pts[i], pts[(i + 1) % n])
				);
			}
			this.ComputeCenter();
		}
		public void Compute()
		{
			this.ComputeCenter();
			this.ComputeCenter3();
			this.ComputeNormal();
		}
		public void ComputeCenter3()
		{
			Vector3d cc = new Vector3d();
			foreach (Point3 p in this.points3)
			{
				cc += p.pos;
			}
			cc /= this.points3.Count;
			if (this.center == null)
				this.center = new Point3(cc);
			else
				this.center.pos = cc;
		}
		public void ComputeNormal()
		{
			if (this.points3.Count < 2) return;
			this.normal = (this.points3[0].pos - this.points3[1].pos).Cross(
					this.points3[2].pos - this.points3[1].pos).Normalize();
		}
		public void ComputeLocalFrame()
		{
			if (this.points3 == null) return;
			Vector3d o = this.points3[0].pos;
			Vector3d u = this.points3[1].pos;
			Vector3d v = this.points3[this.points3.Count-1].pos;
			Vector3d z = (v - o).Cross(u - o).Normalize();
			Vector3d x = (u - o).Normalize();
			Vector3d y = z.Cross(x).Normalize();
			this.localframe = new CoordinateFrame(o, x, y, z);
		}
		public void ComputeLinesegmentsP3()
		{
			int i = 0, n = this.points3.Count;
			foreach (LineSegment line in this.linesegments)
			{
				line.u3 = this.points3[i];
				line.v3 = this.points3[(i + 1) % n];
				i++;
			}
		}
		public void CreatePropagatingPoints()
		{
			this.propagatepoints3 = new List<Point3>();
			this.propagateKs = new List<int>();
			int n = this.points3.Count;
			for (int i = 0; i < n; ++i)
			{
				Vector3d p = this.points3[i].pos, q = this.points3[(i + 1) % n].pos;
				double l = (p - q).Length();
				int K = (int)(l / propagationpointdensity);
				if (K < 2)
				{
		//			Program.OutputText("@CreatePropagatingPoints, K < 1",true);
					K = 2;
				}
		//		if (K > 10) K = 10;
				this.propagateKs.Add(K);
				double step = 1.0 / (K - 1);
				for (int j = 0; j < K; ++j)
				{
					double r = step * j;
					Vector3d pos = q * r + p * (1 - r);
					this.propagatepoints3.Add(new Point3(pos));
				}
			}
		}
		public void UpdatePropagatingPoints()
		{
			int n = this.points3.Count; int index = 0;
			for (int i = 0; i < n; ++i)
			{
				Vector3d p = this.points3[i].pos, q = this.points3[(i + 1) % n].pos;
				double l = (p - q).Length();
				int K = this.propagateKs[i];
				double step = 1.0 / (K - 1);
				for (int j = 0; j < K; ++j)
				{
					double r = step * j;
					Vector3d pos = q * r + p * (1 - r);
					this.propagatepoints3[index++].pos = pos;
				}
			}
		}
		public void PropagationUpdate()
		{
			this.ComputeCenter3();
			this.UpdatePropagatingPoints();
		}
		public object Clone()
		{
			List<Point2> points = new List<Point2>();
			List<Point3> points3 = new List<Point3>();
			foreach (Point2 p in this.points)
			{
				points.Add(p.Clone() as Point2);
			}
			foreach (Point3 p in this.points3)
			{
				points3.Add(p.Clone() as Point3);
			}
			Polygon another = new Polygon(points);
			another.points3 = points3;
			another.normal = this.normal;
			another.textureindex = this.textureindex;
			if (this.textid != null)
			another.textid = this.textid.Clone() as uint[];
			another.texture = this.texture;
			if (this.center != null) another.center = this.center.Clone() as Point3;
			another.ComputeLinesegmentsP3();
			another.CreatePropagatingPoints();
			return another;
		}
	}
	public class Hull : Shape2D, ICloneable
	{
		public int index = -1;
		public Hull host = null;
		public Hull attachhost = null;
		public Vector2d[] vps = null;
		public int occludcount = 0;
		public double depth = 0;
		public bool[] occluded = null;
		public Hull()
		{
			this.shaptype = ShapeType2D.POLYGON;
		}
		public Hull(List<Point2> pts)
		{
			this.shaptype = ShapeType2D.POLYGON;
			this.points = pts;
			this.linesegments = new List<LineSegment>();
			int n = pts.Count;
			for (int i = 0; i < n; ++i)
			{
				this.linesegments.Add(
					new LineSegment(pts[i], pts[(i + 1) % n])
				);
			}
			this.ComputeCenter();
		}
		public Hull(Point2[] pts)
		{
			this.shaptype = ShapeType2D.POLYGON;
			this.points = new List<Point2>();
			this.points.AddRange(pts);
			this.linesegments = new List<LineSegment>();
			int n = pts.Length;
			for (int i = 0; i < n; ++i)
			{
				this.linesegments.Add(
					new LineSegment(pts[i], pts[(i + 1) % n])
				);
			}
			this.ComputeCenter();
		}
		public object Clone()
		{
			List<Point2> points = new List<Point2>();
			foreach (Point2 pt in this.points)
			{
				points.Add(new Point2(pt.pos));
			}
			Hull another = new Hull(points);
			if (this.vps != null)
				another.vps = this.vps.Clone() as Vector2d[];
			another.depth = this.depth;
			another.attachhost = this.attachhost;
			another.host = this.host;
			another.index = this.index;
			if (this.boxptindice != null)
			{
				another.boxptindice = new List<int>();
				another.boxptindice.AddRange(this.boxptindice);
			}
			int index = 0;
			foreach (LineSegment line in another.linesegments)
			{
				line.vanishingindex = this.linesegments[index++].vanishingindex;
			}
			return another;
		}
	}
	public class PolygonGroup
	{
		public MutualRelation groupstyle = MutualRelation.NONE;
		public List<Polygon> polygons = null;
		public bool visited = false;
		public int index = -1;
		public PolygonGroup(List<Polygon> polys, int id)
		{
			this.polygons = polys;
			this.index = id;
		}
	}
	public class ProxyGroup
	{
		// group for which proxies form one object
		public List<ProxyGroup> reps = null;
		public int groupindex = -1;
		public List<PolyProxy> proxies = null;
		public bool visited = false;
		public ProxyGroup(List<PolyProxy> pxs, int id)
		{
			this.groupindex = id;
			this.proxies = pxs;
		}
	}
	
	public class PolyProxy : ICloneable, IComparable<PolyProxy>
	{
		// attach index refers to floor plane, ceiling plane and 3 wall planes.
		// 0 - floor, 1 - ceiling, 2 - left wall, 3 - middle wall 4 - right wall
		public List<int> enclosingindex = null; // currently not used any more
		public List<Vector2d> srctxtpos = null;
		
		public ProxyType type = ProxyType.QUAD;

		public bool isfixed = false;
		public bool deleted = false;
		public bool occuluded = false;
		public PolyProxy jointproxy = null;
		public List<PolyProxy> mirrorsymmetry = new List<PolyProxy>();
		public List<Symmetry> symmelement = new List<Symmetry>();

		public bool isjointhoster = false;
		public LineSegment jointline = null; // for joint object
		public bool excluedforpropagation = false;


		//  for MRF selection
		public int maxlevel = 0;
		public List<PolyProxy> kids = null;
		public void AddKid(PolyProxy px)
		{
			if (this.kids == null) this.kids = new List<PolyProxy>();
			this.kids.Add(px);
		}
		public int groupindex = -1; // for inter-obejct propagation
		public int index = -1;
		public int segindex = -1;
		public int refindex = -1;
		public int tmpindex = -1;
		public int level = 0;
		public double fittingcost = 0;

		public Vector2d[] GetConvexhull2d()
		{
			using (MemStorage storage = new MemStorage())
			{
				Seq<PointF> seq = new Seq<PointF>(storage);
				for (int i = 0; i < this.points.Count; ++i)
					seq.Push(ImageManipulator.Vector2PointF(this.points[i].pos));
				PointF[] convexhull = seq.GetConvexHull(Emgu.CV.CvEnum.ORIENTATION.CV_COUNTER_CLOCKWISE).ToArray();
				return convexhull.Select(pf => ImageManipulator.PointF2Vecter2d(pf)).ToArray();
			}
		}

		
		public CoordinateFrame frame = null;
		public List<TranslationMetaphor> transmetaphors = new List<TranslationMetaphor>();
		public RotationMetaphor rotationmetaphor = null; // only one

		public List<BendingMetaphor> bendingmetaphors = new List<BendingMetaphor>(); // for bending
		public List<Polygon> bendingslices = null; // for bending operation
		public List<PolyProxy> slicingboxes = null; // for bending operation

		public double xlen, ylen, zlen;
		public bool IntersectRoom(PolyProxy room)
		{
			double x = room.points3[0].pos.x;
			double y = room.points3[2].pos.y;
			double z = room.points3[4].pos.z;
			foreach (Point3 p in this.points3)
			{
				if (p.pos.x * x < 0 || Math.Abs(p.pos.x) > Math.Abs(x) + 1e-4 ||
					p.pos.y * y < 0 || Math.Abs(p.pos.y) > Math.Abs(y) + 1e-4 ||
					p.pos.z * z < 0 || Math.Abs(p.pos.z) > Math.Abs(z) + 1e-4)
					return true;
			}
			return false;
		}
		public bool Intersect(PolyProxy another)
		{
			foreach (Point3 p in this.points3)
			{
				if (another.Contains(p.pos))
					return true;
			}
			foreach (Point3 p in another.points3)
			{
				if (this.Contains(p.pos))
					return true;
			}
			return false;
		}
		public bool Contains(Vector3d p)
		{
			Vector3d abc = this.frame.GetPointLocalCoord(p);
			double e1 = Math.Abs(abc.x) - this.xlen / 2;
			double e2 = Math.Abs(abc.y) - this.ylen / 2;
			double e3 = Math.Abs(abc.z) - this.zlen / 2;
			return  e1 < -1e-4 && e2 < -1e-4 && e3 < -1e-4;
		}

		public Vector2d[] vps = null;

		public Hull convexhull = null;
		public int attachindex = 0;
		public int prevattachindex = 0;
		public bool visited = false;
		public double volume = -1;
		public double depth = 0;
		public double DiagLength()
		{
			if (this.points3 == null) return 1e-8;
			return (this.points3[6].pos - this.points3[0].pos).Length();
		}
		public Point3 center;
		public Point2 center2;
		public List<Polygon> txtquads = null;
		public List<LineSegment> enclosingimglines = null;
		public Quad srcenclosingtxtquad = null; // texture mapping, source enclosing quad, for cylindric/sphere objects
		public Quad tarenclosingtxtquad = null; // texture mapping, target enclosing quad
		public List<Polygon> polygons = new List<Polygon>();
		public List<Point2> points = new List<Point2>();
		public List<Point3> points3 = new List<Point3>();
		public List<Point[]> objprofile = null;
		public PolyProxy onproxy = null; // object that it lies on
		public List<PolyProxy> onobjects = null; // objects that lies on it
		public void AddOnObject(PolyProxy px)
		{
			if (this.onobjects == null) this.onobjects = new List<PolyProxy>();
			this.onobjects.Add(px);
		}
		public PolyProxy() { }
		public PolyProxy(Point2[] pts, Point3[] pts3)
		{
			this.Create(pts, pts3);
		}
		public PolyProxy(List<Point2> pts, List<Point3> pts3)
		{
			this.Create(pts, pts3);
		}
		public PolyProxy(Polygon bot, Polygon top)
		{
			List<Point2> pts2 = new List<Point2>();
			List<Point3> pts3 = new List<Point3>();
			pts2.AddRange(bot.points);
			pts2.AddRange(top.points);
			pts3.AddRange(bot.points3);
			pts3.AddRange(top.points3);
			this.Create(pts2, pts3);
		}
		protected void Create(Point2[] pts, Point3[] pts3)
		{
			this.points.AddRange(pts);
			this.points3.AddRange(pts3);
			this.AssignIndex();
		}
		protected void Create(List<Point2> pts, List<Point3> pts3)
		{
			this.points = pts;
			this.points3 = pts3;
			this.AssignIndex();
		}
		public bool MakeBox()
		{
			int n = this.points.Count;
			if (n != 8) return false;
			this.Make();
			this.FindVanishingPoints();
			this.type = ProxyType.BOX;
			this.Compute3DInfo();
			this.EstablishOpposite();
			return true;
		}
		public bool MakePlanarQuad()
		{
			int n = this.points.Count;
			if (n != 4) return false;
			this.polygons.Clear();
			Polygon poly = new Polygon(this.points);
			poly.points3 = this.points3;
			this.AddPolygon(poly);
			this.type = ProxyType.QUAD;
			this.Compute3DInfo();
			this.FindVanishingPoints();
			return true;
		}
		public void Make()
		{
			int n = this.points.Count;
			int k = n / 2;
			this.polygons.Clear();
			Point2[] pts2 = new Point2[k];
			for (int i = 0; i < k; ++i) {
				pts2[i] = this.points[i];
			}
			Point3[] pts3 = new Point3[k];
			for (int i = 0; i < k; ++i){
				pts3[i] = this.points3[i];
			}
			Polygon bot = new Polygon(pts2);
			bot.points3.AddRange(pts3);

			for (int i = 0; i < k; ++i) {
				pts2[i] = this.points[k+i];
			}
			for (int i = 0; i < k; ++i) {
				pts3[i] = this.points3[k+i];
			}
			Polygon top = new Polygon(pts2);
			top.points3.AddRange(pts3);
			this.AddPolygon(bot);
			this.AddPolygon(top);

			for (int i = 0; i < k; ++i)
			{
				int j = i, l = (i + 1) % k;
				if (i > 1) { j = (i + 1) % k; l = i; }
				Point2[] points2 = new Point2[4] {
			        bot.points[j],bot.points[l], top.points[l],top.points[j],
			    };
				Point3[] points3 = new Point3[4] {
			        bot.points3[j],bot.points3[l], top.points3[l],top.points3[j],
			    };
				List<Point2> tmplist = new List<Point2>();
				tmplist.AddRange(points2);
				Polygon plg = new Polygon(tmplist);
				plg.points3.AddRange(points3);
				this.AddPolygon(plg);
			}
		}
		public void AssignIndex()
		{
			int index = 0;
			foreach (Point2 p in this.points)
			{
				p.index = index++;
			}
		}
		public void ComputeVolume()
		{
			if (this.polygons.Count < 2)  
				this.volume = 0;
			Polygon bot = this.polygons[0];
			Polygon top = this.polygons[1];
			double h = (top.points3[0].pos - bot.points3[0].pos).Length();
			int n = bot.points3.Count;
			Vector3d o = bot.points3[0].pos;
			double sum = 0;
			for (int i = 1; i < n - 1; ++i) {
				Vector3d u = bot.points3[i].pos, v = bot.points3[i+1].pos;
				sum += ((u-o).Cross(v-o)).Length() / 2.0;
			}
			this.volume = sum*h;
		}
		public void ComputeCenter()
		{
			Vector3d c = new Vector3d();
			foreach (Point3 pt in this.points3)
			{
				c += pt.pos;
			}
			if (this.center == null)
				this.center = new Point3(c / this.points3.Count);
			else
				this.center.pos = c / this.points3.Count;
		}
		public void AddPolygon(Polygon plg)
		{
			this.polygons.Add(plg);
			plg.hostproxy = this;
		}
		public void Compute3DInfo()
		{
			this.ComputeCenter();
			Vector2d c2 = new Vector2d();
			foreach (Point2 pt in this.points)
			{
				c2 += pt.pos;
			}
			if (this.center2 != null)
				this.center2.pos = c2 / this.points.Count;
			else
				this.center2 = new Point2(c2 / this.points.Count);
			foreach (Polygon poly in this.polygons)
			{
				poly.ComputeCenter3();
				poly.ComputeNormal();
				if (poly.normal.Dot(poly.center.pos - this.center.pos) < 0)
					poly.normal = new Vector3d() - poly.normal;
			}
			if (this.polygons.Count == 6) // a box
			{
				Vector3d x = (this.polygons[2].center.pos - this.polygons[4].center.pos);
				Vector3d y = (this.polygons[3].center.pos - this.polygons[5].center.pos);
				Vector3d z = (this.polygons[1].center.pos - this.polygons[0].center.pos);
				this.xlen = x.Length();
				this.ylen = y.Length();
				this.zlen = z.Length();
				x = x.Normalize();
				z = z.Normalize();
				y = y.Normalize();// z.Cross(x).Normalize(); // make sure it is right-hand convention

				this.frame = new CoordinateFrame(this.center.pos, x, y, z);
			}

			this.ComputePointNormals();
		}
		private void ComputePointNormals()
		{
			foreach (Point3 p3 in this.points3)
			{
				Vector3d normal = new Vector3d();
				foreach (Polygon plg in this.polygons)
				{
					if (plg.points3.Contains(p3))
					{
						normal += plg.normal;
					}
				}
				p3.normal = normal.Normalize();
			}
		}
		private void FindVanishingPoints()
		{
			this.vps = new Vector2d[3];
			if (this.points.Count == 4) // planar
			{
				Vector3d x1 = new Vector3d(this.points[1].pos, 1).Cross(new Vector3d(this.points[0].pos, 1));
				Vector3d x2 = new Vector3d(this.points[3].pos, 1).Cross(new Vector3d(this.points[2].pos, 1));
				this.vps[0] = x1.Cross(x2).HomogenousNormalize().ToVector2d();

				Vector3d y1 = new Vector3d(this.points[0].pos, 1).Cross(new Vector3d(this.points[3].pos, 1));
				Vector3d y2 = new Vector3d(this.points[1].pos, 1).Cross(new Vector3d(this.points[2].pos, 1));
				this.vps[1] = y1.Cross(y2).HomogenousNormalize().ToVector2d();

				this.vps[2] = new Vector2d(200, 1e6);
			}

			if (this.points.Count == 8)
			{
				Vector3d x1 = new Vector3d(this.points[1].pos, 1).Cross(new Vector3d(this.points[0].pos, 1));
				Vector3d x2 = new Vector3d(this.points[5].pos, 1).Cross(new Vector3d(this.points[4].pos, 1));
				this.vps[0] = x1.Cross(x2).HomogenousNormalize().ToVector2d();

				Vector3d y1 = new Vector3d(this.points[2].pos, 1).Cross(new Vector3d(this.points[1].pos, 1));
				Vector3d y2 = new Vector3d(this.points[6].pos, 1).Cross(new Vector3d(this.points[5].pos, 1));
				this.vps[1] = y1.Cross(y2).HomogenousNormalize().ToVector2d();

				Vector3d z1 = new Vector3d(this.points[0].pos, 1).Cross(new Vector3d(this.points[4].pos, 1));
				Vector3d z2 = new Vector3d(this.points[1].pos, 1).Cross(new Vector3d(this.points[5].pos, 1));
				this.vps[2] = z1.Cross(z2).HomogenousNormalize().ToVector2d();
			}
		}
		private void EstablishOpposite()
		{
			if (this.type != ProxyType.BOX)
			{
				return;
			}
			this.polygons[0].opposite = this.polygons[1];
			this.polygons[1].opposite = this.polygons[0];
			this.polygons[2].opposite = this.polygons[3];
			this.polygons[4].opposite = this.polygons[2];
			this.polygons[3].opposite = this.polygons[5];
			this.polygons[5].opposite = this.polygons[3];
		}
		public object Clone_old()
		{
			PolyProxy another = new PolyProxy();
			Polygon bot = this.polygons[0].Clone() as Polygon;
			if (this.polygons.Count == 1)
			{
				another.AddPolygon(bot);
				another.points = bot.points;
				another.points3 = bot.points3;
			}
			else
			{
				Polygon top = this.polygons[1].Clone() as Polygon;
				another = MakePolyProxy(bot, top);
			}
			int j = 0;
			foreach (Polygon plg in another.polygons)
			{
				plg.textid = this.polygons[j++].textid;
			}
			another.prevattachindex = another.attachindex
					= this.attachindex;
			another.center = new Point3(this.center.pos);
			another.txtquads = new List<Polygon>();
			another.index = this.index;
			another.type = this.type;
			foreach (Polygon plg in another.polygons)
			{
				plg.CreatePropagatingPoints();
			}
			if (this.txtquads != null)
			foreach (Polygon plg in this.txtquads)
			{
				int index = this.polygons.IndexOf(plg);
				Polygon plg2 = another.polygons[index];
				plg2.textureindex = plg.textureindex;
				another.txtquads.Add(plg2);
			}
			another.Compute3DInfo();
			return another;
		}
		public object Clone()
		{
			PolyProxy another = new PolyProxy();
			List<Point2> pts2 = new List<Point2>();
			List<Point3> pts3 = new List<Point3>();
			foreach (Point2 pt in this.points)
			{
				pts2.Add(pt.Clone() as Point2);
			}
			foreach (Point3 pt in this.points3)
			{
				pts3.Add(pt.Clone() as Point3);
			}
			another.points = pts2;
			another.points3 = pts3;

			List<Polygon> plgs = new List<Polygon>();
			foreach (Polygon plg in this.polygons)
			{
				plgs.Add(plg.Clone() as Polygon);
			}

			// polygons share points
			int j = 0;
			foreach (Polygon plg in this.polygons)
			{
				Polygon ply = plgs[j];
				for (int i = 0; i < plg.points.Count; ++i)
				{
					int index = plg.points[i].index;
					ply.points[i] = pts2[index];
					ply.points3[i] = pts3[index];
				}
				++j;
			}
			another.polygons = plgs;
			foreach (Polygon ply in plgs)
				ply.hostproxy = another;

			// 0--
			another.prevattachindex = another.attachindex
					= this.attachindex;
			another.center = new Point3(this.center.pos);
			another.index = this.index;
			another.type = this.type;
			another.txtquads = new List<Polygon>();
			if (this.txtquads != null) // copy texture quads
				foreach (Polygon plg in this.txtquads)
				{
					int index = this.polygons.IndexOf(plg);
					Polygon plg2 = another.polygons[index];
					plg2.textureindex = plg.textureindex;
					another.txtquads.Add(plg2);
				}
			another.Compute3DInfo();
			return another;
		}
		public static PolyProxy MakePolyProxy(Polygon bot, Polygon top)
		{
			if (top.points.Count != bot.points.Count) return null;
			int n = bot.points.Count;
			List<Point2> pts2 = new List<Point2>();
			pts2.AddRange(bot.points);
			pts2.AddRange(top.points);
			List<Point3> pts3 = new List<Point3>();
			pts3.AddRange(bot.points3);
			pts3.AddRange(top.points3);
			int index = 0;
			foreach (Point2 pt in pts2)
			{
				pt.index = index++;
			}
			PolyProxy proxy = new PolyProxy(pts2, pts3);
			proxy.AddPolygon(bot);
			proxy.AddPolygon(top);
			for (int i = 0; i < n; ++i)
			{
				Point2[] points2 = new Point2[4] {
					bot.points[i],bot.points[(i+1)%n], top.points[(i+1)%n],top.points[i],
				};
				Point3[] points3 = new Point3[4] {
					bot.points3[i],bot.points3[(i+1)%n], top.points3[(i+1)%n],top.points3[i],
				};
				List<Point2> tmplist = new List<Point2>();
				tmplist.AddRange(points2);
				Polygon plg = new Polygon(tmplist);
				plg.points3.AddRange(points3);
				proxy.AddPolygon(plg);
			}
			proxy.Compute3DInfo();
			return proxy;
		}
		public static void OptimizeBox(PolyProxy box, Vector3d c, double xlen, double ylen, double zlen)
		{
			Vector3d xdir = box.polygons[2].center.pos - box.polygons[4].center.pos;
			Vector3d ydir = box.polygons[3].center.pos - box.polygons[5].center.pos;
			Vector3d zdir = box.polygons[1].center.pos - box.polygons[0].center.pos;
			int sign = box.polygons[2].normal.Dot(box.polygons[4].normal) < 0 ? -1 : 1;
			Vector3d xaxis = (box.polygons[2].normal + box.polygons[4].normal * sign).Normalize();
			if (xaxis.Dot(xdir) < 0) xaxis = new Vector3d() - xaxis;
			sign = box.polygons[3].normal.Dot(box.polygons[5].normal) < 0 ? -1 : 1;
			Vector3d yaxis = (box.polygons[3].normal + box.polygons[5].normal * sign).Normalize();
			if (yaxis.Dot(ydir) < 0) yaxis = new Vector3d() - yaxis;
			sign = box.polygons[1].normal.Dot(box.polygons[0].normal) < 0 ? -1 : 1;
			Vector3d zaxis = (box.polygons[1].normal + box.polygons[0].normal * sign).Normalize();
			if (zaxis.Dot(zdir) < 0) zaxis = new Vector3d() - zaxis;
			xlen /= 2;
			ylen /= 2;
			zlen /= 2;
			box.points3[0].pos = c + new Vector3d(xlen, -ylen, -zlen);
			box.points3[1].pos = c + new Vector3d(xlen, ylen, -zlen);
			box.points3[2].pos = c + new Vector3d(-xlen, ylen, -zlen);
			box.points3[3].pos = c + new Vector3d(-xlen, -ylen, -zlen);
			box.points3[4].pos = c + new Vector3d(xlen, -ylen, zlen);
			box.points3[5].pos = c + new Vector3d(xlen, ylen, zlen);
			box.points3[6].pos = c + new Vector3d(-xlen, ylen, zlen);
			box.points3[7].pos = c + new Vector3d(-xlen, -ylen, zlen);
		}

		#region IComparable<PolyProxy> Members
		public int CompareTo(PolyProxy other)
		{
			return Math.Sign(other.depth - this.depth);
		}
		#endregion

	}
	public class RepContainer : PolyProxy
	{
		public List<PolyProxy> inobjects = null; // objects that in the box, for repetition editing
		private int currcopylevel = 0;
		private Vector3d[] axes = null;
		private double[] baseboxlengths = null;
		private List<List<PolyProxy>>[] layers;
		public RepContainer(List<Point2> pts2, List<Point3> pts3)
		{
			this.Create(pts2, pts3);
		}
		public RepContainer(Point2[] pts2, Point3[] pts3)
		{
			this.Create(pts2, pts3);
		}
		public void Analysis()
		{
			// get x y z layers
			this.axes = new Vector3d[3];
			this.axes[0] = this.frame.x;
			this.axes[1] = this.frame.y;
			this.axes[2] = this.frame.z;
			this.layers = new List<List<PolyProxy>>[3];
			for (int i = 0; i < 3; ++i)
			{
				this.layers[i] = new List<List<PolyProxy>>();
			}
			Vector3d[] centers = new Vector3d[this.inobjects.Count];
			for (int i = 0; i < centers.Length; ++i)
			{
				centers[i] = this.frame.GetPointLocalCoord(this.inobjects[i].center.pos);
			}
			// find the bottom most box
			double minx = double.MaxValue, miny = double.MaxValue, minz = double.MaxValue;
			PolyProxy basebox = null; int index = 0; Vector3d basecenter = new Vector3d();
			foreach (PolyProxy box in this.inobjects)
			{
				Vector3d c = centers[index];
				if (minx >= c.x) minx = c.x; else continue;
				if (miny >= c.y) miny = c.y; else continue;
				if (minz >= c.z) minz = c.z; else continue;
				basebox = box;
				basecenter = c;
				index++;
			}
			// get the layers
			double[] lens = new double[3] { basebox.xlen, basebox.ylen, basebox.zlen };
			for (int i = 0; i < 3; ++i)
			{
				int step = 0;
				while (true)
				{
					double c = basecenter[i] + (step++ * lens[i]);
					List<PolyProxy> layer = new List<PolyProxy>();
					int id = 0; bool added = false;
					foreach (Vector3d v in centers)
					{
						if (Math.Abs(v[i] - c) < lens[i]/2)
						{
							layer.Add(this.inobjects[id]);
							added = true;
						}
						id++;
					}
					if (!added)
						break;
					else
						this.layers[i].Add(layer);
				}
			}
			this.baseboxlengths = lens;
		}
		public void Move(Vector3d start, Vector3d end, bool positive)
		{
			Vector3d t = end - start;
			int i = this.GetResizingDirection(t);
			if (i == -1) return;
			double len = this.baseboxlengths[i];
			double r = t.Length() / len;
			int l = (int)r;
			if (positive)
			{
				if (l > currcopylevel)
				{
					this.AddLayer(i, t);
					currcopylevel = l;
				}
				else if (l < currcopylevel)
				{
					this.RemoveLayer(i);
					currcopylevel = l;
				}
			}
			else
			{
				if (l > currcopylevel)
				{
					this.RemoveLayer(i);
					currcopylevel = l;
				}
				else if (l < currcopylevel)
				{
					this.AddLayer(i,t);
					currcopylevel = l;
				}
			}
		}
		public List<PolyProxy> GetBoxes()
		{
			List<PolyProxy> boxes = new List<PolyProxy>();
			foreach (List<List<PolyProxy>> lyers in this.layers)
			{
				foreach (List<PolyProxy> layer in lyers)
				{
					foreach (PolyProxy box in layer)
					{
						if (!boxes.Contains(box))
						{
							boxes.Add(box);
						}
					}
				}
			}
			return boxes;
		}
		private void AddLayer(int i, Vector3d dir)
		{
			int n = this.layers[i].Count;
			if (n < 1) return;
			List<PolyProxy> newlayer = new List<PolyProxy>();
			foreach (PolyProxy px in this.layers[i][n - 1])
			{
				PolyProxy newbox = ImageManipulator.CopyBox(px, dir, 1);
				newlayer.Add(newbox);
				foreach (List<List<PolyProxy>> lyers in this.layers)
				{
					if (lyers != this.layers[i])
					{
						foreach (List<PolyProxy> layer in lyers)
						{
							if (layer.Contains(px))
							{
								layer.Add(newbox);
							}
						}
					}
				}
			}
			this.layers[i].Add(newlayer);
		}
		private void RemoveLayer(int i)
		{
			int n = this.layers[i].Count;
			if (n < 2) return;
			foreach (PolyProxy px in this.layers[i][n - 1])
			{
				foreach (List<List<PolyProxy>> lyers in this.layers)
				{
					if (lyers != this.layers[i])
					{
						foreach (List<PolyProxy> layer in lyers)
						{
							if (layer.Contains(px))
							{
								layer.Remove(px);
							}
						}
					}
				}
			}
			this.layers[i].RemoveAt(n - 1);
		}
		private int GetResizingDirection(Vector3d dir)
		{
			int index = 0;
			foreach (Vector3d v in axes)
			{
				if (Math.Abs(v.Dot(dir.Normalize())) > 0.8)
					return index;
				index++;
			}
			return -1;
		}
	}
	
	public unsafe class CameraCalibrator
	{
		public CameraCalibrator()
		{
		}
		public double whomogenous = 1.0;
		public Matrix3d camera_K = null;
		public Matrix3d camera_R = null;
		public Vector3d camera_t;
		public Matrix3d KR = null;
		public Vector3d Kt;
		public Vector3d origin;
		public double xprojectionscale, yprojectionscale, zprojectionscale;
		public Vector3d xvanishingpoint, yvanishingpoint, zvanishingpoint;
		public Matrix3d homographymatrix = null; // to ground
		public Vector2d principalpoint;

		// ------------------------------------------------------------------------------
		// calibrate matrix with 3, 2 or 1 vanishing points
		// ------------------------------------------------------------------------------
		public bool Calibrate(Vector2d vx, Vector2d vy, Vector2d vz, double w, double h)
		{
			//vx = new Vector2d(0, 0);
			//vy = new Vector2d(4, 1);
			//vz = new Vector2d(1, 4);
			double threshold = 50*Math.Max(w,h);

			// Store the vanishing points
			this.xvanishingpoint = new Vector3d(vx, 1);
			this.yvanishingpoint = new Vector3d(vy, 1);
			this.zvanishingpoint = new Vector3d(vz, 1);

			// Check if Vanishing Points lie at infinity
			bool[] infchk = new bool[3];
			infchk[0] = (Math.Abs(vx.x) > threshold || Math.Abs(vx.y) > threshold);
			infchk[1] = (Math.Abs(vy.x) > threshold || Math.Abs(vy.y) > threshold);
			infchk[2] = (Math.Abs(vz.x) > threshold || Math.Abs(vz.y) > threshold);

			int chkcount = 0;
			for (int i = 0; i < 3; ++i) {
				if (infchk[i]) chkcount++;
			}

			Program.OutputText("chkcount = " + chkcount, true);

			double f = 0, u0 = 0, v0 = 0; // focal length, principal point
			//None
			if (chkcount == 0) {
				double Mats_11 = vy.x + vx.x;
				double Mats_12 = vx.y + vy.y;
				double Mats_13 = vy.x * vx.x + vx.y * vy.y;
				double Mats_21 = vy.x + vz.x;
				double Mats_22 = vy.y + vz.y;
				double Mats_23 = vy.x * vz.x + vy.y * vz.y;
				double Mats_31 = vz.x + vx.x;
				double Mats_32 = vz.y + vx.y;
				double Mats_33 = vz.x * vx.x + vz.y * vx.y;

				double A_11 = Mats_11 - Mats_21; double A_12 = Mats_12 - Mats_22;
				double A_21 = Mats_11 - Mats_31; double A_22 = Mats_12 - Mats_32;
				double b_1 = Mats_13 - Mats_23; double b_2 = Mats_13 - Mats_33;
				double detA = A_11 * A_22 - A_12 * A_21;
				u0 = (A_22 * b_1 - A_12 * b_2) / detA;
				v0 = (A_11 * b_2 - A_21 * b_1) / detA;

				double temp = Mats_11 * u0 + Mats_12 * v0 - Mats_13 - u0 * u0 - v0 * v0;
				if (temp < 0) {
					Program.OutputText("Calibration failed: focal length negative!!", true);
					return false;
				}
				f = Math.Sqrt(temp);

				// the geometric way
				Vector2d cp = ComputeTrianglePendicularFoot(vx, vy, vz);
				double ff = Math.Sqrt(-(vz - cp).Dot(vx - cp));
				Matrix3d K = new Matrix3d();
				K[0, 0] = K[1, 1] = ff;
				K[2, 2] = 1;
				K[0, 2] = cp.x;
				K[1, 2] = cp.y;
				this.principalpoint = cp;

				double S = (new Vector3d(vx - vz, 0).Cross(new Vector3d(vy - vz, 0))).Length();
				double sx = (new Vector3d(vy - cp, 0).Cross(new Vector3d(vz - cp, 0))).Length();
				double sy = (new Vector3d(vx - cp, 0).Cross(new Vector3d(vz - cp, 0))).Length();
				double sz = (new Vector3d(vx - cp, 0).Cross(new Vector3d(vy - cp, 0))).Length();
				double r1 = sx / S;
				double r2 = sy / S;
				double r3 = sz / S;
				Vector3d q1 = new Vector3d((vx - cp) / f, 1) * r1;
				Vector3d q2 = new Vector3d((vy - cp) / f, 1) * r2;
				Vector3d q3 = new Vector3d((vz - cp) / f, 1) * r3;
				q1 = q1.Normalize();
				q2 = q2.Normalize();
				q3 = q3.Normalize();

				Vector3d q33 = q1.Cross(q2).Normalize();
				if (q33.Dot(q3) < 0)
				    q3 = (new Vector3d() - q33).Normalize();
				else
					q3 = q33;
				Vector3d q23 = q3.Cross(q1).Normalize();
				if (q23.Dot(q2) < 0)
					q2 = (new Vector3d() - q23).Normalize();
				else
					q2 = q23;


				double e1 = q1.Dot(q2);
				double e2 = q1.Dot(q3);
				double e3 = q2.Dot(q3);
				double ee = e1 + e2 + e3;
				Program.OutputText("rotational " + ee, true);

				Matrix3d R = new Matrix3d(q1, q2, q3);

				this.camera_K = K;
				this.camera_R = R;
			} else if (chkcount == 1) {
				Vector2d v1 = vx, v2 = vy;
				if (infchk[0] == true) {
					v1 = vy;
					v2 = vz;
				} else if (infchk[1] == true) {
					v1 = vx;
					v2 = vz;
				}
				double r = ((w / 2 - v1.x) * (v2.x - v1.x) + (h / 2 - v1.y) * (v2.y - v1.y)) / (Math.Pow(v2.x - v1.x, 2) + Math.Pow(v2.y - v1.y, 2));
				u0 = v1.x + r * (v2.x - v1.x);
				v0 = v1.y + r * (v2.y - v1.y);
				double temp = u0 * (v1.x + v2.x) + v0 * (v2.y + v1.y) - (v1.x * v2.x + v2.y * v1.y + u0 * u0 + v0 * v0);
				f = Math.Sqrt(temp);
			}
			if (chkcount == 1) {
				Matrix3d K = new Matrix3d();
				K[0, 0] = K[1, 1] = f;
				K[2, 2] = 1;
				K[0, 2] = u0;
				K[1, 2] = v0;
				Matrix3d Q = K.Inverse();
				Vector3d vecx = Q * new Vector3d(vx, 1);
				vecx = vecx.Normalize();
				//	if (vx.x < u0)
				//		vecx = new Vector3d()-vecx;
				Vector3d vecz = Q * new Vector3d(vz, 1);
				vecz = vecz.Normalize();
				Vector3d vecy = vecz.Cross(vecx).Normalize();
				this.camera_R = new Matrix3d(vecx, vecy, vecz);
				this.camera_K = K;
			}

			if (chkcount == 2)
			{
				u0 = w / 2; v0 = h / 2;
				Vector3d vecx = new Vector3d();
				Vector3d vecy = new Vector3d();
				Vector3d vecz = new Vector3d();
				if (infchk[0]) {
					vecx = new Vector3d(vx, 0).Normalize();
				}
				if (infchk[1]) {
					vecy = new Vector3d(vy, 0).Normalize();
					vecy = vecy * -Math.Sign(vecy.y);
				}
				if (infchk[2]) {
					vecz = new Vector3d(vz, 0).Normalize();
				}

				if (infchk[0] && infchk[1]) {
					u0 = vz.x;
					v0 = vz.y;
					vecz = vecx.Cross(vecy);
				} else if (infchk[1] && infchk[2]) {
					u0 = vx.x;
					v0 = vx.y;
					vecx = vecy.Cross(vecz);
				} else {
					u0 = vy.x;
					v0 = vy.y;
					vecy = vecz.Cross(vecx);
				}

				this.camera_R = new Matrix3d(new Vector3d() - vecx, vecy, vecz);
				f = 500;
				Matrix3d K = new Matrix3d();
				K[0, 0] = K[1, 1] = f;
				K[2, 2] = 1;
				K[0, 2] = u0;
				K[1, 2] = v0;
				this.camera_K = K;
			}

			Program.OutputText("principal point = (" + camera_K[0, 2] + "," + camera_K[1, 2] + ")", true);
			return true;
		}
		public void ComputeFullProjectionMatrix(double cameraheight, Vector2d originimgpos)
		{
			// C = -R'*T (a*o=KT) => C = -R'*aK^-1*o, with C.z = cameraheight;
			Matrix3d RT = this.camera_R.Transpose() * this.camera_R;
			double err = Math.Abs(RT.Det() - 1);
			Program.OutputText("--- camera R accuracy = " + err, true);
		//	if (err > 1e-5) {
		//		throw new Exception();
		//	}

			Vector3d o = new Vector3d(originimgpos, 1);
			Matrix3d K_inv = this.camera_K.Inverse();
			Vector3d q = (this.camera_R.Transpose() * K_inv) * o;
			double alpha = -cameraheight / q.z;
			this.camera_t = K_inv * o * alpha;
			this.origin = o;

			this.KR = this.camera_K * this.camera_R;
			this.Kt = this.camera_K * this.camera_t;
			Vector3d KR_1 = new Vector3d(this.KR[0, 0], this.KR[1, 0], this.KR[2, 0]);
			Vector3d KR_2 = new Vector3d(this.KR[0, 1], this.KR[1, 1], this.KR[2, 1]);
			this.homographymatrix = new Matrix3d(KR_1, KR_2, Kt).Inverse();
			Matrix3d X = new Matrix3d(this.xvanishingpoint, this.yvanishingpoint, this.zvanishingpoint);
			Matrix3d I = X.Inverse() * this.KR;
			this.xprojectionscale = I[0, 0] / alpha;
			this.yprojectionscale = I[1, 1] / alpha;
			this.zprojectionscale = I[2, 2] / alpha;
			
		}
		public void ComputeFullProjectionMatrix(Vector2d originimgpos, Vector2d p, double height_p, Vector2d q, double height_q)
		{
			// given image origin position, and two points (has same x,y coordinates) with their z values in 3d, fully calibrate the camera
			// Kt = alpha*o applies, the projection matrix has only one parameter P = f(alpha);
			// solve for P*(x,y,height_p,1) = r1*(p,1); P*(x,y,height_q,1) = r2*(q,1);
			Matrix3d RT = this.camera_R.Transpose() * this.camera_R;
			double err = Math.Abs(RT.Det() - 1);
			Program.OutputText("--- camera R accuracy = " + err, true);

			// solve for r1, r2, r1*(p,1)-r2*(q,1) = KR_3;
			this.KR = this.camera_K * this.camera_R;
			double[] KR_3 = new double[3]{this.KR[0, 2], this.KR[1, 2], this.KR[2, 2]};
			MatrixNd A = new MatrixNd(new Vector3d[2] {new Vector3d(p,1), new Vector3d(q,1)}); // 3x2
			MatrixNd ATA = A.Transpose().Mul(A);
			Matrix2d ATA_inv = ATA.ToMatrix2D().Inverse();
			MatrixNd Inv = new MatrixNd(2, 2);
			Inv[0, 0] = ATA_inv.A;
			Inv[1, 0] = ATA_inv.C;
			Inv[0, 1] = ATA_inv.B;
			Inv[1, 1] = ATA_inv.D;
			double[] res = new double[2];
			Inv.Mul(A.Transpose()).Multiply(KR_3, res);
			double r1 = res[0], r2 = res[1];

			// solve for alpha
			Vector3d b = new Vector3d(q, 1) * r2;
			Vector3d KR_1 = new Vector3d(this.KR[0, 0], this.KR[1, 0], this.KR[2, 0]);
			Vector3d KR_2 = new Vector3d(this.KR[0, 1], this.KR[1, 1], this.KR[2, 1]);
			Vector3d o = new Vector3d(originimgpos, 1);
			Matrix3d M = new Matrix3d(KR_1,KR_2,o);
			Vector3d x = M.Inverse() * b;
			double alpha = x[2];

			Matrix3d K_inv = this.camera_K.Inverse();
			this.camera_t = K_inv * o * alpha;
			this.origin = o;

			// compute parameters
			this.Kt = this.camera_K * this.camera_t;
			this.homographymatrix = new Matrix3d(KR_1, KR_2, Kt).Inverse();
			Matrix3d X = new Matrix3d(this.xvanishingpoint, this.yvanishingpoint, this.zvanishingpoint);
			Matrix3d I = X.Inverse() * this.KR;
			this.xprojectionscale = I[0, 0] / alpha;
			this.yprojectionscale = I[1, 1] / alpha;
			this.zprojectionscale = I[2, 2] / alpha;

		}
		public void ComputeParameters()
		{
			this.KR = this.camera_K * this.camera_R;
			this.Kt = this.camera_K * this.camera_t;
			this.origin = new Vector3d(this.ComputePointImgPos(new Vector3d(0, 0, 0)),1);
			double alpha = this.Kt.Dot(this.origin)/this.origin.SquareLength();
			Vector3d KR_1 = new Vector3d(this.KR[0, 0], this.KR[1, 0], this.KR[2, 0]);
			Vector3d KR_2 = new Vector3d(this.KR[0, 1], this.KR[1, 1], this.KR[2, 1]);
			this.homographymatrix = new Matrix3d(KR_1, KR_2, Kt).Inverse();
			Vector2d vx = (this.KR * new Vector3d(1, 0, 0)).HomogenousNormalize().ToVector2d();
			Vector2d vy = (this.KR * new Vector3d(0, 1, 0)).HomogenousNormalize().ToVector2d();
			Vector2d vz = (this.KR * new Vector3d(0, 0, 1)).HomogenousNormalize().ToVector2d();
			this.xvanishingpoint = new Vector3d(vx,1);
			this.yvanishingpoint = new Vector3d(vy,1);
			this.zvanishingpoint = new Vector3d(vz,1);
			Matrix3d X = new Matrix3d(this.xvanishingpoint, this.yvanishingpoint, this.zvanishingpoint);
			Matrix3d I = X.Inverse() * this.KR;
			this.xprojectionscale = I[0, 0] / alpha;
			this.yprojectionscale = I[1, 1] / alpha;
			this.zprojectionscale = I[2, 2] / alpha;
		}
		public void GetGLMatrices(out double[] glprojmatrix, out double[] glmodelviewmatrix, double w, double h, double znear, double zfar)
		{
			double[,] mat = new double[3, 4];
			for (int i = 0; i < 3; ++i)
			{
				for (int j = 0; j < 3; ++j)
				{
					mat[i, j] = this.KR[i, j];
				}
			}
			mat[0, 3] = this.Kt[0];
			mat[1, 3] = this.Kt[1];
			mat[2, 3] = this.Kt[2];

			Matrix3d LHC = new Matrix3d();
			LHC[0, 0] = LHC[1, 1] = LHC[2, 2] = -1;
			LHC[0, 0] = 1;
			double[,] icpara = new double[3, 4];
			double[,] trans = new double[3, 4];
			if (arParamDecompMat(mat, icpara, trans) < 0)
			{
				throw new Exception();
			}
			Matrix3d R = new Matrix3d();
			for (int i = 0; i < 3; i++)
			{
				for (int j = 0; j < 3; j++)
				{
					R[i, j] = trans[i, j];
				}
			}
			Matrix3d LHCR = LHC * R;
			Matrix4d modelViewMatrix = new Matrix4d(LHCR);
			modelViewMatrix[0, 3] = trans[0, 3];
			modelViewMatrix[1, 3] = trans[1, 3];
			modelViewMatrix[2, 3] = trans[2, 3];
			modelViewMatrix[1, 3] = modelViewMatrix[1, 3] * (-1);
			modelViewMatrix[2, 3] = modelViewMatrix[2, 3] * (-1);
			modelViewMatrix[3, 3] = 1.0;
			glmodelviewmatrix = modelViewMatrix.Transpose().ToArray();

			Matrix4d H_inv = new Matrix4d();
			H_inv[0, 0] = 2.0 / w;
			H_inv[0, 2] = -1;
			H_inv[1, 1] = -2.0 / h;
			H_inv[1, 2] = 1.0;
			H_inv[3, 2] = 1.0;
			Matrix3d K = new Matrix3d();
			for (int i = 0; i < 3; i++)
			{
				for (int j = 0; j < 3; j++)
				{
					K[i, j] = icpara[i, j] / icpara[2, 2];
				}
			}
			Matrix3d y = K * LHC;
			Matrix4d y_ = new Matrix4d(y);
			Matrix4d result = H_inv * (y_);
			double C_ = -(zfar + znear) / (zfar - znear);
			double D_ = -(2 * zfar * znear) / (zfar - znear);
			result[2, 2] = C_;
			result[2, 3] = D_;
			glprojmatrix = result.Transpose().ToArray();
		}
		public Vector2d[] GetVPs()
		{
			return new Vector2d[3] {
				this.xvanishingpoint.ToVector2d(),
				this.yvanishingpoint.ToVector2d(),
				this.zvanishingpoint.ToVector2d()
			};
		}
		public Vector3d Compute3DCoordinate(Vector2d impos, Vector2d groundrefpos)
		{
			Vector3d vanishingline = this.xvanishingpoint.Cross(this.yvanishingpoint);
			Vector3d pointT = new Vector3d(impos, this.whomogenous);
			Vector3d pointB = new Vector3d(groundrefpos, this.whomogenous);
			Vector3d pointQ = new Vector3d(groundrefpos, this.whomogenous);
			Vector3d pointV = this.zvanishingpoint;
			Vector3d pointS = this.homographymatrix * pointQ;
			pointS.HomogenousNormalize();
			Vector3d BT = pointB.Cross(pointT);
			Vector3d VT = pointV.Cross(pointT);
			int sign = BT.Dot(VT) < 0 ? -1 : 1;
			pointS.z = -sign * this.origin.Dot(vanishingline) * Math.Abs(BT.Length()) /
							(this.zprojectionscale * pointB.Dot(vanishingline) * Math.Abs(VT.Length()));

			return pointS;
		}
		public Vector3d Compute3DCoordinate(Vector2d impos, double height)
		{
			Vector2d[] pt2 = new Vector2d[4] {
				new Vector2d(0,0),
				new Vector2d(1,0),
				new Vector2d(1,1),
				new Vector2d(0,1)
			};
			Vector2d[] impt2 = new Vector2d[4] {
				this.ComputePointImgPos(new Vector3d(pt2[0],height)),
				this.ComputePointImgPos(new Vector3d(pt2[1],height)),
				this.ComputePointImgPos(new Vector3d(pt2[2],height)),
				this.ComputePointImgPos(new Vector3d(pt2[3],height))
			};
			Matrix3d H = ComputeHomographyMatrixCV(impt2, pt2);
			Vector3d outpt3 = H * new Vector3d(impos, 1);
			outpt3.HomogenousNormalize();
			outpt3.z = height;
			return outpt3;
		}
		public Vector3d[] Compute3DCoordinate(List<Point2> impos, double height)
		{
			Vector2d[] pt2 = new Vector2d[4] {
				new Vector2d(0,0),
				new Vector2d(1,0),
				new Vector2d(1,1),
				new Vector2d(0,1)
			};
			Vector2d[] impt2 = new Vector2d[4] {
				this.ComputePointImgPos(new Vector3d(pt2[0],height)),
				this.ComputePointImgPos(new Vector3d(pt2[1],height)),
				this.ComputePointImgPos(new Vector3d(pt2[2],height)),
				this.ComputePointImgPos(new Vector3d(pt2[3],height))
			};
			Matrix3d H = ComputeHomographyMatrixCV(impt2, pt2);
			Vector3d[] outpt3 = new Vector3d[impos.Count];
			int index = 0;
			foreach (Point2 p in impos)
			{
				Vector3d point = H * new Vector3d(p.pos, 1);
				point.HomogenousNormalize();
				point.z = height;
				outpt3[index++] = point;
			}
			return outpt3;
		}
		public Vector3d ComputePointGround3dPos(Vector2d imgpos)
		{
			Vector3d point = this.homographymatrix * new Vector3d(imgpos, 1);
			point.HomogenousNormalize();
			point.z = 0;
			return point;
		}
		public Vector3d[] ComputePointGround3dPos(Point[] points)
		{
			Vector3d[] result = new Vector3d[points.Length];
			for (int i = 0; i < points.Length; ++i)
			{
				Point p = points[i];
				result[i] = this.ComputePointGround3dPos(new Vector2d(p.X, p.Y));
			}
			return result;
		}

		public Vector2d ComputePointImgPos(Vector3d p)
		{
			// the camera projection matrix
			Vector3d q = this.KR * p + this.Kt;
			q.HomogenousNormalize();
			return q.ToVector2d();
		}
		public Vector3d GetCameraPos()
		{
			return new Vector3d()-this.camera_R.Inverse() * this.camera_t;
		}

		// IO
		public void Save(StreamWriter sw)
		{
			for (int i = 0; i < 3; ++i)
			{
				for (int j = 0; j < 3; ++j)
				{
					sw.Write(this.camera_K[i,j] + " ");
				}
			}
			sw.Write("\n");
			for (int i = 0; i < 3; ++i)
			{
				for (int j = 0; j < 3; ++j)
				{
					sw.Write(this.camera_R[i, j] + " ");
				}
			}
			sw.Write("\n");
			sw.WriteLine(this.camera_t.x + " " + this.camera_t.y + " " + this.camera_t.z);
		}
		public void Read(StreamReader sr)
		{
			this.camera_K = new Matrix3d();
			this.camera_R = new Matrix3d();
			this.camera_t = new Vector3d();
			char[] delimiters = { ' ', '\t' };
			string s = "";
			while (sr.Peek() > -1)
			{
				s = sr.ReadLine();
				string[] tokens = s.Split(delimiters);
				for (int i = 0; i < 3; ++i)
				{
					for (int j = 0; j < 3; ++j)
					{
						this.camera_K[i, j] = double.Parse(tokens[i*3+j]);
					}
				}
				s = sr.ReadLine();
				tokens = s.Split(delimiters);
				for (int i = 0; i < 3; ++i)
				{
					for (int j = 0; j < 3; ++j)
					{
						this.camera_R[i, j] = double.Parse(tokens[i * 3 + j]);
					}
				}
				s = sr.ReadLine();
				tokens = s.Split(delimiters);
				for (int i = 0; i < 3; ++i)
				{
					this.camera_t[i] = double.Parse(tokens[i]);
				}
			}
		}
		public void PackCameraData(BinaryWriter bw)
		{
			List<byte> allbytes = new List<byte>();
			for (int i = 0; i < 3; ++i)
			{
				for (int j = 0; j < 3; ++j)
				{
					allbytes.AddRange(BitConverter.GetBytes(this.camera_K[i, j]));
				}
			}
			for (int i = 0; i < 3; ++i)
			{
				for (int j = 0; j < 3; ++j)
				{
					allbytes.AddRange(BitConverter.GetBytes(this.camera_R[i, j]));
				}
			}
			allbytes.AddRange(BitConverter.GetBytes(this.camera_t.x));
			allbytes.AddRange(BitConverter.GetBytes(this.camera_t.y));
			allbytes.AddRange(BitConverter.GetBytes(this.camera_t.z));

			bw.Write(BitConverter.GetBytes((Int32)allbytes.Count));
			bw.Write(allbytes.ToArray());
			Program.OutputText("Packing camera data succeeded ...", true);
		}
		public void UnpackCameraData(BinaryReader br)
		{
			int count = br.ReadInt32();

			this.camera_K = new Matrix3d();
			this.camera_R = new Matrix3d();
			this.camera_t = new Vector3d();
			for (int i = 0; i < 3; ++i)
			{
				for (int j = 0; j < 3; ++j)
				{
					this.camera_K[i, j] = br.ReadDouble();
				}
			}
			for (int i = 0; i < 3; ++i)
			{
				for (int j = 0; j < 3; ++j)
				{
					this.camera_R[i, j] = br.ReadDouble();
				}
			}
			for (int i = 0; i < 3; ++i)
			{
				this.camera_t[i] = br.ReadDouble();
			}
			Program.OutputText("Unpacking camera data succeeded ...", true);
		}

		// ------------------------------------------------------------------------------
		// QR decomposition for camera matrix
		// ------------------------------------------------------------------------------
		public static void argConvGLcpara2(double[,] cparam, int width, int height, double gnear, double gfar, double[] m)
		{
			double[,] icpara = new double[3, 4];
			double[,] trans = new double[3, 4];
			double[,] p = new double[3, 3];
			double[,] q = new double[4, 4];
			int i, j;

			/* Usually we do not need to take care of this */
			/* If the specified cparam includes translation and rotation
			  components, this function divides it into perspective projection
			  component and translation/rotation components. */
			/* Usually cparam does not include translation/rotation components. */
			if (arParamDecompMat(cparam, icpara, trans) < 0)
			{
				throw new Exception();
			}

			/* Camera parameters are converted openGL representation. */
			/* Camera parameter represents the transformation from camera coordinates
			   to screen coordinates[pixel]. OpenGL projection matrix represents the
			   transformation from the camera coordinates to normalized view volume. */
			for (i = 0; i < 3; i++)
			{
				for (j = 0; j < 3; j++)
				{
					p[i, j] = icpara[i, j] / icpara[2, 2];
				}
			}
			q[0, 0] = (2.0 * p[0, 0] / width);
			q[0, 1] = (2.0 * p[0, 1] / width);
			q[0, 2] = ((2.0 * p[0, 2] / width) - 1.0);
			q[0, 3] = 0.0;

			q[1, 0] = 0.0;
			q[1, 1] = (2.0 * p[1, 1] / height);
			q[1, 2] = ((2.0 * p[1, 2] / height) - 1.0);
			q[1, 3] = 0.0;

			q[2, 0] = 0.0;
			q[2, 1] = 0.0;
			q[2, 2] = (gfar + gnear) / (gfar - gnear);
			q[2, 3] = -2.0 * gfar * gnear / (gfar - gnear);

			q[3, 0] = 0.0;
			q[3, 1] = 0.0;
			q[3, 2] = 1.0;
			q[3, 3] = 0.0;

			/* This multiplies translation/rotation component to above calculated
			   matrix. Usually we do not need to take care of this. */
			for (i = 0; i < 4; i++)
			{
				for (j = 0; j < 3; j++)
				{
					m[i + j * 4] = q[i, 0] * trans[0, j]
							 + q[i, 1] * trans[1, j]
							 + q[i, 2] * trans[2, j];
				}
				m[i + 3 * 4] = q[i, 0] * trans[0, 3]
						 + q[i, 1] * trans[1, 3]
						 + q[i, 2] * trans[2, 3]
						 + q[i, 3];
			}
		}
		public static int arParamDecompMat(double[,] source, double[,] cpara, double[,] trans)
		{
			int r, c;
			double[,] Cpara = new double[3, 4];
			double rem1, rem2, rem3;

			if (source[2, 3] >= 0)
			{
				for (r = 0; r < 3; r++)
				{
					for (c = 0; c < 4; c++)
					{
						Cpara[r, c] = source[r, c];
					}
				}
			}
			else
			{
				for (r = 0; r < 3; r++)
				{
					for (c = 0; c < 4; c++)
					{
						Cpara[r, c] = -(source[r, c]);
					}
				}
			}

			for (r = 0; r < 3; r++)
			{
				for (c = 0; c < 4; c++)
				{
					cpara[r, c] = 0.0;
				}
			}
			cpara[2, 2] = norm(Cpara[2, 0], Cpara[2, 1], Cpara[2, 2]);
			trans[2, 0] = Cpara[2, 0] / cpara[2, 2];
			trans[2, 1] = Cpara[2, 1] / cpara[2, 2];
			trans[2, 2] = Cpara[2, 2] / cpara[2, 2];
			trans[2, 3] = Cpara[2, 3] / cpara[2, 2];

			cpara[1, 2] = dot(trans[2, 0], trans[2, 1], trans[2, 2],
							   Cpara[1, 0], Cpara[1, 1], Cpara[1, 2]);
			rem1 = Cpara[1, 0] - cpara[1, 2] * trans[2, 0];
			rem2 = Cpara[1, 1] - cpara[1, 2] * trans[2, 1];
			rem3 = Cpara[1, 2] - cpara[1, 2] * trans[2, 2];
			cpara[1, 1] = norm(rem1, rem2, rem3);
			trans[1, 0] = rem1 / cpara[1, 1];
			trans[1, 1] = rem2 / cpara[1, 1];
			trans[1, 2] = rem3 / cpara[1, 1];

			cpara[0, 2] = dot(trans[2, 0], trans[2, 1], trans[2, 2],
							   Cpara[0, 0], Cpara[0, 1], Cpara[0, 2]);
			cpara[0, 1] = dot(trans[1, 0], trans[1, 1], trans[1, 2],
							   Cpara[0, 0], Cpara[0, 1], Cpara[0, 2]);
			rem1 = Cpara[0, 0] - cpara[0, 1] * trans[1, 0] - cpara[0, 2] * trans[2, 0];
			rem2 = Cpara[0, 1] - cpara[0, 1] * trans[1, 1] - cpara[0, 2] * trans[2, 1];
			rem3 = Cpara[0, 2] - cpara[0, 1] * trans[1, 2] - cpara[0, 2] * trans[2, 2];
			cpara[0, 0] = norm(rem1, rem2, rem3);
			trans[0, 0] = rem1 / cpara[0, 0];
			trans[0, 1] = rem2 / cpara[0, 0];
			trans[0, 2] = rem3 / cpara[0, 0];

			trans[1, 3] = (Cpara[1, 3] - cpara[1, 2] * trans[2, 3]) / cpara[1, 1];
			trans[0, 3] = (Cpara[0, 3] - cpara[0, 1] * trans[1, 3]
									   - cpara[0, 2] * trans[2, 3]) / cpara[0, 0];

			for (r = 0; r < 3; r++)
			{
				for (c = 0; c < 3; c++)
				{
					cpara[r, c] /= cpara[2, 2];
				}
			}

			return 0;
		}
		public static double norm(double a, double b, double c)
		{
			return (Math.Sqrt(a * a + b * b + c * c));
		}
		public static double dot(double a1, double a2, double a3, double b1, double b2, double b3)
		{
			return (a1 * b1 + a2 * b2 + a3 * b3);
		}
		public static Matrix3d ComputeHomographyMatrixCV(Vector2d[] imgpts, Vector2d[] refpts)
		{
			int n = imgpts.Length;

			PointF[] src = new PointF[n];
			PointF[] tar = new PointF[n];
			for (int i = 0; i < n; ++i)
			{
				src[i] = new PointF((float)imgpts[i].x, (float)imgpts[i].y);
				tar[i] = new PointF((float)refpts[i].x, (float)refpts[i].y);
			}

			Matrix<double> mat = CameraCalibration.FindHomography(src, tar, Emgu.CV.CvEnum.HOMOGRAPHY_METHOD.RANSAC, 1e-1);
			Matrix3d H = new Matrix3d();
			for (int i = 0; i < 3; ++i)
			{
				for (int j = 0; j < 3; ++j)
				{
					H[i, j] = mat[i, j];
				}
			}

			return H;
		}
		public static double[,] ComputeProjectionMatrixP(Vector2d[] imgpts, Vector3d[] spacepts)
		{
			// this function computes the projection matrix given image-to-space points correspondence
			// using DLT algorithm for approximation
			// require: cv eigendecompose, num of points >= 6
			int n = imgpts.Length;
			double[,] mat = new double[n*2, 12];
			for (int i = 0, j = 0; i < n; ++i,j+=2)
			{
				double x = imgpts[i].x, y = imgpts[i].y;
				int jj = j + 1;
				mat[j,4] = spacepts[i].x;
				mat[j,5] = spacepts[i].y;
				mat[j,6] = spacepts[i].z;
				mat[j,7] = 1.0;
				mat[j,8] = -y*spacepts[i].x;
				mat[j,9] = -y*spacepts[i].y;
				mat[j,10] = -y*spacepts[i].z;
				mat[j,11] = -y;
				mat[jj,0] = spacepts[i].x;
				mat[jj,1] = spacepts[i].y;
				mat[jj,2] = spacepts[i].z;
				mat[jj,3] = 1.0;
				mat[jj,8] = -x*spacepts[i].x;
				mat[jj,9] = -x*spacepts[i].y;
				mat[jj,10] = -x*spacepts[i].z;
				mat[jj,11] = -x;
			}
			// perform eigen decomposition
			// if n > 6, decompose ATA, else directly decompose A
			Matrix<double> eigvec = new Matrix<double>(12, 12);
			Matrix<double> eigval = new Matrix<double>(12, 1);
			Matrix<double> cvmat = new Matrix<double>(mat);
			double[,] p = new double[3,4];
		//	if (n > 6)
			{
				cvmat = cvmat.Transpose().Mul(cvmat);
			}
			CvInvoke.cvEigenVV(cvmat.Ptr, eigvec.Ptr, eigval.Ptr, 1e-30, 0, 0);
			for (int i = 0; i < 3; ++i)
			{
				for (int j = 0; j < 4; ++j)
				{
					p[i, j] = eigvec[11, i * 4 + j];
				}
			}
			// ||p|| = 1 from cv
			return p; 
		}
		public static Vector2d ComputeTriangleV3(Vector2d H, Vector2d v1, Vector2d v2)
		{
			// this function compute the last point of an triangle, given two points
			// and H, the pendicular foot. The algorithm uses the orthogonalty to
			// solve for the unknow X(x,y) by two linear euqations.
			// (X-H).(V2-V1) = 0 && (X-V1).(V2-H) = 0;
			Vector2d v2H = v2 - H;
			Vector2d v21 = v2 - v1;
			Matrix2d A = new Matrix2d(v2H,v21).Transpose();
			Vector2d b = new Vector2d(v2H.Dot(v1), v21.Dot(H));
			Vector2d x = A.Inverse() * b;
			return x;
		}
		public static Vector2d ComputeTrianglePendicularFoot(Vector2d v1, Vector2d v2, Vector2d v3)
		{
			// this function compute the pendicular foot of an triangle, given three points
			// v1, v2 and v3. The algorithm uses the orthogonalty to
			// solve for the unknow H(x,y) by two linear euqations.
			// (H-V1).(V3-V2) = 0 && (H-V3).(V2-V1) = 0;
			Vector2d v21 = v2 - v1;
			Vector2d v32 = v3 - v2;
			Matrix2d A = new Matrix2d(v21, v32).Transpose();
			Vector2d b = new Vector2d(v21.Dot(v3), v32.Dot(v1));
			Vector2d H = A.Inverse() * b;
			return H;
		}
	}
	public class GeomCamCalibrator
	{
		// using geometry primitives for camera calibration
		public List<Point2> points = new List<Point2>();
		public List<LineSegment> linesegments = new List<LineSegment>();
		public GeomCamCalibrator()
		{
		}
		public void AddPoint(Point2 p)
		{
			if (this.points.Count >= 6) return; // a maximum is 6
			this.points.Add(p);
		}
		public void AddLineSegment(Point2 p, Point2 q)
		{
			LineSegment line = new LineSegment(p, q);
			this.linesegments.Add(line);
			if (this.points.Count == 6)
				this.Finish();
		}
		public void Finish()
		{
			if (this.points.Count != 6) return;
			this.linesegments.Add(new LineSegment( //3
				this.points[0],this.points[2]
				));
			this.linesegments.Add(new LineSegment( //4
				this.points[2], this.points[4]
				));
			this.linesegments.Add(new LineSegment( //5
				this.points[1], this.points[3]
				));
			this.linesegments.Add(new LineSegment( //6
				this.points[3], this.points[5]
				));
		}
		public void GetInitialValues(out Vector2d[] vps, out Vector2d origin)
		{
			Vector2d vy = ImageManipulator.FindLinesIntersectPoint(this.linesegments[3], this.linesegments[5]);
			Vector2d vx = ImageManipulator.FindLinesIntersectPoint(this.linesegments[4], this.linesegments[6]);
			Vector2d vz = ImageManipulator.FindLinesIntersectPoint(this.linesegments[0], this.linesegments[1]);
			Vector2d vz2 = ImageManipulator.FindLinesIntersectPoint(this.linesegments[1], this.linesegments[2]);
			Vector2d vz3 = ImageManipulator.FindLinesIntersectPoint(this.linesegments[0], this.linesegments[2]);
			vz = (vz + vz2 + vz3) / 3;
			if (vx.x < vy.x)
			{
				Vector2d tmp = vx;
				vx = vy;
				vy = tmp;
			}
			vps = new Vector2d[3] { vx, vy, vz };
			Vector3d line1 = new Vector3d(vx, 1).Cross(new Vector3d((this.points[1].pos+this.points[3].pos)/2, 1));
			Vector3d line2 = new Vector3d(vy, 1).Cross(new Vector3d((this.points[5].pos+this.points[3].pos)/2, 1));
			origin = line1.Cross(line2).HomogenousNormalize().ToVector2d();
		}
	}
	
	public class HexagonFinder
	{
		private class cNode
		{
			public int index = -1;
			public List<cNode> nbrs = null;
			public cNode parent = null;
			public Vector2d pos;
			public bool visited = false;
			public cNode(int index, Vector2d p)
			{
				this.index = index;
				this.pos = p;
				this.nbrs = new List<cNode>();
			}
		}
		public static Vector2d[] FindHexagon(Vector3d[] lines)
		{
			// this function finds the intersections of the six lines
			// returns the hexagon that formed by these intersecting points, if exist
			// these lines are supposing to be lines etherating from vanishing points
			// that touch the boundary of an object
			int n = lines.Length; int index = 0;
			if (n != 6) throw new ArgumentException("line number must be 6!");
			List<cNode> connectnodes = new List<cNode>();
			List<int>[] linenodes = new List<int>[n];
			List<double>[] linenodespos = new List<double>[n];
			for (int i = 0; i < n; ++i) {
				linenodes[i] = new List<int>();
				linenodespos[i] = new List<double>();
			}
			for (int i = 0; i < n; i+=2)
			{
				Vector3d u = lines[i], v = lines[i + 1];
				for (int j = i+2; j < n; ++j)
				{
					Vector3d r = lines[j];
					cNode node = new cNode(index, u.Cross(r).HomogenousNormalize().ToVector2d());
					connectnodes.Add(node);
					linenodes[i].Add(index);
					linenodespos[i].Add(node.pos.x);
					linenodes[j].Add(index);
					linenodespos[j].Add(node.pos.x);
					index++;
					node = new cNode(index, v.Cross(r).HomogenousNormalize().ToVector2d());
					connectnodes.Add(node);
					linenodes[i+1].Add(index);
					linenodespos[i+1].Add(node.pos.x);
					linenodes[j].Add(index);
					linenodespos[j].Add(node.pos.x);
					index++;
				}
			}
			// sort points on each line and build relations
			for (int i = 0; i < n; ++i)
			{
				double[] pos = linenodespos[i].ToArray();
				int[] indice = linenodes[i].ToArray();
				Array.Sort(pos, indice);
				for (int j = 0; j < indice.Length - 1; ++j)
				{
					int k = indice[j], l = indice[j+1];
					connectnodes[k].nbrs.Add(connectnodes[l]);
					connectnodes[l].nbrs.Add(connectnodes[k]);
				}
			}
			// now find the hexagon, first remove those that are connecting to only one 
			// node, then remove those that are connecting to two nodes
			List<cNode> singles = new List<cNode>();
			foreach (cNode node in connectnodes)
			{
				if (node.nbrs.Count < 3)
				{
					singles.Add(node);
				}
			}
			for (int i = 0; i < singles.Count; ++i)
			{
				cNode node = singles[i];
				for (int j = 0; j < node.nbrs.Count; ++j)
				{
					node.nbrs[j].nbrs.Remove(node);
				}
				connectnodes.Remove(node);
			}
			// neighbor 2
			if (connectnodes.Count > 6)
			{
				singles.Clear();
				foreach (cNode node in connectnodes)
				{
					if (node.nbrs.Count == 2 && node.nbrs[0].nbrs.Count > 2 &&
						node.nbrs[1].nbrs.Count > 2)
					{
						singles.Add(node);
					}
				}
				foreach (cNode node in singles)
					connectnodes.Remove(node);
			}
			// sort by angle
			int total = connectnodes.Count;
			cNode[] nodes = connectnodes.ToArray();
			Vector2d center = new Vector2d();
			foreach (cNode node in nodes)
				center += node.pos;
			center /= total;
			double[] angles = new double[total];
			Vector2d ydir = new Vector2d(0, 1);
			for (int i = 0; i < total; ++i)
			{
				Vector2d t = (nodes[i].pos - center).Normalize();
				double cos = ydir.Dot(t);
				if (cos < -1) cos = -1; else if (cos > 1) cos = 1;
				double angle = Math.Acos(cos);
				if (t.x * ydir.y - t.y * ydir.x > 0)
				{
					angle = 2 * Math.PI - angle;
				}
				angles[i] = angle;
			}

			Array.Sort(angles, nodes);

			Vector2d[] hexgonpoints = new Vector2d[total];
			for (int i = 0; i < total; ++i)
			{
				hexgonpoints[i] = nodes[i].pos;
			}
			return hexgonpoints;
		}
		public static Vector2d[] FindHexagon(Vector3d[] lines, Vector2d center)
		{
			// this function finds the intersections of the six lines
			// returns the hexagon that formed by these intersecting points, if exist
			// these lines are supposing to be lines etherating from vanishing points
			// that touch the boundary of an object
			int n = lines.Length;
			if (n != 6) throw new ArgumentException("line number must be 6!");
			List<Vector2d> intersects = new List<Vector2d>();
			for (int i = 0; i < n; i += 2)
			{
				Vector3d u = lines[i], v = lines[i + 1];
				for (int j = i + 2; j < n; ++j)
				{
					Vector3d r = lines[j];
					intersects.Add(u.Cross(r).HomogenousNormalize().ToVector2d());
					intersects.Add(v.Cross(r).HomogenousNormalize().ToVector2d());
				}
			}
			// sort points by angles
			Vector2d[] points = intersects.ToArray();
			double[] angles = new double[points.Length];
			Vector2d ydir = new Vector2d(0, 1);
			for (int i = 0; i < points.Length; ++i)
			{
				Vector2d t = (points[i] - center).Normalize();
				double cos = ydir.Dot(t);
				if (cos < -1) cos = -1; else if (cos > 1) cos = 1;
				double angle = Math.Acos(cos);
				if (t.x * ydir.y - t.y * ydir.x > 0)
				{
					angle = 2*Math.PI - angle;
				}
				angles[i] = angle;
			}
			
			Array.Sort(angles, points);

			// now find the hexagon, visit points in ccw order
			Vector2d[] hexgonpoints = new Vector2d[angles.Length];
			return hexgonpoints;
		}
	}
	public class Metaphor
	{
		public int type = 0; // 0 for translation, 1 for rotation.
		public bool visible = true;
		public Point3 center = null;
		public Vector3d axis;
		public Vector2d screendir;
		public Vector3d[] points = null;
		public Point3 start, end;
		public Metaphor()
		{
		}
		public void ReleaseMetaphorPoints()
		{
			this.start = this.end = null;
		}
		public virtual Transform GetTransformation()
		{
			return new Transform();
		}
	}
	public class RotationMetaphor : Metaphor
	{
		public double radius = 0;
		public RotationMetaphor(Vector3d c, Vector3d axis, double radius, int num)
		{
			this.type = 1;
			this.center = new Point3(c);
			this.axis = axis;
			this.radius = radius;
			this.points = new Vector3d[num];
			this.Compute();
		}
		public override Transform GetTransformation()
		{
			// get the rotation matrix when the user drags from point p to q
			// p q are supposed to be on the circle
			TransformStyle style = TransformStyle.Rotation;
			if (this.start == null || this.end == null)
			{
				return new Transform(Matrix4d.IdentityMatrix(), style);
			}
			Vector3d u = this.start.pos - this.center.pos;
			Vector3d v = this.end.pos - this.center.pos;
			Vector3d x = u.Cross(v);
			if (double.IsNaN(x.x) || x.Length() < 1e-8)
			{
				return new Transform(Matrix4d.IdentityMatrix(), style);
			}
			x = x.Normalize();
			double cos = u.Normalize().Dot(v.Normalize());
			if (cos < -1) cos = -1; else if (cos > 1) cos = 1;
			Matrix4d T = Matrix4d.RotationMatrix(x, Math.Acos(cos));
			return new Transform(T, style);
		}
		public void Compute()
		{
			Vector3d s = new Vector3d(-this.axis.y, this.axis.x, 0);
			s = s.Normalize();
			if (axis.x == 0 && axis.y == 0)
			{
				s = new Vector3d(1, 0, 0);
			}
			int num = this.points.Length;
			double dtheta = Math.PI * 2 / num;
			for (int i = 0; i < num; ++i)
			{
				double angle = i * dtheta;
				Matrix4d R = Matrix4d.RotationMatrix(axis, angle);
				Matrix3d Q = R.ToMatrix3d();
				Vector3d t = Q * s;
				this.points[i] = this.center.pos + t * radius;
			}
		}
	}
	public class TranslationMetaphor : Metaphor
	{
		public double length = 0;
		public TranslationMetaphor(Vector3d c, Vector3d axis, double length)
		{
			this.type = 0;
			this.length = length;
			this.center = new Point3(c);
			this.axis = axis;
			this.points = new Vector3d[2];
			this.points[0] = c;
			this.points[1] = c + axis * length;
			this.start = new Point3(c);
		}
		public override Transform GetTransformation()
		{
			// get the rotation matrix when the user drags from point p to q
			// p q are supposed to be on the circle
			TransformStyle style = TransformStyle.Translation;
			if (this.start == null || this.end == null)
			{
				return new Transform(Matrix4d.IdentityMatrix(), style);
			}
			return new Transform(Matrix4d.TranslationMatrix(this.end.pos - this.start.pos), style);
		}
		public void ReleaseMetaphorPoints()
		{
			this.end = null;
		}
		public void Update()
		{
			this.points[0] = this.center.pos;
			this.points[1] = this.center.pos + axis * this.length;
		}
	}
	public class BendingMetaphor
	{
		public int index = -1;
		public bool symmetryupdated = false;
		public Vector3d trans;
		public Point3 center = null;
		public List<BendingMetaphor> symmetries = new List<BendingMetaphor>();
		public List<Symmetry> symmetryelements = new List<Symmetry>();
		public BendingMetaphor(Vector3d center)
		{
			this.center = new Point3(center);
		}
		public BendingMetaphor(int index, Vector3d center)
		{
			this.index = index;
			this.center = new Point3(center);
		}
		public List<TranslationMetaphor> metaphors = new List<TranslationMetaphor>();
		public void AddMetaphor(TranslationMetaphor mtphor)
		{
			this.metaphors.Add(mtphor);
		}
		public void RemoveAllMetaphors()
		{
			this.metaphors.Clear();
		}
		public void Translate(Vector3d t)
		{
			this.trans = t;
			this.center.pos = this.center.oldpos + t;
			foreach (TranslationMetaphor mtphor in this.metaphors)
			{
				mtphor.center.pos = mtphor.center.oldpos + t;
			}
		}
		public void SymmetryPropagate()
		{
			int index = 0;
			foreach (BendingMetaphor symmmetaphor in this.symmetries)
			{
				Symmetry symm = this.symmetryelements[index++];
				if (!symmmetaphor.symmetryupdated)
				{
					symmmetaphor.symmetryupdated = true;
					Vector3d mt = ImageManipulator.GetMirrorSymmetryVector(this.trans, symm.axis);
					symmmetaphor.Translate(mt);
				}
			}
		}
		public void TranslateUpdate()
		{
			this.center.UpdateOldPos();
			foreach (TranslationMetaphor mtphor in this.metaphors)
			{
				mtphor.center.UpdateOldPos();
				mtphor.start.pos = mtphor.center.pos;
				mtphor.Update();
			}
		}
	}
	public class RigidTransMetaphor
	{
		public Point3 center = null;
		public RigidTransMetaphor(Vector3d center)
		{
			this.center = new Point3(center);
		}
		public List<TranslationMetaphor> transmetaphors = new List<TranslationMetaphor>();
		public List<RotationMetaphor> rotatmetaphors = new List<RotationMetaphor>();
		public void AddMetaphor(TranslationMetaphor mtphor)
		{
			this.transmetaphors.Add(mtphor);
		}
		public void AddMetaphor(RotationMetaphor mtphor)
		{
			this.rotatmetaphors.Add(mtphor);
		}
		public void Translate(Vector3d t)
		{
			this.center.pos = this.center.oldpos + t;
			foreach (TranslationMetaphor mtphor in this.transmetaphors)
			{
				mtphor.center.pos = mtphor.center.oldpos + t;
			}
			foreach (RotationMetaphor mtphor in this.rotatmetaphors)
			{
				mtphor.center.pos = mtphor.center.oldpos + t;
			}
		}
		public void TranslateUpdate()
		{
			this.center.UpdateOldPos();
			foreach (TranslationMetaphor mtphor in this.transmetaphors)
			{
				mtphor.center.UpdateOldPos();
				mtphor.start.pos = mtphor.center.pos;
				mtphor.Update();
			}
			foreach (RotationMetaphor mtphor in this.rotatmetaphors)
			{
				mtphor.center.UpdateOldPos();
				mtphor.start.pos = mtphor.center.pos;
				mtphor.Compute();
			}
		}
	}
	public class RegionGraph
	{
		// this class is for subdividing a polygon into regions,
		// which is used in proxy decomposition, *specific useage*
		public class RNode
		{
			// default 0(outer nodes), 1 denote nodes on outer edges, 2 is inner nodes
			public Point3 point = null;
			public int type = 0;
			public bool visited = false;
			public List<RNode> adjnodes = null;
			public void AddNeighbor(RNode another)
			{
				if (this.adjnodes == null)
					this.adjnodes = new List<RNode>();
				if (!this.adjnodes.Contains(another))
				{
					this.adjnodes.Add(another);
				}
			}
			public void RemoveNeighbor(RNode another)
			{
				this.adjnodes.Remove(another);
			}
			public RNode(Point3 p, int t)
			{
				this.type = t;
				this.point = p;
			}
		}
		public class REdge
		{
			public RNode node1 = null;
			public RNode node2 = null;
			public REdge(RNode n1, RNode n2)
			{
				this.node1 = n1;
				this.node2 = n2;
				this.node1.AddNeighbor(this.node2); // a directed edge
				this.node2.AddNeighbor(this.node1); // a directed edge
			}
		}
		public List<RNode> nodes = new List<RNode>();
		public List<REdge> outeredges = new List<REdge>();
		public List<REdge> inneredges = new List<REdge>();
		public REdge selectededge = null;
		public RegionGraph() { }
		public void CreateNode(Point3 p, int t)
		{
			this.nodes.Add(new RNode(p, t));
		}
		public void CreateOuterEdge(RNode n1, RNode n2)
		{
			// create a directed edge from n1 to n2
			this.outeredges.Add(new REdge(n1, n2));
		}
		public void CreateInnerEdge(RNode n1, RNode n2)
		{
			// create a directed edge from n1 to n2
			this.inneredges.Add(new REdge(n1, n2));
		}
		public void RemoveOuterEdge(REdge e)
		{
			e.node1.RemoveNeighbor(e.node2);
			e.node2.RemoveNeighbor(e.node1);
			this.outeredges.Remove(e);
		}
		public void RemoveInnerEdge(REdge e)
		{
			e.node1.RemoveNeighbor(e.node2);
			e.node2.RemoveNeighbor(e.node1);
			this.inneredges.Remove(e);
		}
		public void CreateOuterGraph(Polygon plg)
		{
			// create the outer graph based on the polygon
			int n = plg.points3.Count;
			for (int i = 0; i < n; ++i)
			{
				this.CreateNode(plg.points3[i],0);
			}
			for (int i = 0; i < n; ++i)
			{
				RNode n1 = this.nodes[i];
				RNode n2 = this.nodes[(i+1)%n];
				this.CreateOuterEdge(n1,n2);
			}
		}
		public Point3 GetOuterEdgePoint(Vector3d p)
		{
			double min = double.MaxValue; Point3 pt = null;
			foreach (REdge edge in this.outeredges)
			{
				double r;
				Vector3d u = edge.node1.point.pos;
				Vector3d v = edge.node2.point.pos;
				double d = ImageManipulator.Distance2LineSegment(p, u, v,out r);
				if (d < min)
				{
					this.selectededge = edge;
					pt = new Point3(u * (1 - r) + v * r);
					min = d;
				}
			}
			return pt;
		}
		public void InsertOuterNode(RNode node)
		{
			if (this.selectededge == null) return;
			REdge e = this.selectededge;
			node.type = 1;
			this.nodes.Add(node);
			this.RemoveOuterEdge(e);
			if (e.node1.type != 1)
				this.CreateOuterEdge(e.node1, node);
			if (e.node2.type != 1)
				this.CreateOuterEdge(node, e.node2);
			this.selectededge = null;
		}
		public List<List<Point3>> GetOutLoops()
		{
			// fisrt get the inner loop
			List<RNode> innerloop = new List<RNode>();
			foreach (RNode node in this.nodes)
			{
				if (node.type == 1)
				{
					Queue<RNode> Q = new Queue<RNode>();
					Q.Enqueue(node);
					node.visited = true;
					innerloop.Add(node);
					while (Q.Count > 0)
					{
						RNode s = Q.Dequeue();
						foreach (RNode adjnode in s.adjnodes)
						{
							if (!adjnode.visited && adjnode.type != 0)
							{
								Q.Enqueue(adjnode);
								adjnode.visited = true;
								innerloop.Add(adjnode);
								break;
							}
						}
					}
					break;
				}
			}
			foreach (RNode node in innerloop) node.visited = false;
			List<List<Point3>> loops = new List<List<Point3>>();
			foreach (RNode node in this.nodes)
			{
				if (node.type != 1) continue;
				bool valid = false;
				foreach (RNode adjnode in node.adjnodes)
				{
					if (adjnode.type != 1)
					{
						valid = true;
						break;
					}
				}
				if (valid) // starting from an inserting point
				{
					Queue<RNode> Q = new Queue<RNode>();
					Q.Enqueue(node);
					List<RNode> loopnodes = new List<RNode>();
					loopnodes.Add(node);
					node.visited = true;
					bool find = false;
					while (Q.Count() > 0 && !find)
					{
						RNode s = Q.Dequeue(); 
						foreach (RNode adjnode in s.adjnodes)
						{
							if (!adjnode.visited)
							{
								if (adjnode.type == 0)
								{
									Q.Enqueue(adjnode);
									loopnodes.Add(adjnode);
									adjnode.visited = true;
									break;
								}
								else if (adjnode.type == 1 && loopnodes.Count > 1) // meet another inserting point
								{
									find = true;
									loopnodes.Add(adjnode);
									break;
								}
							}
						}
					}
					// look for an inner path connecting the two inserting points
					int n = innerloop.Count;
					int j = innerloop.IndexOf(loopnodes[0]);
					int i = innerloop.IndexOf(loopnodes[loopnodes.Count-1]); // from j to i
					loopnodes[0].visited = loopnodes[loopnodes.Count - 1].visited = false;
					// there are two paths, pick the one that does not contain any intermediate 1-type node
					List<List<RNode>> paths = new List<List<RNode>>();
					for (int itr = 0; itr < 2; ++itr)
					{
						int sign = itr == 0 ? 1 : -1; bool good = false;
						for (int k = (i + sign + n) % n; k != j; k = (k + sign + n) % n)
						{
							if (innerloop[k].type == 1)
							{
								good = true;
								break;
							}
						}
						if (!good)
						{
							List<RNode> path = new List<RNode>();
							for (int k = (i + sign + n) % n; k != j; k = (k + sign + n) % n)
							{
								path.Add(innerloop[k]);
							}
							paths.Add(path);
						}
					}
					List<RNode> longestpath = null; int max = 0;
					foreach (List<RNode> path in paths)
					{
						if (max < path.Count)
						{
							longestpath = path;
							max = path.Count;
						}
					}
					if (longestpath != null) loopnodes.AddRange(longestpath);
					if (loopnodes.Count < 3) continue;
					List<Point3> loop = new List<Point3>();
					foreach (RNode nd in loopnodes)
						loop.Add(nd.point);
					loops.Add(loop);
				}
			}
			List<Point3> iloop = new List<Point3>();
			foreach (RNode node in innerloop)
			{
				iloop.Add(node.point);
			}
			loops.Add(iloop);
			return loops;
		}

	}
	public class CubicInterpolator
	{
		public static double getValue(double[] p, double x)
		{
			return p[1] + 0.5 * x * (p[2] - p[0] + x * (2.0 * p[0] - 5.0 * p[1] + 4.0 * p[2] - p[3] + x * (3.0 * (p[1] - p[2]) + p[3] - p[0])));
		}
	}
	public class BicubicInterpolator : CubicInterpolator
	{
		private double[] arr = new double[4];
		public double getValue(double[][] p, double x, double y)
		{
			arr[0] = getValue(p[0], y);
			arr[1] = getValue(p[1], y);
			arr[2] = getValue(p[2], y);
			arr[3] = getValue(p[3], y);
			return getValue(arr, x);
		}
	}
	public class CachedBicubicInterpolator
	{
		private double a00, a01, a02, a03;
		private double a10, a11, a12, a13;
		private double a20, a21, a22, a23;
		private double a30, a31, a32, a33;
		public void updateCoefficients(double[][] p)
		{
			a00 = p[1][1];
			a01 = -.5 * p[1][0] + .5 * p[1][2];
			a02 = p[1][0] - 2.5 * p[1][1] + 2 * p[1][2] - .5 * p[1][3];
			a03 = -.5 * p[1][0] + 1.5 * p[1][1] - 1.5 * p[1][2] + .5 * p[1][3];
			a10 = -.5 * p[0][1] + .5 * p[2][1];
			a11 = .25 * p[0][0] - .25 * p[0][2] - .25 * p[2][0] + .25 * p[2][2];
			a12 = -.5 * p[0][0] + 1.25 * p[0][1] - p[0][2] + .25 * p[0][3] + .5 * p[2][0] - 1.25 * p[2][1] + p[2][2] - .25 * p[2][3];
			a13 = .25 * p[0][0] - .75 * p[0][1] + .75 * p[0][2] - .25 * p[0][3] - .25 * p[2][0] + .75 * p[2][1] - .75 * p[2][2] + .25 * p[2][3];
			a20 = p[0][1] - 2.5 * p[1][1] + 2 * p[2][1] - .5 * p[3][1];
			a21 = -.5 * p[0][0] + .5 * p[0][2] + 1.25 * p[1][0] - 1.25 * p[1][2] - p[2][0] + p[2][2] + .25 * p[3][0] - .25 * p[3][2];
			a22 = p[0][0] - 2.5 * p[0][1] + 2 * p[0][2] - .5 * p[0][3] - 2.5 * p[1][0] + 6.25 * p[1][1] - 5 * p[1][2] + 1.25 * p[1][3] + 2 * p[2][0] - 5 * p[2][1] + 4 * p[2][2] - p[2][3] - .5 * p[3][0] + 1.25 * p[3][1] - p[3][2] + .25 * p[3][3];
			a23 = -.5 * p[0][0] + 1.5 * p[0][1] - 1.5 * p[0][2] + .5 * p[0][3] + 1.25 * p[1][0] - 3.75 * p[1][1] + 3.75 * p[1][2] - 1.25 * p[1][3] - p[2][0] + 3 * p[2][1] - 3 * p[2][2] + p[2][3] + .25 * p[3][0] - .75 * p[3][1] + .75 * p[3][2] - .25 * p[3][3];
			a30 = -.5 * p[0][1] + 1.5 * p[1][1] - 1.5 * p[2][1] + .5 * p[3][1];
			a31 = .25 * p[0][0] - .25 * p[0][2] - .75 * p[1][0] + .75 * p[1][2] + .75 * p[2][0] - .75 * p[2][2] - .25 * p[3][0] + .25 * p[3][2];
			a32 = -.5 * p[0][0] + 1.25 * p[0][1] - p[0][2] + .25 * p[0][3] + 1.5 * p[1][0] - 3.75 * p[1][1] + 3 * p[1][2] - .75 * p[1][3] - 1.5 * p[2][0] + 3.75 * p[2][1] - 3 * p[2][2] + .75 * p[2][3] + .5 * p[3][0] - 1.25 * p[3][1] + p[3][2] - .25 * p[3][3];
			a33 = .25 * p[0][0] - .75 * p[0][1] + .75 * p[0][2] - .25 * p[0][3] - .75 * p[1][0] + 2.25 * p[1][1] - 2.25 * p[1][2] + .75 * p[1][3] + .75 * p[2][0] - 2.25 * p[2][1] + 2.25 * p[2][2] - .75 * p[2][3] - .25 * p[3][0] + .75 * p[3][1] - .75 * p[3][2] + .25 * p[3][3];
		}
		public double getValue(double x, double y)
		{
			double x2 = x * x;
			double x3 = x2 * x;
			double y2 = y * y;
			double y3 = y2 * y;

			return (a00 + a01 * y + a02 * y2 + a03 * y3) +
			(a10 + a11 * y + a12 * y2 + a13 * y3) * x +
			(a20 + a21 * y + a22 * y2 + a23 * y3) * x2 +
			(a30 + a31 * y + a32 * y2 + a33 * y3) * x3;
		}
	}
	public class CachedVector3dBicubicInterpolator
	{
		private Vector3d a00, a01, a02, a03;
		private Vector3d a10, a11, a12, a13;
		private Vector3d a20, a21, a22, a23;
		private Vector3d a30, a31, a32, a33;
		public void updateCoefficients(Vector3d[,] p)
		{
			a00 = p[1, 1];
			a01 = -.5 * p[1, 0] + .5 * p[1, 2];
			a02 = p[1, 0] - 2.5 * p[1, 1] + 2 * p[1, 2] - .5 * p[1, 3];
			a03 = -.5 * p[1, 0] + 1.5 * p[1, 1] - 1.5 * p[1, 2] + .5 * p[1, 3];
			a10 = -.5 * p[0, 1] + .5 * p[2, 1];
			a11 = .25 * p[0, 0] - .25 * p[0, 2] - .25 * p[2, 0] + .25 * p[2, 2];
			a12 = -.5 * p[0, 0] + 1.25 * p[0, 1] - p[0, 2] + .25 * p[0, 3] + .5 * p[2, 0] - 1.25 * p[2, 1] + p[2, 2] - .25 * p[2, 3];
			a13 = .25 * p[0, 0] - .75 * p[0, 1] + .75 * p[0, 2] - .25 * p[0, 3] - .25 * p[2, 0] + .75 * p[2, 1] - .75 * p[2, 2] + .25 * p[2, 3];
			a20 = p[0, 1] - 2.5 * p[1, 1] + 2 * p[2, 1] - .5 * p[3, 1];
			a21 = -.5 * p[0, 0] + .5 * p[0, 2] + 1.25 * p[1, 0] - 1.25 * p[1, 2] - p[2, 0] + p[2, 2] + .25 * p[3, 0] - .25 * p[3, 2];
			a22 = p[0, 0] - 2.5 * p[0, 1] + 2 * p[0, 2] - .5 * p[0, 3] - 2.5 * p[1, 0] + 6.25 * p[1, 1] - 5 * p[1, 2] + 1.25 * p[1, 3] + 2 * p[2, 0] - 5 * p[2, 1] + 4 * p[2, 2] - p[2, 3] - .5 * p[3, 0] + 1.25 * p[3, 1] - p[3, 2] + .25 * p[3, 3];
			a23 = -.5 * p[0, 0] + 1.5 * p[0, 1] - 1.5 * p[0, 2] + .5 * p[0, 3] + 1.25 * p[1, 0] - 3.75 * p[1, 1] + 3.75 * p[1, 2] - 1.25 * p[1, 3] - p[2, 0] + 3 * p[2, 1] - 3 * p[2, 2] + p[2, 3] + .25 * p[3, 0] - .75 * p[3, 1] + .75 * p[3, 2] - .25 * p[3, 3];
			a30 = -.5 * p[0, 1] + 1.5 * p[1, 1] - 1.5 * p[2, 1] + .5 * p[3, 1];
			a31 = .25 * p[0, 0] - .25 * p[0, 2] - .75 * p[1, 0] + .75 * p[1, 2] + .75 * p[2, 0] - .75 * p[2, 2] - .25 * p[3, 0] + .25 * p[3, 2];
			a32 = -.5 * p[0, 0] + 1.25 * p[0, 1] - p[0, 2] + .25 * p[0, 3] + 1.5 * p[1, 0] - 3.75 * p[1, 1] + 3 * p[1, 2] - .75 * p[1, 3] - 1.5 * p[2, 0] + 3.75 * p[2, 1] - 3 * p[2, 2] + .75 * p[2, 3] + .5 * p[3, 0] - 1.25 * p[3, 1] + p[3, 2] - .25 * p[3, 3];
			a33 = .25 * p[0, 0] - .75 * p[0, 1] + .75 * p[0, 2] - .25 * p[0, 3] - .75 * p[1, 0] + 2.25 * p[1, 1] - 2.25 * p[1, 2] + .75 * p[1, 3] + .75 * p[2, 0] - 2.25 * p[2, 1] + 2.25 * p[2, 2] - .75 * p[2, 3] - .25 * p[3, 0] + .75 * p[3, 1] - .75 * p[3, 2] + .25 * p[3, 3];
		}
		public Vector3d getValue(double x, double y)
		{
			double x2 = x * x;
			double x3 = x2 * x;
			double y2 = y * y;
			double y3 = y2 * y;

			return (a00 + a01 * y + a02 * y2 + a03 * y3) +
			(a10 + a11 * y + a12 * y2 + a13 * y3) * x +
			(a20 + a21 * y + a22 * y2 + a23 * y3) * x2 +
			(a30 + a31 * y + a32 * y2 + a33 * y3) * x3;
		}
	}
	public class Symmetry
	{
		public Vector3d center;
		public Vector3d axis;
		public Symmetry(Vector3d c, Vector3d ax)
		{
			this.center = c;
			this.axis = ax;
		}
	}
	public class Options
	{
		public bool UseDominantLines { get; set; }
		public bool DebugShow { get; set; }
		public Options()
		{
			UseDominantLines = true;
		}
	}
	public class ICMLabeling
	{
		private int numnodes;
		private int numlabels;
		private int numpairs;
		private int[] pairs;
		private double[] unarycost;
		private double[] dist;
		private double[] wcost;
		private int[][] groups = null;
		public ICMLabeling(int _numnodes, int _numlabels, double[] _unarycost,
			int _numpairs, int[] _pairs, double[] _dist, double[] _wcost, int[] labels)
		{
			this.numnodes = _numnodes;
			this.numlabels = _numlabels;
			this.numpairs = _numpairs;
			this.unarycost = _unarycost;
			this.wcost = _wcost;
			this.dist = _dist;
			this.pairs = _pairs;
		}
		public void SetConstraints(int[][] _groups)
		{
			this.groups = _groups;
		}
		public int[] Process()
		{
			List<int[]> allcombs = new List<int[]>();
			List<int> current = new List<int>();
			FindAllCombs(allcombs, current, this.groups);
			double minerror = double.MaxValue; int[] bestlabel = null;
			foreach (int[] comb in allcombs)
			{
				int[] label = new int[this.numnodes];
				foreach (int j in comb)
				{
					label[j] = 1;
				}
				double error = this.ComputeEnergy(label);
				if (error < minerror)
				{
					minerror = error;
					bestlabel = label;
				}
			}
			Program.OutputText("- minimum error = " + minerror, true);
			return bestlabel;
		}
		private double ComputeEnergy(int[] labels)
		{
			double sum = 0; int index = 0;
			// unary term
			foreach (int l in labels)
			{
				sum += this.unarycost[index++ * this.numlabels + l];
			}
			// pairwise term
			for (int i = 0, j = 0; i < this.numpairs; ++i, j += 2)
			{
				int p = this.pairs[j], q = this.pairs[j + 1];
				double s = this.wcost[i]*(this.dist[labels[p]*this.numlabels+labels[q]]);
				sum += s;
			}
			return sum;
		}
		private static void FindAllCombs(List<int[]> allcombs, List<int> current, int[][] groups)
		{
			if (current.Count == groups.Length)
			{
				allcombs.Add(current.ToArray());
				current.RemoveAt(current.Count-1);
				return;
			}
			int k = current.Count;
			for (int i = 0; i < groups[k].Length; ++i)
			{
				int t = groups[k][i];
				current.Add(t);
				FindAllCombs(allcombs, current, groups);
			}
			if (current.Count > 0)
				current.RemoveAt(current.Count - 1);
		}
	}

	public class Segment
	{
		public Vector3d p1, p2;
		public Segment(Vector3d p1, Vector3d p2)
		{
			this.p1 = p1;
			this.p2 = p2;
		}
	}
	public class Line
	{
		public Vector3d pos;
		public Vector3d dir;

		public Line(Vector3d pos, Vector3d dir)
		{
			this.pos = pos;
			this.dir = dir.Normalize();
		}

		public bool Intersect(Polygon plg, out Vector3d interpt)
		{
			interpt = new Vector3d();
			plg.ComputeLocalFrame();

			// 1. (p - p0).*n = 0
			// 2. p = d*l + l0
			// d = (p0 - l0).*n / (l.*n)
			Vector3d p0 = plg.localframe.o;
			Vector3d n = plg.localframe.z;
			Vector3d l0 = this.pos;
			Vector3d l = this.dir;
			double temp = l.Dot(n);
			if (Math.Abs(temp) < 1e-5)
				return false;
			double d = (p0 - l0).Dot(n) / temp;
			interpt = d * l + l0;
			return true;
		}
	}

}
