﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text;
using System.Timers;

using Emgu.CV;
using Emgu.CV.Structure;
using Emgu.CV.UI;
using MyGeometry;

namespace i3_ImageManipulator
{

    public struct ShadowOptimizationData
    {
        public double[] heights;
        public Vector2d[] bottomcenters, shadowcenters;
        public Vector3d[] lightpositions;   // initial value.

        public double[][] weights;
        public double[][] lambdas;
        public int[][] lambdaindices;   // which seg in polygon.
        public Vector2d[][] shadowprofiles;
        public Vector2d[][] shadowpolygons;
    }

    public unsafe partial class ImageManipulator : IDisposable
    {
        private readonly Bgr ShadowMaskColor = new Bgr(255, 0, 0);
        private readonly int CompletionExtendWidth = 20;

        public int LightSamplingNumber { get; set; }
        public double LightSize { get; set; }
        public double Lightness { get; set; }

        private Image<Gray, byte> binaryshadow = null, binshadowcopy = null;
        private List<Point[]> shadows = new List<Point[]>();
        private List<Point[]> shadowprofiles = new List<Point[]>();
        private List<Point[]> shadowconvexhulls = new List<Point[]>();
        private List<PolyProxy[]> shadowproxies = new List<PolyProxy[]>();
        private List<Vector3d[]> shadowprofilepts3d = new List<Vector3d[]>();
        private List<Vector3d[]> shadowpts3d = new List<Vector3d[]>();
        private List<Vector3d[][]> shadowpolys = new List<Vector3d[][]>();
        private List<double[]> shadowerrs = new List<double[]>();
        private List<Vector3d> lightpositions = new List<Vector3d>();
        private Vector3d optimlightposition;
        private Vector3d[][] optimizationprocesspositions = null;
        private List<Vector3d[]> optimshadowpolys = new List<Vector3d[]>();
        private List<Vector3d[]> enlargedoptimshadowpolys = new List<Vector3d[]>();
        private List<Vector3d[]> currentshadowpolys = new List<Vector3d[]>();
        private List<Vector3d[]> currenthardshadowpolys = new List<Vector3d[]>();
        private List<Vector3d[]> currentsoftshadowpolys = new List<Vector3d[]>();
        private List<Vector3d[]> enlargedcurrentshadowpolys = new List<Vector3d[]>();
        private List<Vector3d[]> enlargedcurrenthardshadowpolys = new List<Vector3d[]>();
        private List<Vector3d[]> enlargedcurrentsoftshadowpolys = new List<Vector3d[]>();
        
        // soft shadow test.
        private List<Vector3d[][]> currentshadowpolygonarray = new List<Vector3d[][]>();

        private double[,] scalemap = null;//, extscalemap = null;
        private double[] lightnesses = null;
        private int[,] labelmap = null;
        private Dictionary<Polygon, int> polyids = new Dictionary<Polygon, int>();
        private Dictionary<int, Polygon> idpolys = new Dictionary<int, Polygon>();
        private ImageViewer shadowviewer = new ImageViewer();
        private ImageViewer shadowlayerviewer = new ImageViewer();

        private List<double> shadowpolyheights = new List<double>();
        private List<double[]> shadowpolybottomcenters = new List<double[]>();
        private List<double[]> shadowcenters = new List<double[]>();

        private List<Point[]> completioncontours = new List<Point[]>();

        private void ShadowCalculationWithAnnotation()
        {
            EstimateLightPositions();
            LightPositionOptimization();
            CalculateShadowWarppingData();

            ShowShadowPolys();
            ShowTopView();
            ShowOptimizedShadows();
           // ShowOptimizedShadowPolygons();
            //ShowCompletionBoundary();
        }


        private void ShadowCalculationWithLoadedLightInfo()
        {
            if (this.islightinfoloaded)
            {
                //this.GenerateOptimizedShadows();
                this.GenerateOptimizedShadowPolygonsNew();
                this.CalculateShadowWarppingData();
                this.ShowOptimizedShadows();
            }
        }

        private void ShowCompletionBoundary()
        {
            this.completioncontours.Clear();

            Image<Gray, byte> temp = new Image<Gray, byte>(this.image.Size);
            for (int i = 0; i < this.polyproxies.Count; ++i)
            {
                PolyProxy proxy = this.polyproxies[i];
                Vector3d[] shadowpoly = this.optimshadowpolys[i];
                foreach (Polygon plg in proxy.polygons)
                {
                    Point[] points = plg.points
                        .Select(p2 => Vector2Point(p2.pos))
                        .ToArray();
                    temp.FillConvexPoly(points, new Gray(255));
                    temp.DrawPolyline(points, true, new Gray(255), this.CompletionExtendWidth);
                }
                List<Point> shadowpoly2d = shadowpoly
                    .Select(p3 => Vector2Point(this.GLUProject(p3).ToVector2d()))
                    .ToList();
                //shadowpoly2d.RemoveAt(shadowpoly2d.Count - 1);
                Point[] points2d = shadowpoly2d.ToArray();
                temp.FillConvexPoly(points2d, new Gray(255));
                temp.DrawPolyline(points2d, true, new Gray(255), this.CompletionExtendWidth);
            }
            Contour<Point> contour = temp.FindContours();
            while (contour != null)
            {
                this.completioncontours.Add(contour.ToArray());
                contour = contour.HNext;
            }
            
            Image<Bgr, byte> composition = this.image.Copy();
            // draw completion polys.
            for (int i = 0; i < this.completioncontours.Count; ++i)
            {
                Point[] c = this.completioncontours[i];
                //this.DrawAntialiasedPolyline(composition, contour, (new Bgr(Color.LimeGreen)).MCvScalar, 2);
                composition.DrawPolyline(c, true, new Bgr(Color.Yellow), 2);
            }
            ImageViewer viewer = new ImageViewer(composition, "completion boundary");
            viewer.Show();

            ImageViewer viewer1 = new ImageViewer(temp, "completion mask");
            viewer1.Show();
        }


        private void DrawAntialiasedPolyline(IImage img, Point[] pts, MCvScalar color, int thickness)
        {
            for (int i = 0; i < pts.Length; ++i)
            {
                int next = (i + 1) % pts.Length;
                CvInvoke.cvLine(img.Ptr, pts[i], pts[next], color, thickness, Emgu.CV.CvEnum.LINE_TYPE.CV_AA, 0);
            }
        }

        private void DrawShadowPolygon(Image<Bgr, byte> image, Vector3d[] polygon, Color linecolor, Color pointcolor)
        {
            Point[] pts = polygon
                .Select(p => Vector2Point(this.cameracalibrator.ComputePointImgPos(p)))
                .ToArray();
            image.DrawPolyline(pts, true, new Bgr(linecolor), 1);
            foreach (Point p in pts)
                image.Draw(new CircleF(new PointF(p.X, p.Y), 2), new Bgr(pointcolor), 1);
        }

        private void ShowCurrentCompositionResults()
        {
            if (this.resultimage != null)
            {
                Image<Bgr, byte> composition = this.resultimage.Copy();

                // draw shadow polys.
                //for (int i = 0; i < this.currentshadowpolys.Count; ++i)
                //{
                //    this.DrawShadowPolygon(composition, currentshadowpolys[i], Color.Red, Color.Violet);
                //    this.DrawShadowPolygon(composition, currenthardshadowpolys[i], Color.Green, Color.Violet);
                //    this.DrawShadowPolygon(composition, currentsoftshadowpolys[i], Color.Blue, Color.Violet);
                //}

                // draw shadow polys.
                //foreach (Vector3d[][] polygons in this.currentshadowpolygonarray)
                //    foreach (Vector3d[] polygon in polygons)
                //        this.DrawShadowPolygon(composition, polygon, Color.Red, Color.Violet);

                this.shadowviewer.Image = composition;
                this.shadowviewer.Text = "composition result";
                this.shadowviewer.ClientSize = this.shadowviewer.Image.Size;
                this.shadowviewer.Show();
            }
            if (this.shadowlayer != null)
            {
                this.shadowviewer.Image = this.shadowlayer;
                this.shadowviewer.Text = "shadow layer";
                this.shadowviewer.ClientSize = this.shadowviewer.Image.Size;
                this.shadowviewer.Show();
            }
        }

        ImageViewer stepwiseviewer = new ImageViewer();

        public void ShowOptimizedShadowPolygonsStepwiseAuto()
        {
            Timer timer = new Timer();
            timer.Elapsed += new ElapsedEventHandler((obj, eea) => this.ShowOptimizedShadowPolygonsStepwise());
            timer.Interval = 30;
            timer.AutoReset = true;
            timer.SynchronizingObject = this.parentview;
            timer.Enabled = true;
        }
        public void ShowOptimizedShadowPolygonsStepwise()
        {
			return;
			//if (stepwiseviewer == null) return;
			//Image<Bgr, byte> composition = this.image.Copy();
			//stepwiseviewer.Text = "optimized shadows polygons stepwise";
			//stepwiseviewer.TopMost = true;

			//int allsteps = this.optimizationprocesspositions.Sum(p3s => p3s.Length);
			//this.optimizationshadowpolygonsstep %= allsteps;

			//// draw shadow polys.
			//int step = 0;
			//for (int i = 0; i < this.optimizationprocesspositions.Length; ++i)
			//{
			//    Vector3d[] positions = this.optimizationprocesspositions[i];
			//    foreach (Vector3d lightpos in positions)
			//    {
			//        if (step == this.optimizationshadowpolygonsstep)
			//        {
			//            foreach (PolyProxy proxy in this.polyproxies)
			//            {
			//                Vector3d[] polygon = this.CalculateProxyShadowNew(proxy, lightpos);
			//                Point[] pts = polygon
			//                    .Select(p => Vector2Point(this.cameracalibrator.ComputePointImgPos(p)))
			//                    .ToArray();

			//                //composition.DrawPolyline(pts, true, new Bgr(Color.Red), 1);
			//                this.DrawAntialiasedPolyline(composition, pts, (new Bgr(colorMall[i])).MCvScalar, 2);
			//            }
			//            break;
			//        }
			//        step++;
			//    }
			//    if (step == this.optimizationshadowpolygonsstep)
			//        break;
			//}

			//this.optimizationshadowpolygonsstep++;

			//stepwiseviewer.Image = composition;
			//stepwiseviewer.ClientSize = stepwiseviewer.Image.Size;
			//stepwiseviewer.Show();
        }
        private void ShowOptimizedShadowPolygons()
        {
            Image<Bgr, byte> composition = this.image.Copy();
            ImageViewer viewer = new ImageViewer();
            viewer.Text = "optimized shadows polygons";

            // draw shadow polys.
            for (int i = 0; i < this.optimizationprocesspositions.Length; ++i)
            {
                Vector3d[] positions = this.optimizationprocesspositions[i];
                foreach (Vector3d lightpos in positions)
                {
                    foreach (PolyProxy proxy in this.polyproxies)
                    {
                        Vector3d[] polygon = this.CalculateProxyShadowNew(proxy, lightpos);
                        Point[] pts = polygon
                            .Select(p => Vector2Point(this.cameracalibrator.ComputePointImgPos(p)))
                            .ToArray();

                        //composition.DrawPolyline(pts, true, new Bgr(Color.Red), 1);
                        this.DrawAntialiasedPolyline(composition, pts, (new Bgr(colorMall[i])).MCvScalar, 1);
                    }
                }
            }

            viewer.Image = composition;
            viewer.ClientSize = viewer.Image.Size;
            viewer.Show();
        }
        private void ShowOptimizedShadows()
        {
			return;
			//Image<Bgr, byte> composition = this.image.Copy();

			//// draw shadow polys.
			//for (int i = 0; i < this.optimshadowpolys.Count; ++i)
			//{
			//    Vector3d[] pts3d = optimshadowpolys[i];
			//    Point[] pts = pts3d
			//        .Select(p => Vector2Point(this.cameracalibrator.ComputePointImgPos(p)))
			//        .ToArray();

			//    //composition.DrawPolyline(pts, true, new Bgr(Color.Red), 1);
			//    this.DrawAntialiasedPolyline(composition, pts, (new Bgr(Color.Red)).MCvScalar, 1);
			//    foreach (Point p in pts)
			//        composition.Draw(new CircleF(new PointF(p.X, p.Y), 2), new Bgr(Color.Violet), 1);
			//}

			//ImageViewer viewer = new ImageViewer(composition, "optimized shadows polygons");
			//viewer.Show();
        }

        private void ShowTopView()
        {
            List<Vector2d[]> proxypts = new List<Vector2d[]>();
            List<Vector2d[]> shadowpts = new List<Vector2d[]>();
            List<Vector2d[]> lightlinepts = new List<Vector2d[]>();
            List<Vector2d> lightpts = new List<Vector2d>();
            List<Vector2d[]> optimpts = new List<Vector2d[]>(); 
            Vector2d optimlp = this.optimlightposition.ToVector2d();

            //proxypts.Add(this.roomlayout.box.polygons[0].points3
            //    .Select(p => p.pos.ToVector2d())
            //    .ToArray());
            foreach (PolyProxy proxy in this.polyproxies)
            {
                proxypts.Add(proxy.polygons[0].points3
                    .Select(p => p.pos.ToVector2d())
                    .ToArray());
            }
            foreach (Vector3d[][] shadowpoly in shadowpolys)
            {
                shadowpts.Add(shadowpoly[0]
                    .Select(p => p.ToVector2d())
                    .ToArray());
            }
            for (int i = 0; i < this.lightpositions.Count; ++i)
            {
                Vector2d c = (shadowpts[i][0] + shadowpts[i][4]) / 2;
                Vector2d sc = (shadowpts[i][1] + shadowpts[i][3]) / 2;
                Vector2d dir = (sc - c).Normalize();
                Vector2d[] lines = new Vector2d[2];
                lines[0] = c - 100 * dir;
                lines[1] = sc + 100 * dir;
                lightlinepts.Add(lines);
            }
            for (int i = 0; i < this.lightpositions.Count; ++i)
            {
                lightpts.Add(this.lightpositions[i].ToVector2d());
            }
            for (int i = 0; i < this.optimizationprocesspositions.Length; ++i)
            {
                optimpts.Add(this.optimizationprocesspositions[i].Select(p3 => p3.ToVector2d()).ToArray());
            }

            double minx, maxx, miny, maxy;
            minx = miny = double.MaxValue;
            maxx = maxy = double.MinValue;
            foreach (Vector2d[] pts in proxypts)
            {
                foreach (Vector2d p in pts)
                {
                    minx = Math.Min(minx, p.x);
                    maxx = Math.Max(maxx, p.x);
                    miny = Math.Min(miny, p.y);
                    maxy = Math.Max(maxy, p.y);
                }
            }
            foreach (Vector2d[] pts in shadowpts)
            {
                foreach (Vector2d p in pts)
                {
                    minx = Math.Min(minx, p.x);
                    maxx = Math.Max(maxx, p.x);
                    miny = Math.Min(miny, p.y);
                    maxy = Math.Max(maxy, p.y);
                }
            }
            foreach (Vector2d p in lightpts)
            {
                minx = Math.Min(minx, p.x);
                maxx = Math.Max(maxx, p.x);
                miny = Math.Min(miny, p.y);
                maxy = Math.Max(maxy, p.y);
            }
            {
                Vector2d p = optimlp;
                minx = Math.Min(minx, p.x);
                maxx = Math.Max(maxx, p.x);
                miny = Math.Min(miny, p.y);
                maxy = Math.Max(maxy, p.y);
            }
            foreach (Vector2d[] pts in optimpts)
            {
                foreach (Vector2d p in pts)
                {
                    minx = Math.Min(minx, p.x);
                    maxx = Math.Max(maxx, p.x);
                    miny = Math.Min(miny, p.y);
                    maxy = Math.Max(maxy, p.y);
                }
            }

            //double factor = 1;
            //double xlen = factor * (maxx - minx);
            //double ylen = factor * (maxy - miny);
            //minx -= xlen;
            //maxx += xlen;
            //miny -= ylen;
            //maxy += ylen;

            // switch x and y.
            int width = 1000;
            int height = (int)(width * (maxx - minx) / (maxy - miny));
            foreach (Vector2d[] pts in proxypts)
            {
                for (int i = 0; i < pts.Length; ++i)
                {
                    Vector2d p = pts[i];
                    pts[i].y = (p.x - minx) * height / (maxx - minx);
                    pts[i].x = (p.y - miny) * width / (maxy - miny);
                }
            }
            foreach (Vector2d[] pts in shadowpts)
            {
                for (int i = 0; i < pts.Length; ++i)
                {
                    Vector2d p = pts[i];
                    pts[i].y = (p.x - minx) * height / (maxx - minx);
                    pts[i].x = (p.y - miny) * width / (maxy - miny);
                }
            }
            foreach (Vector2d[] pts in lightlinepts)
            {
                for (int i = 0; i < pts.Length; ++i)
                {
                    Vector2d p = pts[i];
                    pts[i].y = (p.x - minx) * height / (maxx - minx);
                    pts[i].x = (p.y - miny) * width / (maxy - miny);
                }
            }
            for (int i = 0; i < lightpts.Count; ++i)
            {
                Vector2d p = lightpts[i];
                double x = (p.x - minx) * height / (maxx - minx);
                double y = (p.y - miny) * width / (maxy - miny);
                lightpts[i] = new Vector2d(y, x);
            }
            {
                Vector2d p = optimlp;
                double x = (p.x - minx) * height / (maxx - minx);
                double y = (p.y - miny) * width / (maxy - miny);
                optimlp.x = y;
                optimlp.y = x;
            }
            for (int i = 0; i < optimpts.Count; ++i)
            {
                Vector2d[] pts = optimpts[i];
                int j = 0;
                foreach (Vector2d p in pts)
                {
                    double x = (p.x - minx) * height / (maxx - minx);
                    double y = (p.y - miny) * width / (maxy - miny);
                    optimpts[i][j] = new Vector2d(y, x);
                    j++;
                }
            }

            // draw.
            Point[][] points1 = Vector2dsToPoints(proxypts.ToArray());
            Point[][] points2 = Vector2dsToPoints(shadowpts.ToArray());
            Point[][] points3 = Vector2dsToPoints(lightlinepts.ToArray());
            Image<Bgr, byte> topview = new Image<Bgr, byte>(width, height);
            topview.DrawPolyline(points2, true, new Bgr(0, 0, 255), 1);
            topview.DrawPolyline(points1, true, new Bgr(0, 255, 255), 1);
            topview.DrawPolyline(points3, false, new Bgr(0, 255, 0), 1);

            for (int i = 0; i < lightpts.Count; ++i)
            {
                Vector2d center = lightpts[i];
                topview.Draw(new CircleF(new PointF((float)center.x, (float)center.y), 3f), new Bgr(Color.Orange), 4);
            }
            for (int i = 0; i < optimpts.Count; ++i)
            {
                for (int j = 0; j < optimpts[i].Length; ++j)
                {
                    Vector2d center = optimpts[i][j];
                    topview.Draw(new CircleF(new PointF((float)center.x, (float)center.y), 3f), new Bgr(colorMall[i]), 4);
                }
                Vector2d finalpos = optimpts[i][optimpts[i].Length - 1];
                topview.Draw(new CircleF(new PointF((float)finalpos.x, (float)finalpos.y), 3f), new Bgr(Color.White), 4);
            }
            //topview.Draw(new CircleF(new PointF((float)optimlp.x, (float)optimlp.y), 3f), new Bgr(Color.White), 4);

            ImageViewer viewer = new ImageViewer(topview);
            viewer.Size = topview.Size;
            viewer.Show();
        }

        private Point[][] Vector2dsToPoints(Vector2d[][] pts)
        {
            Point[][] result = new Point[pts.GetLength(0)][];
            for (int i = 0; i < result.GetLength(0); ++i)
                result[i] = pts[i].Select(p => new Point((int)p.x, (int)p.y)).ToArray();
            return result;
        }

        private void ShowShadowPolys()
        {
            Image<Bgr, byte> composition = this.image.Copy();

            // draw shadow polys.
            for (int i = 0; i < this.shadowpolys.Count; ++i)
            {
                Vector3d[] pts3d = shadowpolys[i][0];
                Point[] pts = pts3d
                    .Select(p => Vector2Point(this.cameracalibrator.ComputePointImgPos(p)))
                    .ToArray();

                composition.DrawPolyline(pts, true, new Bgr(Color.Red), 1);
                foreach (Point p in pts)
                    composition.Draw(new CircleF(new PointF(p.X, p.Y), 2), new Bgr(Color.Orange), 1);
            }

            ImageViewer viewer = new ImageViewer(composition);
            viewer.Show();
        }

        private void ShowShadows()
        {
            // draw shadows.
            Image<Bgr, byte> original = this.image.Copy();
            Image<Bgr, byte> mask = this.image.CopyBlank();
            mask.SetValue(ShadowMaskColor, binaryshadow);
            Image<Bgr, byte> composition = original * 0.5 + mask * 0.5;
            original.Dispose();
            mask.Dispose();

            // draw convex hulls.
            composition.DrawPolyline(this.shadowconvexhulls.ToArray(), true, new Bgr(Color.Red), 1);
            foreach (Point[] pts in this.shadowconvexhulls)
                foreach (Point p in pts)
                    composition.Draw(new CircleF(new PointF(p.X, p.Y), 2), new Bgr(Color.Orange), 1);

            ImageViewer viewer = new ImageViewer(composition);
            viewer.Show();
        }

        private bool ReadBinaryShadowImage(string name)
        {
            if (!File.Exists(name))
            {
                Program.OutputText("No binary shadow file exists!", true);
                return false;
            }
            binaryshadow = new Image<Gray, byte>(name);
            binshadowcopy = binaryshadow.Copy();
            return true;
        }

        private Dictionary<int, PolyProxy> GetLabelProxies()
        {
            Dictionary<int, PolyProxy> result = new Dictionary<int, PolyProxy>();
            foreach (PolyProxy proxy in this.polyproxies)
                foreach (Polygon poly in proxy.polygons)
                    //if (poly.visible)
                        if (poly.textureindex != -1)
                            result.Add(poly.textureindex, proxy);
            return result;
        }

        private void FitShadows()
        {
            ComputeShadowProxyRelations();
            ComputeShadowConvexHulls();

            for (int i = 0; i < this.shadowconvexhulls.Count; ++i)
            {
                Point[] hull = this.shadowconvexhulls[i];
                FitShadows(this.shadowproxies[i][0], this.shadowpts3d[i], hull);
            }
        }

        private void FitShadows(PolyProxy proxy, Vector3d[] shadowpts, Point[] hull)
        {
            Polygon bottom = proxy.polygons[0];
            bottom.ComputeLocalFrame();
            CoordinateFrame localframe = bottom.localframe;
            //this.useraxes.Add(new KeyValuePair<Vector3d, Vector3d>(localframe.o, localframe.y));
            double xmin, xmax, ymin, ymax;
            xmin = ymin = double.MaxValue;
            xmax = ymax = double.MinValue;
            // z should be zero.
            Vector3d[] btmpts3d = bottom.points3
                .Select(p => p.pos)
                .ToArray();
            Vector3d[] btmtranspts3d = bottom.points3
                .Select(p => localframe.GetPointLocalCoord(p.pos))
                .ToArray();
            for (int i = 0; i < btmtranspts3d.Length; ++i)
            {
                Vector3d pt3 = btmtranspts3d[i];
                xmin = Math.Min(pt3.x, xmin);
                xmax = Math.Max(pt3.x, xmax);
                ymin = Math.Min(pt3.y, ymin);
                ymax = Math.Max(pt3.y, ymax);
            }

            Vector3d[] pts3d = hull
                .Select(p => this.cameracalibrator.ComputePointGround3dPos(new Vector2d(p.X, p.Y)))
                .ToArray();
            Vector2d[] pts2d = pts3d
                .Select(p => p.ToVector2d())
                .ToArray();
            Vector2d[] transpts2d = pts3d
				.Select(p => bottom.localframe.GetPointLocalCoord(p).ToVector2d())
                .ToArray();
            int signx = 0, signy = 0;
            int negx = 0, posx = 0;
            int negy = 0, posy = 0;
            for (int i = 0; i < transpts2d.Length; ++i)
            {
                if (transpts2d[i].x < xmin) negx += 1;
                if (transpts2d[i].x > xmax) posx += 1;
                if (transpts2d[i].y < ymin) negy += 1;
                if (transpts2d[i].y > ymax) posy += 1;
            }
            signx = (negx < posx) ? 1 : -1;
            signy = (negy < posy) ? 1 : -1;
            double centerx = (xmin + xmax) / 2;
            double centery = (ymin + ymax) / 2;
            int corner1index = -1, corner2index = -1, corner3index = -1;
            for (int i = 0; i < btmtranspts3d.Length; ++i)
            {
                if (signx == Math.Sign(btmtranspts3d[i].x - centerx) &&
                    signy == Math.Sign(btmtranspts3d[i].y - centery))
                {
                    corner2index = i;
                    break;
                }
            }
            corner1index = (corner2index - 1 + btmpts3d.Length) % btmpts3d.Length;
            corner3index = (corner2index + 1) % btmpts3d.Length;
            Vector3d corner1 = btmpts3d[corner1index];
            Vector3d corner2 = btmpts3d[corner2index];
            Vector3d corner3 = btmpts3d[corner3index];
            if (((corner2 - corner1).Cross(corner3 - corner2)).z < 0)
            {
                int temp = corner1index;
                corner1index = corner3index;
                corner3index = temp;
                corner1 = btmpts3d[corner1index];
                corner3 = btmpts3d[corner3index];
            }
            Vector2d cor1 = corner1.ToVector2d();
            Vector2d cor2 = corner2.ToVector2d();
            Vector2d cor3 = corner3.ToVector2d();

            //this.userballs.Add(corner1);

            // candidate 3-edges group.
            List<int[]> edgegroups = new List<int[]>();
            for (int i = 0; i < hull.Length; ++i)
            {
                for (int j = i + 1; j < hull.Length; ++j)
                {
                    for (int k = j + 1; k < hull.Length; ++k)
                    {
                        edgegroups.Add(new int[] { i, j, k });
                        edgegroups.Add(new int[] { j, k, i });
                        edgegroups.Add(new int[] { k, i, j });
                    }
                }
            }

            List<double> shadowpolyerrs = new List<double>();
            List<Vector3d[]> shadowpolys = new List<Vector3d[]>();
            foreach (int[] group in edgegroups)
            {
                int pindex1 = group[0];
                int pindex2 = group[1];
                int pindex3 = group[2];
                Vector2d seg1u = pts2d[pindex1];
                Vector2d seg1v = pts2d[(pindex1 + 1) % hull.Length];
                Vector2d seg1c = (seg1u + seg1v) / 2;
                Vector2d seg2u = pts2d[pindex2];
                Vector2d seg2v = pts2d[(pindex2 + 1) % hull.Length];
                Vector2d seg2c = (seg2u + seg2v) / 2;
                Vector2d seg3u = pts2d[pindex3];
                Vector2d seg3v = pts2d[(pindex3 + 1) % hull.Length];
                Vector2d seg3c = (seg3u + seg3v) / 2;

                Vector2d seg1p = seg1u;
                if ((seg1u - cor1).Cross(seg1v - cor1) < 0) seg1p = seg1v;
                Vector2d seg2p = seg2u;
                if ((seg2v - seg2u).Cross(cor2 - cor1) > 0) seg2p = seg2v;
                Vector2d seg3p = seg3u;
                if ((seg3v - seg3u).Cross(cor3 - cor2) > 0) seg3p = seg3v;
                Vector2d dir1 = (seg1p - cor1).Normalize();
                Vector2d dir2 = (cor2 - cor1).Normalize();
                Vector2d dir3 = (cor3 - cor2).Normalize();
                if (dir1.Dot(dir2) < 0) continue;

                double deg1 = Math.Acos(dir1.Dot((seg1v - seg1u).Normalize())) * 180 / Math.PI;
                if (deg1 > 45) continue;
                double deg2 = Math.Acos(dir2.Dot((seg2v - seg2u).Normalize())) * 180 / Math.PI;
                if (deg2 > 45) continue;
                double deg3 = Math.Acos(dir3.Dot((seg3v - seg3u).Normalize())) * 180 / Math.PI;
                if (deg3 > 45) continue;

                Vector2d interpt1;
                bool isinter1 = IntersectionHelper.LinesIntersection(cor1, seg1p, seg2p, seg2p + dir2, out interpt1);
                if (!isinter1) continue;
                if ((interpt1 - seg1v).Dot(dir1) <= 0 || (seg2u - interpt1).Dot(dir2) <= 0) continue;
                Vector2d interpt2;
                bool isinter2 = IntersectionHelper.LinesIntersection(seg2p, seg2p + dir2, seg3p, seg3p + dir3, out interpt2);
                if (!isinter2) continue;
                if ((interpt2 - seg2v).Dot(dir2) <= 0 || (seg3u - interpt2).Dot(dir3) <= 0) continue;
                double lenfac = (cor3 - cor2).Length() / (cor2 - cor1).Length();
                Vector2d interpt3 = interpt2 + dir3 * lenfac * (interpt2 - interpt1).Length();

                Vector2d[] shadowpoly = new Vector2d[6];
                shadowpoly[0] = cor1;
                shadowpoly[1] = interpt1;
                shadowpoly[2] = interpt2;
                shadowpoly[3] = interpt3;
                shadowpoly[4] = cor3;
                shadowpoly[5] = cor2;
                shadowpolys.Add(shadowpoly.Select(p => new Vector3d(p)).ToArray());

                // fitting errors.
                double error = 0;
                for (int k = 0; k < shadowpts.Length; ++k)
                {
                    Vector2d p = shadowpts[k].ToVector2d();
                    bool isinpoly = Shape2D.PointInPoly(p, shadowpoly);
                    double dist = Shape2D.Point2PolyBoundaryDist(p, shadowpoly);
                    error += (isinpoly ? 0 : dist);
                }
                shadowpolyerrs.Add(error);
            }

            Vector3d[][] temppolys = shadowpolys.ToArray();
            double[] temperrs = shadowpolyerrs.ToArray();
            Array.Sort(temperrs, temppolys);
            this.shadowpolys.Add(temppolys);
            this.shadowerrs.Add(temperrs);
        }

        private void ComputeShadowConvexHulls()
        {
            using (MemStorage storage = new MemStorage())
            {
                for (int i = 0; i < this.shadows.Count; ++i)
                {
                    Point[] shadow = this.shadows[i];
                    Image<Gray, byte> tempbin = this.binaryshadow.CopyBlank();
                    for (int j = 0; j < shadow.Length; ++j)
                        tempbin[shadow[j].Y, shadow[j].X] = new Gray(255);

                    List<Contour<Point>> contours = new List<Contour<Point>>();
                    for (Contour<Point> contour = tempbin.FindContours(); contour != null; contour = contour.HNext)
                        contours.Add(contour);

                    Contour<Point> outer = null;
                    for (int j = 0; j < contours.Count; ++j)
                    {
                        bool allinthiscontour = true;
                        for (int k = 0; k < contours.Count; ++k)
                        {
                            if (k == j) continue;
                            for (int m = 0; m < contours[k].Total; ++m)
                            {
                                Point cp = contours[k][m];
                                if (contours[j].InContour(new PointF(cp.X, cp.Y)) < 0)
                                {
                                    allinthiscontour = false;
                                    break;
                                }
                            }
                            if (!allinthiscontour) break;
                        }
                        if (allinthiscontour)
                        {
                            outer = contours[j];
                            break;
                        }
                    }
                    if (outer == null)
                        throw new ApplicationException("FitShadows(): No outer contour!");

                    //this.shadowprofiles.Add(outer.ToArray());
                    //Vector3d[] pts3d = this.cameracalibrator.ComputePointGround3dPos(outer.ToArray());
                    //this.shadowprofilepts3d.Add(pts3d);

					Point[] convexhull = outer.GetConvexHull(Emgu.CV.CvEnum.ORIENTATION.CV_COUNTER_CLOCKWISE).ToArray();
					this.shadowprofiles.Add(convexhull);
					Vector3d[] pts3d = this.cameracalibrator.ComputePointGround3dPos(convexhull);
                    this.shadowprofilepts3d.Add(pts3d);

                    Seq<Point> convpts = outer.GetConvexHull(Emgu.CV.CvEnum.ORIENTATION.CV_CLOCKWISE);
                    this.shadowconvexhulls.Add(convpts.ToArray());
                }
            }
        }

        private void ComputeShadowProxyRelations()
        {
            for (int i = 0; i < this.shadows.Count; ++i)
            {
                Point[] shadow = this.shadows[i];
                PolyProxy[] proxies = this.shadowproxies[i];
                if (proxies.Length == 1) continue;
                double[] mindists = new double[proxies.Length];
                for (int j = 0; j < mindists.Length; ++j)
                    mindists[j] = double.MaxValue;

                Vector3d[] shadowpts = new Vector3d[shadow.Length];
                for (int j = 0; j < shadow.Length; ++j)
                {
                    Point p = shadow[j];
                    Vector3d p3 = this.cameracalibrator.ComputePointGround3dPos(new Vector2d(p.X, p.Y));
                    shadowpts[j] = p3;
                    for (int k = 0; k < proxies.Length; ++k)
                    {
                        PolyProxy proxy = proxies[k];
                        Point3[] pts3 = proxy.polygons[0].points3.ToArray();
                        Vector2d[] polypts = pts3.Select(pt3 => pt3.pos.ToVector2d()).ToArray();
                        double dist = Shape2D.Point2PolyBoundaryDist(p3.ToVector2d(), polypts);
                        mindists[k] = Math.Min(dist, mindists[k]);
                    }
                }
                this.shadowpts3d.Add(shadowpts);

                double mindist = double.MaxValue;
                PolyProxy minpoly = null;
                for (int j = 0; j < mindists.Length; ++j)
                {
                    if (mindists[j] < mindist)
                    {
                        mindist = mindists[j];
                        minpoly = proxies[j];
                    }
                }

                this.shadowproxies[i] = new PolyProxy[] { minpoly };
            }
        }

        private void EstimateLightPositions()
        {
            List<Vector2d[]> lines = new List<Vector2d[]>();
            for (int i = 0; i < this.shadowpolys.Count; ++i)
            {
                Vector3d[] pts3d = shadowpolys[i][0];
                Vector2d c1 = pts3d[0].ToVector2d();
                Vector2d c3 = pts3d[4].ToVector2d();
                Vector2d p1 = pts3d[1].ToVector2d();
                Vector2d p3 = pts3d[3].ToVector2d();
                lines.Add(new Vector2d[] { (c1 + c3) / 2, (p1 + p3) / 2 });
            }

            for (int i = 0; i < lines.Count; ++i)
            {
                for (int j = i + 1; j < lines.Count; ++j)
                {
                    Vector2d plightground;
                    bool isinter = IntersectionHelper.LinesIntersection(
                        lines[i][0], lines[i][1], lines[j][0], lines[j][1], out plightground);
                    if (!isinter)
                        throw new ApplicationException("EstimateLight(): no intersection during calculation of light position");
                    if ((plightground - lines[i][0]).Dot(lines[i][0] - lines[i][1]) < 0)
                        plightground = (plightground - lines[i][0]).Length() * ((lines[i][0] - lines[i][1]).Normalize()) + lines[i][0];
                    double h = this.shadowproxies[i][0].zlen;
                    double H = (plightground - lines[i][1]).Length() * h / (lines[i][0] - lines[i][1]).Length();
                    this.lightpositions.Add(new Vector3d(plightground, H));
                }
            }
        }

        private void LightPositionOptimization()
        {
            //OutputOptimizationData();
            ShadowOptimizationData data = GenerateShadowOptimizationData();
            //ShadowNonLinearOptimizer op = new ShadowNonLinearOptimizer();
            //this.optimlightposition = op.Optimize(data);
            //Program.OutputText("optimized lightpos: " + this.optimlightposition.ToString(), true);

            LightPositionOptimizer op = new LightPositionOptimizer();
            this.optimlightposition = op.Optimize(data);
            Program.OutputText("optimized lightpos: " + this.optimlightposition.ToString(), true);

            this.LightSize = this.optimlightposition.z / 100;
            this.optimizationprocesspositions = op.OptimizationProcessPositions;

            GenerateOptimizedShadows();
        }

        private ShadowOptimizationData GenerateShadowOptimizationData()
        {
            ShadowOptimizationData data = new ShadowOptimizationData();
            data.heights = new double[this.shadowpolys.Count];
            data.bottomcenters = new Vector2d[data.heights.Length];
            data.shadowcenters = new Vector2d[data.heights.Length];

            // bottom polygon centers of proxies.
            // shadow centers of shadow polygons (not the whole, the square part).
            // proxy heights.
            for (int i = 0; i < this.shadowpolys.Count; ++i)
            {
                Vector2d[] lines = new Vector2d[5];
                Vector2d c1 = this.shadowpolys[i][0][0].ToVector2d();
                Vector2d c3 = this.shadowpolys[i][0][4].ToVector2d();
                Vector2d p1 = this.shadowpolys[i][0][1].ToVector2d();
                Vector2d p3 = this.shadowpolys[i][0][3].ToVector2d();
                Vector2d bc = (c1 + c3) / 2;
                Vector2d sc = (p1 + p3) / 2;

                data.bottomcenters[i] = bc;
                data.shadowcenters[i] = sc;
                data.heights[i] = this.shadowproxies[i][0].zlen;
            }

            // initial light positions.
            data.lightpositions = this.lightpositions.ToArray();
            //data.lightposition = this.lightpositions[0];
            //data.lightposition = this.lightpositions[1];
            //data.lightposition = this.lightpositions[2];
            //data.lightposition = new Vector3d(2.95247733493365, 0.137008159474699, 3.07788370881121);
            //data.lightposition = new Vector3d(1.96117011019955, 0.986181775192071, 2.38025911338542);

            data = ComputeLambdas(data);

            return data;
        }

        private ShadowOptimizationData ComputeLambdas(ShadowOptimizationData data)
        {
            // shadow profiles.
            // shadow polygons.
            data.shadowprofiles = new Vector2d[data.heights.Length][];
            data.shadowpolygons = new Vector2d[data.heights.Length][];
            for (int i = 0; i < this.shadowprofilepts3d.Count; ++i)
            {
                data.shadowprofiles[i] = this.shadowprofilepts3d[i]
                    .Select(p => p.ToVector2d())
                    .ToArray();

                Vector2d c1 = this.shadowpolys[i][0][0].ToVector2d();
                Vector2d c2 = this.shadowpolys[i][0][5].ToVector2d();
                Vector2d c3 = this.shadowpolys[i][0][4].ToVector2d();
                //Vector2d lg = data.lightposition.ToVector2d();
                //double lh = data.lightposition.z;
                //double h = data.heights[i];

                data.shadowpolygons[i] = new Vector2d[6];
                data.shadowpolygons[i][0] = c1;
                //data.shadowpolygons[i][1] = (c1 - lg) * h / (lh - h) + c1;
                //data.shadowpolygons[i][2] = (c2 - lg) * h / (lh - h) + c2;
                //data.shadowpolygons[i][3] = (c3 - lg) * h / (lh - h) + c3;
                data.shadowpolygons[i][4] = c3;
                data.shadowpolygons[i][5] = c2;
            }

            // weigths, lambdas, lambda indices.
            //data.weights = new double[data.shadowprofiles.Length][];
            //data.lambdas = new double[data.shadowprofiles.Length][];
            //data.lambdaindices = new int[data.shadowprofiles.Length][];
            //for (int i = 0; i < data.shadowprofiles.Length; ++i)
            //{
            //    Vector2d[] profile = data.shadowprofiles[i];
            //    Vector2d[] polygon = data.shadowpolygons[i];
            //    data.weights[i] = new double[profile.Length];
            //    data.lambdas[i] = new double[profile.Length];
            //    data.lambdaindices[i] = new int[profile.Length];
            //    for (int j = 0; j < profile.Length; ++j)
            //    {
            //        Vector2d p = profile[j];
            //        bool inpoly = Shape2D.PointInPoly(p, polygon);
            //        data.weights[i][j] = inpoly ? 1 : LightPositionOptimizer.OutWeight;
            //        int segindex;
            //        double lambda;
            //        Shape2D.Point2PolyBoundaryDist(p, polygon, out segindex, out lambda);
            //        data.lambdas[i][j] = lambda;
            //        data.lambdaindices[i][j] = segindex;
            //    }
            //}
            return data;
        }

        private void GenerateOptimizedShadowPolygonsNew()
        {
            for (int i = 0; i < this.polyproxies.Count; ++i)
            {
                PolyProxy proxy = this.polyproxies[i];
                Vector3d[] shadowpoly = this.CalculateProxyShadowNew(proxy, this.optimlightposition);
                this.optimshadowpolys.Add(shadowpoly);
            }
        }

        private void GenerateOptimizedShadows()
        {
            for (int i = 0; i < this.polyproxies.Count; ++i)
            {
                PolyProxy proxy = this.polyproxies[i];
                Vector3d[] shadowpoly = this.CalculateProxyShadowNew(proxy, this.optimlightposition);
                //Vector3d[] shadowpoly = this.CalculateProxyShadow(proxy, this.optimlightposition);
                this.optimshadowpolys.Add(shadowpoly);
            }
        }

        private void OutputOptimizationData()
        {
            string dirname = Path.GetDirectoryName(this.imagename);
            string filename = Path.GetFileNameWithoutExtension(this.imagename);
            StreamWriter sw = new StreamWriter(Path.Combine(dirname, filename + "_optimdata.txt"));
            for (int i = 0; i < this.shadowpolys.Count; ++i)
            {
                sw.WriteLine("h: {0}", this.shadowproxies[i][0].zlen);
                Vector2d[] lines = new Vector2d[5];
                Vector2d c1 = this.shadowpolys[i][0][0].ToVector2d();
                Vector2d c3 = this.shadowpolys[i][0][4].ToVector2d();
                Vector2d p1 = this.shadowpolys[i][0][1].ToVector2d();
                Vector2d p3 = this.shadowpolys[i][0][3].ToVector2d();
                Vector2d bc = (c1 + c3) / 2;
                Vector2d sc = (p1 + p3) / 2;

                sw.WriteLine("b: ({0}, {1})", bc.x, bc.y);
                sw.WriteLine("s: ({0}, {1})", sc.x, sc.y);
                sw.WriteLine();
            }

            for (int i = 0; i < this.lightpositions.Count; ++i)
            {
                Vector3d lp = this.lightpositions[i];
                sw.WriteLine("l: ({0}, {1}, {2})", lp.x, lp.y, lp.z);
            }
            sw.Close();

            Program.OutputText("Write optimization data to " + Path.Combine(dirname, filename + "_optimdata.txt"), true);
        }

        public void DoShadowMapping()
        {
            //if (this.selectedproxy == null) return;
            //this.previoushandle = this.selectedproxy;
            if (this.optimshadowpolys.Count == 0) return;
            //this.resultimage = this.imlayer.Copy();

            this.CalculateAndWarpShadow();
        }

        private void CalculateAndWarpShadow()
        {
            this.currentshadowpolys.Clear();
            this.currenthardshadowpolys.Clear();
            this.currentsoftshadowpolys.Clear();
            this.currentshadowpolygonarray.Clear();
            foreach (PolyProxy proxy in this.polyproxies)
            {
                //Vector3d[] poly, hardpoly, softpoly;
                //CalculateProxyShadowPolygonsNew(proxy, out poly, out hardpoly, out softpoly);
                //this.currentshadowpolys.Add(poly);
                //this.currenthardshadowpolys.Add(hardpoly);
                //this.currentsoftshadowpolys.Add(softpoly);

                CalculateProxyShadowPolygonsAll(proxy);
            }
            //this.ExtendShadowPolygonInnerside();
            //this.GenerateFakeShadowsOnFloor();

            this.GenerateFakeShadowsOnFloorNew();

		//	this.ShowCurrentCompositionResults();
        }

        Image<Gray, byte> shadowlayer = null;

        private void GenerateFakeShadowsOnFloorNew()
        {
            double scale = Math.Pow(this.Lightness, 1.0 / Math.Pow(this.LightSamplingNumber, 1));

            this.shadowlayer = new Image<Gray, byte>(this.image.Size);
            this.shadowlayer.SetValue(new Gray(255));

            for (int k = 0; k < this.polyproxies.Count; ++k)
            {
                Vector2d[][] polygons = this.currentshadowpolygonarray[k]
                    .Select(pts3d => pts3d
                        .Select(p => p.ToVector2d())
                        .ToArray())
                    .ToArray();

                foreach (Vector2d[] polygon in polygons)
                {
                    Vector2d[] polygonimgpos = polygon
                        .Select(p => this.cameracalibrator.ComputePointImgPos(new Vector3d(p)))
                        .ToArray();

                    int xmin, xmax, ymin, ymax;
                    xmin = ymin = int.MaxValue;
                    xmax = ymax = int.MinValue;
                    for (int m = 0; m < polygonimgpos.Length; ++m)
                    {
                        xmin = Math.Min(xmin, (int)Math.Floor(polygonimgpos[m].x));
                        ymin = Math.Min(ymin, (int)Math.Floor(polygonimgpos[m].y));
                        xmax = Math.Max(xmax, (int)Math.Ceiling(polygonimgpos[m].x));
                        ymax = Math.Max(ymax, (int)Math.Ceiling(polygonimgpos[m].y));
                    }
                    xmin = Math.Max(xmin, 0);
                    ymin = Math.Max(ymin, 0);
                    xmax = Math.Min(xmax, this.imgwidth - 1);
                    ymax = Math.Min(ymax, this.imgheight - 1);

                    if (xmax <= xmin || ymax <= ymin)
                    continue;

                    for (int y = ymin; y < ymax; ++y)
                    {
                        for (int x = xmin; x < xmax; ++x)
                        {
                            Vector3d p3 = this.cameracalibrator.ComputePointGround3dPos(new Vector2d(x, y));
                            Vector2d p2 = p3.ToVector2d();
                            if (Shape2D.PointInPoly(p2, polygon))
                            {
                                //Bgr c = resultimage[y, x];
                                //resultimage[y, x] = new Bgr(c.Blue * scale, c.Green * scale, c.Red * scale);

                                this.shadowlayer[y, x] = new Gray(this.shadowlayer[y, x].Intensity * scale);
                            }
                        }
                    }

                    //xmin = Math.Max(xmin - 2, 0);
                    //ymin = Math.Max(ymin - 2, 0);
                    //xmax = Math.Min(xmax + 2, this.imgwidth - 1);
                    //ymax = Math.Min(ymax + 2, this.imgheight - 1);
                    //Rectangle rect = new Rectangle(xmin, ymin, xmax - xmin, ymax - ymin);
                    //List<Vector2d[]> polys = new List<Vector2d[]>();
                    //polys.Add(polygon);
                    //this.PolygonBlur(this.resultimage, polys.ToArray(), rect, 2);
                }
            }

            //this.resultimage = this.resultimage.SmoothGaussian(9);
            this.shadowlayer = this.shadowlayer.SmoothGaussian(3);
            for (int i = 0; i < this.resultimage.Rows; ++i)
            {
                for (int j = 0; j < this.resultimage.Cols; ++j)
                {
                    double dark = this.shadowlayer[i, j].Intensity / 255.0;
                    Bgr color = this.resultimage[i, j];
                    color.Blue *= dark;
                    color.Green *= dark;
                    color.Red *= dark;
                    this.resultimage[i, j] = color;
                }
            }

            this.CreateGLTxture(this.resultimage, layertxtid);
        }

        private void CalculateProxyShadowPolygonsAll(PolyProxy proxy)
        {
            double[] heights = this.polyproxies
                 .Select(pro => pro.zlen)
                 .OrderBy(h => h)
                 .ToArray();

            List<Vector3d[]> polygons = new List<Vector3d[]>();

            Vector3d lightpos = this.optimlightposition;
            polygons.Add(this.CalculateProxyShadowNew(proxy, lightpos));

            Vector3d offsetlp1 = lightpos + new Vector3d(0, 0, 5 * heights[0]);
            polygons.Add(this.CalculateProxyShadowNew(proxy, offsetlp1));

            Vector3d[] lightpositions = GetLightpositionSamplings(proxy.center.pos, lightpos);
            polygons.AddRange(this.CalculateProxyShadowArray(proxy, lightpositions));

            this.currentshadowpolygonarray.Add(polygons.ToArray());
        }

        private void PixelBlur(Image<Bgr, byte> src, Image<Bgr, byte> dest, int tx, int ty, int xoff, int yoff, int rad)
        {
            int x = tx;
            int y = ty;
            Bgr total = new Bgr(0, 0, 0);
            int sum = 0;
            for (int ky = -rad; ky <= rad; ++ky)
            {
                int srcy = y + ky - yoff;
                if (srcy < 0 || srcy >= src.Height)
                    continue;
                for (int kx = -rad; kx <= rad; ++kx)
                {
                    int srcx = x + kx - xoff;
                    if (srcx < 0 || srcx >= src.Width)
                        continue;
                    total.Blue += src[srcy, srcx].Blue;
                    total.Green += src[srcy, srcx].Green;
                    total.Red += src[srcy, srcx].Red;
                    sum++;
                }
            }
            total.Blue /= sum;
            total.Green /= sum;
            total.Red /= sum;
            dest[y, x] = total;
        }

        private void PolygonBlur(Image<Bgr, byte> img, Vector2d[][] polys, Rectangle rect, int blurrad)
        {
            PointF[][] impolys = polys
                .Select(poly => poly
                    .Select(p => {
                        Vector2d p2 = this.cameracalibrator.ComputePointImgPos(new Vector3d(p));
                        return new PointF((float)p2.x, (float)p2.y);
                    })
                    .ToArray())
                .ToArray();

            int xmin = rect.Left;
            int xmax = rect.Right;
            int ymin = rect.Top;
            int ymax = rect.Bottom;
            SizeF tempoff = new SizeF(rect.X, rect.Y);
            
            Image<Gray, byte> temp = new Image<Gray, byte>(rect.Width, rect.Height);
            foreach (PointF[] impoly in impolys)
            {
                for (int i = 0; i < impoly.Length; ++i)
                {
                    PointF curr = impoly[i] - tempoff;
                    PointF next = impoly[(i + 1) % impoly.Length] - tempoff;
                    temp.Draw(new LineSegment2DF(curr, next), new Gray(255), 1);
                }
            }
            //ImageViewer viewer = new ImageViewer(temp);
            //viewer.AutoSize = true;
            //viewer.Show();

            int[] xoff = new int[] { -1, 0, 0, 0, 1 };
            int[] yoff = new int[] { 0, -1, 0, 1, 0 };

            Image<Bgr, byte> src = img.Copy(rect);
            for (int y = ymin; y < ymax; ++y)
            {
                int yy = y - ymin;
                for (int x = xmin; x < xmax; ++x)
                {
                    int xx = x - xmin;
                    bool isblurneeded = false;
                    for (int k = 0; k < xoff.Length; ++k)
                    {
                        int tempy = yy + yoff[k];
                        int tempx = xx + xoff[k];
                        if (tempy >= 0 && tempy < temp.Height &&
                            tempx >= 0 && tempx < temp.Width &&
                            temp[tempy, tempx].Intensity > 0)
                        {
                            isblurneeded = true;
                            break;
                        }
                    }
                    if (isblurneeded)
                        this.PixelBlur(src, img, x, y, xmin, ymin, blurrad);
                }
            }
        }

        private void GenerateFakeShadowsOnFloor()
        {
            Vector2d[][] currentshadowpoly2d = this.enlargedcurrentshadowpolys
                .Select(pts3d => pts3d
                    .Select(p => p.ToVector2d())
                    .ToArray())
                .ToArray();
            Vector2d[][] currenthardshadowpoly2d = this.enlargedcurrenthardshadowpolys
                .Select(pts3d => pts3d
                    .Select(p => p.ToVector2d())
                    .ToArray())
                .ToArray();
            Vector2d[][] currentsoftshadowpoly2d = this.enlargedcurrentsoftshadowpolys
                .Select(pts3d => pts3d
                    .Select(p => p.ToVector2d())
                    .ToArray())
                .ToArray();

            for (int k = 0; k < this.polyproxies.Count; ++k)
            {
                Vector2d[] poly = currentshadowpoly2d[k];
                Vector2d[] hardpoly = currenthardshadowpoly2d[k];
                Vector2d[] softpoly = currentsoftshadowpoly2d[k];

                Vector2d polycenter = Vector2d.Zero;
                foreach (Vector2d p2 in poly)
                    polycenter += p2;
                polycenter /= poly.Length;
                Vector2d hardpolycenter = Vector2d.Zero;
                foreach (Vector2d p2 in hardpoly)
                    hardpolycenter += p2;
                hardpolycenter /= hardpoly.Length;
                Vector2d softpolycenter = Vector2d.Zero;
                foreach (Vector2d p2 in softpoly)
                    softpolycenter += p2;
                softpolycenter /= softpoly.Length;

                List<Vector2d> allpolypts = new List<Vector2d>();
                allpolypts.AddRange(softpoly);
                Vector2d[] allpolyimgpos = allpolypts
                    .Select(p => this.cameracalibrator.ComputePointImgPos(new Vector3d(p)))
                    .ToArray();
                int xmin, xmax, ymin, ymax;
                xmin = ymin = int.MaxValue;
                xmax = ymax = int.MinValue;
                for (int m = 0; m < allpolyimgpos.Length; ++m)
                {
                    xmin = Math.Min(xmin, (int)Math.Floor(allpolyimgpos[m].x));
                    ymin = Math.Min(ymin, (int)Math.Floor(allpolyimgpos[m].y));
                    xmax = Math.Max(xmax, (int)Math.Ceiling(allpolyimgpos[m].x));
                    ymax = Math.Max(ymax, (int)Math.Ceiling(allpolyimgpos[m].y));
                }
                xmin = Math.Max(xmin, 0);
                ymin = Math.Max(ymin, 0);
                xmax = Math.Min(xmax, this.imgwidth - 1);
                ymax = Math.Min(ymax, this.imgheight - 1);

                if (xmax <= xmin || ymax <= ymin)
                    continue;

                for (int i = ymin; i < ymax; ++i)
                {
                    for (int j = xmin; j < xmax; ++j)
                    {
                        // shadow first, then textures now...

                        //double darkness = 0.4;
                        double lightness = this.Lightness;
                        if (this.binaryshadow != null)
                            lightness = this.lightnesses[k];// * 0.6;
                        double darkness = 1 - lightness;
                        double hardboundary = darkness * 0.8;
                        double sigma1 = 2, sigma2 = 0.4;
                        double polyboundary = hardboundary * Math.Pow(Math.E, -1 / sigma1);

                        Vector3d p3 = this.cameracalibrator.ComputePointGround3dPos(new Vector2d(j, i));
                        Vector2d p2 = p3.ToVector2d();

                        // outer region of shadow.
                        if (Shape2D.PointInPoly(p2, softpoly))
                        {
                            double scale = 1;
                            if (Shape2D.PointInPoly(p2, poly))
                            {
                                if (Shape2D.PointInPoly(p2, hardpoly))
                                {
                                    double dist1 = (p2 - hardpolycenter).Length();
                                    double dist2 = Shape2D.Point2PolyBoundaryDist(p2, hardpoly);
                                    double lambda = dist1 / (dist1 + dist2);
                                    scale = lambda * hardboundary + (1 - lambda) * darkness;
                                }
                                else
                                {
                                    double dist1 = Shape2D.Point2PolyBoundaryDist(p2, hardpoly);
                                    double dist2 = Shape2D.Point2PolyBoundaryDist(p2, poly);
                                    double dist = dist1 / (dist1 + dist2);
                                    scale = hardboundary * Math.Pow(Math.E, -dist / sigma1);
                                }
                            }
                            else
                            {
                                double dist1 = Shape2D.Point2PolyBoundaryDist(p2, poly);
                                double dist2 = Shape2D.Point2PolyBoundaryDist(p2, softpoly);
                                double dist = dist1 / (dist1 + dist2);
                                scale = polyboundary * Math.Pow(Math.E, -dist / sigma2);
                            }
                            scale = 1 - scale;
                            Bgr c = resultimage[i, j];
                            resultimage[i, j] = new Bgr(c.Blue * scale, c.Green * scale, c.Red * scale);
                        }
                    }
                }

                xmin = Math.Max(xmin - 2, 0);
                ymin = Math.Max(ymin - 2, 0);
                xmax = Math.Min(xmax + 2, this.imgwidth - 1);
                ymax = Math.Min(ymax + 2, this.imgheight - 1);
                Rectangle rect = new Rectangle(xmin, ymin, xmax - xmin, ymax - ymin);
                List<Vector2d[]> polys = new List<Vector2d[]>();
                polys.Add(poly);
                polys.Add(hardpoly);
                polys.Add(softpoly);
                this.PolygonBlur(this.resultimage, polys.ToArray(), rect, 1);
            }

            this.CreateGLTxture(this.resultimage, layertxtid);
        }

        private Dictionary<int, PolyProxy> GetIdProxies()
        {
            Dictionary<int, PolyProxy> result = new Dictionary<int, PolyProxy>();
            foreach (PolyProxy proxy in this.polyproxies)
                foreach (Polygon poly in proxy.polygons)
                    result.Add(this.polyids[poly], proxy);
            return result;
        }

        private void CalculateProxyShadows(PolyProxy proxy, out Vector3d[] currentpoly, out Vector3d[] hardpoly, out Vector3d[] softpoly)
        {
            double[] heights = this.polyproxies
                .Select(pro => pro.zlen)
                .OrderBy(h => h)
                .ToArray();

            Vector3d lightpos = this.optimlightposition;
            Vector3d offsetlp1 = lightpos + new Vector3d(0, 0, 5 * heights[0]);
            Vector3d offsetlp2 = lightpos - new Vector3d(0, 0, Math.Max(heights[heights.Length - 1] + 1e-3, 2 * heights[0]));
            currentpoly = this.CalculateProxyShadow(proxy, lightpos);
            hardpoly = this.CalculateProxyShadow(proxy, offsetlp1);
            softpoly = this.CalculateProxyShadow(proxy, offsetlp2);

            Vector2d p1 = currentpoly[1].ToVector2d();
            Vector2d p2 = currentpoly[2].ToVector2d();
            Vector2d pn1 = currentpoly[currentpoly.Length - 3].ToVector2d();
            Vector2d pn2 = currentpoly[currentpoly.Length - 4].ToVector2d();

            // adjust hardpoly
            double dist1 = Distance2Line(hardpoly[1].ToVector2d(), p1, p2);
            Vector2d dir1 = (p2 - p1).Normalize().PerpendicularLeft;
            hardpoly[1] = new Vector3d(p1 + dir1 * dist1);
            double dist2 = Distance2Line(hardpoly[hardpoly.Length - 3].ToVector2d(), pn1, pn2);
            Vector2d dir2 = (pn2 - pn1).Normalize().PerpendicularRight;
            hardpoly[hardpoly.Length - 3] = new Vector3d(pn1 + dir2 * dist2);

            // adjust softpoly
            double dist3 = Distance2Line(softpoly[1].ToVector2d(), p1, p2);
            Vector2d dir3 = (p2 - p1).Normalize().PerpendicularRight;
            softpoly[1] = new Vector3d(p1 + dir3 * dist3);
            double dist4 = Distance2Line(softpoly[softpoly.Length - 3].ToVector2d(), pn1, pn2);
            Vector2d dir4 = (pn2 - pn1).Normalize().PerpendicularLeft;
            softpoly[softpoly.Length - 3] = new Vector3d(pn1 + dir4 * dist4);
        }

        private void CalculateProxyShadowPolygonsNew(PolyProxy proxy, out Vector3d[] currentpoly, out Vector3d[] hardpoly, out Vector3d[] softpoly)
        {
            double[] heights = this.polyproxies
                .Select(pro => pro.zlen)
                .OrderBy(h => h)
                .ToArray();

            Vector3d lightpos = this.optimlightposition;
            currentpoly = this.CalculateProxyShadowNew(proxy, lightpos);

            Vector3d offsetlp1 = lightpos + new Vector3d(0, 0, 5 * heights[0]);
            hardpoly = this.CalculateProxyShadowNew(proxy, offsetlp1);

            Vector3d[] lightpositions = GetLightpositionSamplings(proxy.center.pos, lightpos);
            //Vector3d[] a = GetLightpositionSamplings(heights, lightpos);
            softpoly = this.CalculateProxyShadowNew(proxy, lightpositions);
        }

        private Vector3d[] GetLightpositionSamplings(Vector3d target, Vector3d lightpos)
        {
            List<Vector3d> lightpositions = new List<Vector3d>();
            Vector3d dir = (target - lightpos).Normalize();
            Vector3d xaxis = new Vector3d(dir.ToVector2d().PerpendicularLeft, 0);
            xaxis = xaxis.Normalize();
            Vector3d yaxis = dir.Cross(xaxis).Normalize();

            double astep = 2 * Math.PI / this.LightSamplingNumber;
            //double rstep = this.LightSize / this.LightSamplingNumber;
            //double radius = 0;
            //for (int k = 0; k < this.LightSamplingNumber; ++k)
            //{
                double angle = 0;
                for (int i = 0; i < this.LightSamplingNumber; ++i)
                {
                    Vector3d posdir = Math.Cos(angle) * xaxis + Math.Sin(angle) * yaxis;
                    lightpositions.Add(lightpos + this.LightSize * posdir);
                    angle += astep;
                }
            //    radius += rstep;
            //}

            return lightpositions.ToArray();
        }

        private Vector3d[] GetLightpositionSamplings(double[] heights, Vector3d lightpos)
        {
            List<Vector3d> lightpositions = new List<Vector3d>();
            double height = Math.Max(heights[heights.Length - 1] + 1e-3, lightpos.z - 2 * heights[0]);
            lightpositions.Add(new Vector3d(lightpos.x, lightpos.y, height));
            lightpositions.Add(lightpos - new Vector3d(LightSize, 0, 0));
            lightpositions.Add(lightpos + new Vector3d(LightSize, 0, 0));
            lightpositions.Add(lightpos - new Vector3d(0, LightSize, 0));
            lightpositions.Add(lightpos + new Vector3d(0, LightSize, 0));
            return lightpositions.ToArray();
        }
        private Vector3d[] CalculateProxyShadowNew(PolyProxy proxy, Vector3d[] lightpositions)
        {
            using (MemStorage storage = new MemStorage())
            {
                Seq<PointF> projections = new Seq<PointF>(storage);
                foreach (Vector3d lightpos in lightpositions)
                {
                    double H = lightpos.z;
                    Vector2d lgp = lightpos.ToVector2d();
                    foreach (Point3 p3 in proxy.points3)
                    {
                        Vector2d pground = p3.pos.ToVector2d();
                        double height = p3.pos.z;
                        double len = height * (pground - lgp).Length() / (H - height);
                        Vector2d pprojection = (pground - lgp).Normalize() * len + pground;
                        projections.Push(Vector2PointF(pprojection));
                    }
                }
                Seq<PointF> poly = projections.GetConvexHull(Emgu.CV.CvEnum.ORIENTATION.CV_COUNTER_CLOCKWISE, storage);
                return poly.Select(pf => new Vector3d(PointF2Vecter2d(pf))).ToArray();
            }
        }
        private Vector3d[][] CalculateProxyShadowArray(PolyProxy proxy, Vector3d[] lightpositions)
        {
            List<Vector3d[]> polygons = new List<Vector3d[]>();
            foreach (Vector3d lightpos in lightpositions)
            {
                double H = lightpos.z;
                Vector2d lgp = lightpos.ToVector2d();
                polygons.Add(CalculateProxyShadowNew(proxy, lightpos));
            }
            return polygons.ToArray();
        }
        private Vector3d[] CalculateProxyShadowNew(PolyProxy proxy, Vector3d lightpos)
        {
            double H = lightpos.z;
            Vector2d lgp = lightpos.ToVector2d();

            using (MemStorage storage = new MemStorage())
            {
                Seq<PointF> projections = new Seq<PointF>(storage);
                foreach (Point3 p3 in proxy.points3)
                {
                    Vector2d pground = p3.pos.ToVector2d();
                    double height = p3.pos.z;
                    double len = height * (pground - lgp).Length() / (H - height);
                    Vector2d pprojection = (pground - lgp).Normalize() * len + pground;
                    projections.Push(Vector2PointF(pprojection));
                }
                Seq<PointF> poly = projections.GetConvexHull(Emgu.CV.CvEnum.ORIENTATION.CV_COUNTER_CLOCKWISE, storage);
                return poly.Select(pf => new Vector3d(PointF2Vecter2d(pf))).ToArray();
            }
        }
        private Vector3d[] CalculateProxyShadowNew(PolyProxy proxy, Vector3d lightpos, out Vector3d[] srcpoints)
        {
            double H = lightpos.z;
            Vector2d lgp = lightpos.ToVector2d();

            using (MemStorage storage = new MemStorage())
            {
                Seq<PointF> projections = new Seq<PointF>(storage);
                foreach (Point3 p3 in proxy.points3)
                {
                    Vector2d pground = p3.pos.ToVector2d();
                    double height = p3.pos.z;
                    double len = height * (pground - lgp).Length() / (H - height);
                    Vector2d pprojection = (pground - lgp).Normalize() * len + pground;
                    projections.Push(Vector2PointF(pprojection));
                }
                Seq<PointF> poly = projections.GetConvexHull(Emgu.CV.CvEnum.ORIENTATION.CV_COUNTER_CLOCKWISE, storage);
                srcpoints = new Vector3d[poly.Total];
                int index = 0;
                foreach (PointF p3 in poly)
                {
                    double mindist = double.MaxValue;
                    int mindistindex = -1;
                    for (int i = 0; i < projections.Total; ++i)
                    {
                        PointF pp3 = projections[i];
                        double dist = (PointF2Vecter2d(p3) - PointF2Vecter2d(pp3)).Length();
                        if (dist < mindist)
                        {
                            mindist = dist;
                            mindistindex = i;
                        }
                    }
                    srcpoints[index++] = proxy.points3[mindistindex].pos;
                }

                Vector3d[] shadowpolygon = poly.Select(pf => new Vector3d(PointF2Vecter2d(pf))).ToArray();
                return shadowpolygon;
            }
        }
        private Vector3d[] CalculateProxyShadow(PolyProxy proxy, Vector3d lightpos)
        {
            double H = lightpos.z;
            Vector2d lgp = lightpos.ToVector2d();

            //double height = proxy.zlen;
            //Polygon bottom = proxy.polygons[0];
            Polygon top = proxy.polygons.Find(p => Math.Abs(p.normal.Dot(Vector3d.Zaxis) - 1) < 1e-4);
            Polygon bottom = proxy.polygons.Find(p => Math.Abs(p.normal.Dot(Vector3d.Zaxis) + 1) < 1e-4);
            double height = (top.center.pos - bottom.center.pos).z;
            Vector2d[] btmpts2d = bottom.points3.Select(p => p.pos.ToVector2d()).ToArray();
            Vector2d btmcenter = (btmpts2d[0] + btmpts2d[2]) / 2;
            int index = -1;
            double mindot = double.MaxValue;
            for (int j = 0; j < btmpts2d.Length; ++j)
            {
                double dot = ((lgp - btmcenter).Normalize()).Dot((btmpts2d[j] - btmcenter).Normalize());
                if (dot < mindot)
                {
                    mindot = dot;
                    index = j;
                }
            }

            int prev = (index - 1 + btmpts2d.Length) % btmpts2d.Length;
            int next = (index + 1) % btmpts2d.Length;

            if ((btmpts2d[index] - btmpts2d[prev]).Cross(btmpts2d[next] - btmpts2d[index]) < 0)
            {
                int temp = prev;
                prev = next;
                next = temp;
            }

            Vector2d c1 = btmpts2d[prev];
            Vector2d c2 = btmpts2d[index];
            Vector2d c3 = btmpts2d[next];

            double len1 = height * (c1 - lgp).Length() / (H - height);
            double len2 = height * (c2 - lgp).Length() / (H - height);
            double len3 = height * (c3 - lgp).Length() / (H - height);

            Vector2d p1 = (c1 - lgp).Normalize() * len1 + c1;
            Vector2d p2 = (c2 - lgp).Normalize() * len2 + c2;
            Vector2d p3 = (c3 - lgp).Normalize() * len3 + c3;

            Vector2d[] shadowpoly = new Vector2d[] { c1, p1, p2, p3, c3, c2 };
            return shadowpoly.Select(p => new Vector3d(p)).ToArray();
        }

        private void CalculateShadowWarppingData()
        {
            this.scalemap = new double[this.imgheight, this.imgwidth];
            Vector2d[][] optimshadowpoly2d = optimshadowpolys
                .Select(pts3d => pts3d
                    .Select(p => p.ToVector2d())
                    .ToArray())
                .ToArray();

            if (this.binaryshadow != null)
            {
                int shadownum = optimshadowpoly2d.GetLength(0);
                this.lightnesses = new double[shadownum];
                int[] count = new int[shadownum];
                for (int i = 0; i < this.imgheight; ++i)
                {
                    for (int j = 0; j < this.imgwidth; ++j)
                    {
                        double intensity1 = GetColorIntensity(this.image[i, j]);
                        double intensity2 = GetColorIntensity(this.imlayer[i, j]);
                        this.scalemap[i, j] = intensity1 / intensity2;
                        if (this.binaryshadow[i, j].Intensity > 0)
                        {
                            for (int k = 0; k < shadownum; ++k)
                            {
                                Vector2d[] poly = optimshadowpoly2d[k];
                                Vector3d p3 = this.cameracalibrator.ComputePointGround3dPos(new Vector2d(j, i));
                                Vector2d p2 = p3.ToVector2d();
                                if (Shape2D.PointInPoly(p2, poly))
                                {
                                    this.lightnesses[k] += this.scalemap[i, j];
                                    count[k]++;
                                    break;
                                }
                            }
                        }
                    }
                }
                double meandarkness = 0;
                int meancount = 0;
                for (int k = 0; k < shadownum; ++k)
                {
                    if (count[k] > 0)
                    {
                        this.lightnesses[k] /= count[k];
                        meandarkness += this.lightnesses[k];
                        meancount++;
                    }
                }
                meandarkness /= meancount;
                for (int k = 0; k < shadownum; ++k)
                    if (count[k] == 0)
                        this.lightnesses[k] = meandarkness;
            }

            ExtendShadowPolygon();
        }

        private void ExtendShadowPolygonInnerside()
        {
            Vector2d[][] currentshadowpoly2d = this.currentshadowpolys
                .Select(pts3d => pts3d
                    .Select(p => p.ToVector2d())
                    .ToArray())
                .ToArray();
            Vector2d[][] currenthardshadowpoly2d = this.currenthardshadowpolys
                .Select(pts3d => pts3d
                    .Select(p => p.ToVector2d())
                    .ToArray())
                .ToArray();
            Vector2d[][] currentsoftshadowpoly2d = this.currentsoftshadowpolys
                .Select(pts3d => pts3d
                    .Select(p => p.ToVector2d())
                    .ToArray())
                .ToArray();

            this.enlargedcurrentshadowpolys.Clear();
            this.enlargedcurrenthardshadowpolys.Clear();
            this.enlargedcurrentsoftshadowpolys.Clear();
            Vector2d[][] enlargedshadowpoly2d = new Vector2d[currentshadowpoly2d.GetLength(0)][];
            Vector2d[][] enlargedhardshadowpoly2d = new Vector2d[currenthardshadowpoly2d.GetLength(0)][];
            Vector2d[][] enlargedsoftshadowpoly2d = new Vector2d[currentsoftshadowpoly2d.GetLength(0)][];
            for (int k = 0; k < currentshadowpoly2d.GetLength(0); ++k)
            {
                Vector2d[] poly = currentshadowpoly2d[k];
                enlargedshadowpoly2d[k] = (Vector2d[])poly.Clone();
                enlargedhardshadowpoly2d[k] = (Vector2d[])currenthardshadowpoly2d[k].Clone();
                enlargedsoftshadowpoly2d[k] = (Vector2d[])currentsoftshadowpoly2d[k].Clone();
                //double maxlen = double.MinValue;
                //for (int i = 0, j = poly.Length - 1; i < poly.Length; ++i, j = i - 1)
                //{
                //    double len = (poly[i] - poly[j]).Length();
                //    maxlen = Math.Max(maxlen, len);
                //}
                //double offset = maxlen / 200;
                //for (int i = 0; i < poly.Length; ++i)
                //{
                //    if (i >= 1 && i <= 3) continue;
                //    int prev = (i + poly.Length - 1) % poly.Length;
                //    Vector2d dir1 = (poly[i] - poly[prev]).Normalize();
                //    Vector2d offsetdir1 = dir1.PerpendicularRight;
                //    if (i == 4) offsetdir1 = Vector2d.Zero;
                //    Vector2d lp1 = poly[i] + offsetdir1 * offset;
                //    int next = (i + 1) % poly.Length;
                //    Vector2d dir2 = (poly[next] - poly[i]).Normalize();
                //    Vector2d offsetdir2 = dir2.PerpendicularRight;
                //    if (i == 0) offsetdir2 = Vector2d.Zero;
                //    Vector2d lp2 = poly[i] + offsetdir2 * offset;
                //    Vector2d interpt;
                //    IntersectionHelper.LinesIntersection(lp1, lp1 + dir1, lp2, lp2 + dir2, out interpt);
                //    enlargedshadowpoly2d[k][i] = interpt;
                //    enlargedhardshadowpoly2d[k][i] = interpt;
                //    enlargedsoftshadowpoly2d[k][i] = interpt;
                //}
                this.enlargedcurrentshadowpolys.Add(enlargedshadowpoly2d[k]
                    .Select(p => new Vector3d(p))
                    .ToArray());
                this.enlargedcurrenthardshadowpolys.Add(enlargedhardshadowpoly2d[k]
                    .Select(p => new Vector3d(p))
                    .ToArray());
                this.enlargedcurrentsoftshadowpolys.Add(enlargedsoftshadowpoly2d[k]
                    .Select(p => new Vector3d(p))
                    .ToArray());
            }
        }

        private void ExtendShadowPolygon()
        {
            Vector2d[][] optimshadowpoly2d = this.optimshadowpolys
                .Select(pts3d => pts3d
                    .Select(p => p.ToVector2d())
                    .ToArray())
                .ToArray();

            this.enlargedoptimshadowpolys.Clear();
            Vector2d[][] enlargedshadowpoly2d = new Vector2d[optimshadowpoly2d.GetLength(0)][];
            for (int k = 0; k < optimshadowpoly2d.GetLength(0); ++k)
            {
                Vector2d[] poly = optimshadowpoly2d[k];
                double maxlen = double.MinValue;
                for (int i = 0, j = poly.Length - 1; i < poly.Length; ++i, j = i - 1)
                {
                    double len = (poly[i] - poly[j]).Length();
                    maxlen = Math.Max(maxlen, len);
                }
                double offset = maxlen / 40;
                enlargedshadowpoly2d[k] = (Vector2d[])poly.Clone();
                for (int i = 1; i < poly.Length - 2; ++i)
                {
                    int prev = (i + poly.Length - 1) % poly.Length;
                    Vector2d dir1 = (poly[i] - poly[prev]).Normalize();
                    Vector2d offsetdir1 = dir1.PerpendicularRight;
                    if (i == 1) offsetdir1 = Vector2d.Zero;
                    Vector2d lp1 = poly[i] + offsetdir1 * offset;
                    int next = (i + 1) % poly.Length;
                    Vector2d dir2 = (poly[next] - poly[i]).Normalize();
                    Vector2d offsetdir2 = dir2.PerpendicularRight;
                    if (i == 3) offsetdir2 = Vector2d.Zero;
                    Vector2d lp2 = poly[i] + offsetdir2 * offset;
                    Vector2d interpt;
                    IntersectionHelper.LinesIntersection(lp1, lp1 + dir1, lp2, lp2 + dir2, out interpt);
                    enlargedshadowpoly2d[k][i] = interpt;
                }
                this.enlargedoptimshadowpolys.Add(enlargedshadowpoly2d[k]
                    .Select(p => new Vector3d(p))
                    .ToArray());
            }
        }
        private int FindNewPixelLabel(int y, int x, Polygon[] allpolygons)
        {
            Vector2d p = new Vector2d(x, y);
            // TODO:
            int label = -1;
            double deepness = double.MaxValue;
            foreach (Polygon poly in allpolygons)
            {
                //Vector2d[] polypoints = poly.points3
                //    .Select(p3 => this.GLUProject(p3.pos).ToVector2d())
                //    .ToArray();
                if (Shape2D.PointInPoly(p, poly))
                {
                    // calculate 3d points.
                    // calculate deepness.
                    Vector3d p3 = this.ProjectScreenPoint2PolyFace(p, poly);
                    p3 = this.GLUProject(p3);
                    if (p3.z < deepness)
                    {
                        deepness = p3.z;
                        //label = poly.textureindex;
                        label = this.polyids[poly];
                    }
                }
            }
            return label;
        }
        private int[,] UpdatePixelLabels(Polygon[] allpolygons)
        {
            int[,] result = new int[this.imgheight, this.imgwidth];
            for (int i = 0; i < this.imgheight; ++i)
                for (int j = 0; j < this.imgwidth; ++j)
                    result[i, j] = this.FindNewPixelLabel(i, j, allpolygons);
            return result;
        }
        private void VisualizeLabelmap()
        {
            Image<Bgr, byte> tmp = GenerateCVImageFromMatrix(this.labelmap);
            this.VisualizeCVImage(tmp, "labelmap");
        }
		private void VisualizeLabelmap(int[,] labelmap, string name)
		{
			int hei = labelmap.GetLength(0);
			int wid = labelmap.GetLength(1);
			Image<Bgr, byte> tmp = new Image<Bgr, byte>(wid, hei);
			int n = this.colorMall.Length;
			for (int i = 0; i < hei; ++i)
			{
				for (int j = 0; j < wid; ++j)
				{
					int lb = labelmap[i, j];
					tmp[i, j] = new Bgr(this.colorMall[(lb + n) % n]);
				}
			}
			ImageViewer view = new ImageViewer(tmp, name);
			view.ClientSize = tmp.Size;
			view.Show();
		}

        private Image<Bgr, byte> GenerateCVImageFromMatrix(int[,] matrix)
        {
            if (matrix == null)
                return null;
            int rows = matrix.GetLength(0);
            int cols = matrix.GetLength(1);
            Image<Bgr, byte> tmp = new Image<Bgr, byte>(cols, rows);
            int n = this.colorMall.Length;
            for (int i = 0; i < rows; ++i)
            {
                for (int j = 0; j < cols; ++j)
                {
                    int lb = matrix[i, j];
                    tmp[i, j] = new Bgr(this.colorMall[(lb + n) % n]);
                }
            }
            return tmp;
        }

        private void GeneratePolygonIds()
        {
            this.polyids.Clear();
            this.idpolys.Clear();

            int id = 0;
            foreach (PolyProxy proxy in this.polyproxies)
            {
                foreach (Polygon poly in proxy.polygons)
                {
                    this.polyids.Add(poly, id);
                    this.idpolys.Add(id, poly);
                    id++;
                }
            }
        }
        public void GenerateLabelmapandMask()
        {
            Image<Gray, byte> labelmap = new Image<Gray, byte>(this.image.Size);
            Image<Gray, byte> mask = new Image<Gray, byte>(this.image.Size);

            this.GeneratePolygonIds();

            List<Polygon> allpolyList = new List<Polygon>();
            //allpolyList.Add(this.roomlayout.box.polygons[0]);
            //allpolyList.Add(this.roomlayout.box.polygons[2]);
            //allpolyList.Add(this.roomlayout.box.polygons[3]);
            foreach (PolyProxy proxy in this.polyproxies)
                allpolyList.AddRange(proxy.polygons);
            Polygon[] allpolygons = allpolyList.ToArray();
            int[,] labels = this.UpdatePixelLabels(allpolygons);

            List<PolyProxy> deletedproxies = new List<PolyProxy>();
            //deletedproxies.Add(this.polyproxies[5]);
            //deletedproxies.Add(this.polyproxies[6]);
            //deletedproxies.Add(this.polyproxies[1]);
            //deletedproxies.Add(this.polyproxies[2]);
            deletedproxies.AddRange(this.polyproxies);

            List<int> deletedlabels = new List<int>();
            foreach (PolyProxy proxy in deletedproxies)
                foreach (Polygon poly in proxy.polygons)
                    deletedlabels.Add(this.polyids[poly]);

            for (int i = 0; i < this.imagelabel.GetLength(0); ++i)
            {
                for (int j = 0; j < this.imagelabel.GetLength(1); ++j)
                {
                    int label = labels[i, j];
                    if (deletedlabels.Contains(label))
                    {
                        mask[i, j] = new Gray(255);
                    }
                }
            }

            allpolyList.Clear();
            //allpolyList.Add(this.roomlayout.box.polygons[0]);
            //allpolyList.Add(this.roomlayout.box.polygons[2]);
            //allpolyList.Add(this.roomlayout.box.polygons[3]);
            foreach (PolyProxy proxy in this.polyproxies)
                if (!deletedproxies.Contains(proxy))
                    allpolyList.AddRange(proxy.polygons);
            allpolygons = allpolyList.ToArray();
            labels = this.UpdatePixelLabels(allpolygons);

            for (int i = 0; i < this.imagelabel.GetLength(0); ++i)
            {
                for (int j = 0; j < this.imagelabel.GetLength(1); ++j)
                {
                    labelmap[i, j] = new Gray(labels[i, j]);
                }
            }

            string dirname = Path.GetDirectoryName(this.imagename);
            string filename = Path.GetFileNameWithoutExtension(this.imagename);
            string labelmapname = Path.Combine(dirname, filename + "_labelmap.png");
            string maskname = Path.Combine(dirname, filename + "_mask.png");
            labelmap.Save(labelmapname);
            Program.OutputText("Save labelmap to " + labelmapname, true);
            mask.Save(maskname);
            Program.OutputText("Save mask to " + maskname, true);

            foreach (Polygon poly in allpolygons)
            {
                poly.H = null;
                poly.localframe = null;
            }
        }

        private Vector3d[] CalculateProxyShadow(PolyProxy proxy, Vector3d[] corners)
        {
            if (corners.Length != 2)
                return null;

            Vector2d farcorner = corners[0].ToVector2d();
            Vector2d nearcorner = corners[1].ToVector2d();

            Polygon bottom = proxy.polygons[0];
            Vector2d[] btmpts2d = bottom.points3.Select(p => p.pos.ToVector2d()).ToArray();
            Vector2d btmcenter = (btmpts2d[0] + btmpts2d[2]) / 2;
            int index = -1;
            double maxdot = double.MinValue;
            for (int j = 0; j < btmpts2d.Length; ++j)
            {
                double dot = ((farcorner - btmcenter).Normalize()).Dot((btmpts2d[j] - btmcenter).Normalize());
                if (dot > maxdot)
                {
                    maxdot = dot;
                    index = j;
                }
            }

            int prev = (index - 1 + btmpts2d.Length) % btmpts2d.Length;
            int next = (index + 1) % btmpts2d.Length;

            if ((btmpts2d[index] - btmpts2d[prev]).Cross(btmpts2d[next] - btmpts2d[index]) < 0)
            {
                int temp = prev;
                prev = next;
                next = temp;
            }

            Vector2d c1 = btmpts2d[prev];
            Vector2d c2 = btmpts2d[index];
            Vector2d c3 = btmpts2d[next];

            Vector2d p1, p3;
            Vector2d p2 = farcorner;
            Vector2d dir = (nearcorner - farcorner).Normalize();
            Vector2d dir1 = (c1 - c2).Normalize();
            Vector2d dir2 = (c3 - c2).Normalize();
            double len1 = (c1 - c2).Length();
            double len2 = (c3 - c2).Length();
            if (dir.Dot(dir1) > dir.Dot(dir2))
            {
                // nearcorner <==> c1.
                double len = Math.Max(len1 + 1e-5, (nearcorner - farcorner).Dot(dir1));
                p1 = p2 + dir1 * len;
                p3 = p2 + (len * len2 / len1) * dir2;
            }
            else
            {
                // nearcorner <==> c3.
                double len = Math.Max(len2 + 1e-5, (nearcorner - farcorner).Dot(dir2));
                p3 = p2 + dir2 * len;
                p1 = p2 + (len * len1 / len2) * dir1;
            }

            Vector2d[] shadowpoly = new Vector2d[] { c1, p1, p2, p3, c3, c2 };
            return shadowpoly.Select(p => new Vector3d(p)).ToArray();
        }

        private bool islightinfoloaded = false;

        public void SaveLightFile(string filename)
        {
            StreamWriter sw = new StreamWriter(filename);
            sw.WriteLine(this.optimlightposition.x + " " + this.optimlightposition.y + " " + this.optimlightposition.z);
            sw.Close();
            Program.OutputText("Saved light file to: " + filename, true);
        }
        public bool ReadLightFile(string filename)
        {
            if (!File.Exists(filename))
            {
                Program.OutputText("Failed to read light files, not exist ...", true);
                return false;
            }
            this.vp2d = new Vector2d[3];
            StreamReader sr = new StreamReader(filename);
            char[] delimiters = { ' ', '\t' };
            while (sr.Peek() > -1)
            {
                string line = sr.ReadLine();
                string[] tokens = line.Split(delimiters, StringSplitOptions.RemoveEmptyEntries);
                double x = double.Parse(tokens[0]);
                double y = double.Parse(tokens[1]);
                double z = double.Parse(tokens[2]);
                this.optimlightposition = new Vector3d(x, y, z);
                this.LightSize = optimlightposition.z / 10;
            }
            sr.Close();
            return true;
        }
        private void InitShadowParameters()
        {
            this.LightSamplingNumber = 30;
            this.Lightness = 0.5;
        }
        public bool AssignTextures()
        {
            Polygon[] polygons = this.selectedfaces.ToArray();
            if (polygons.Length == 0 || polygons.Length % 2 != 0)
                return false;

            int shift = polygons.Length / 2;
            for (int i = 0; i < shift; i += 1)
            {
                Polygon src = polygons[i];
                Polygon tar = polygons[i + shift];
                if (src.points.Count != tar.points.Count)
                    return false;

                tar.assignedpolygon = src;
                tar.assignedtextures = new Vector2d[tar.points.Count];
                for (int j = 0; j < tar.assignedtextures.Length; ++j)
                    tar.assignedtextures[j] = src.points[j].texcoord;
            }
            return true;
        }



        // shadow cast.
        public void CastProxyShadowInTheScene()
        {
            Image<Bgr, byte> composition = this.image.Copy();
           
            this.CalculatePolygonVisibilities();

            Vector3d lp = this.optimlightposition;
            foreach (PolyProxy proxy in this.polyproxies)
            {
                Vector3d[] polygon = this.CalculateProxyShadowNew(proxy, lp);
                Point[] pts = polygon
                    .Select(p => Vector2Point(this.cameracalibrator.ComputePointImgPos(p)))
                    .ToArray();
                this.DrawAntialiasedPolyline(composition, pts, (new Bgr(Color.Red)).MCvScalar, 1);

                Vector2d[] hull = proxy.GetConvexhull2d();
                pts = hull
                    .Select(p => Vector2Point(p))
                    .ToArray();
                this.DrawAntialiasedPolyline(composition, pts, (new Bgr(Color.Yellow)).MCvScalar, 2);

                //if (this.polyproxies.IndexOf(proxy) == 4)
                {
                    LineSegment2D[] allintersegments = this.CastProxyShadowInTheScene(proxy);
                    for (int i = 0; i < allintersegments.Length; ++i)
                    {
                        LineSegment2D seg = allintersegments[i];
                        this.DrawAntialiasedPolyline(composition, new Point[] { seg.P1, seg.P2 }, (new Bgr(Color.SkyBlue)).MCvScalar, 1);
                    }
                }
            }

            ImageViewer viewer = new ImageViewer(composition, "cast shadows polygons");
            viewer.Show();
        }
        public LineSegment2D[] CastProxyShadowInTheScene(PolyProxy mainproxy)
        {
            Vector3d lp = this.optimlightposition;
            Vector2d lgp = lp.ToVector2d();
            double h = lp.z;

            Vector3d[] srcpoints = null;
            Vector3d[] groundpolygon = this.CalculateProxyShadowNew(mainproxy, lp, out srcpoints);

            List<Vector3d> shadowsrcpts = new List<Vector3d>();
            List<Polygon> raypolygons = new List<Polygon>();
            for (int i = 0; i < groundpolygon.Length; ++i)
            {
                int next = (i + 1) % groundpolygon.Length;
                Vector3d p1 = groundpolygon[i];
                Vector3d p2 = groundpolygon[next];
                Polygon plg = new Polygon();
                plg.points3 = new List<Point3>();
                plg.points3.Add(new Point3(lp));
                plg.points3.Add(new Point3(p1));
                plg.points3.Add(new Point3(p2));
                plg.ComputeNormal();
                plg.ComputeLocalFrame();
                raypolygons.Add(plg);

            //using (MemStorage storage = new MemStorage())
            //{
            //    Seq<PointF> projections = new Seq<PointF>(storage);
            //    foreach (Point3 p3 in proxy.points3)
            //    {
            //        Vector2d pground = p3.pos.ToVector2d();
            //        double height = p3.pos.z;
            //        double len = height * (pground - lgp).Length() / (H - height);
            //        Vector2d pprojection = (pground - lgp).Normalize() * len + pground;
            //        projections.Push(Vector2PointF(pprojection));
            //    }
            //    Seq<PointF> poly = projections.GetConvexHull(Emgu.CV.CvEnum.ORIENTATION.CV_COUNTER_CLOCKWISE, storage);
            //    return poly.Select(pf => new Vector3d(PointF2Vecter2d(pf))).ToArray();
            //}
            }

            List<LineSegment2D> allintersegments = new List<LineSegment2D>();
            foreach (PolyProxy proxy in this.polyproxies)
            {
                if (mainproxy == proxy)
                    continue;
                //if (this.polyproxies.IndexOf(proxy) != 1)
                //    continue;

                foreach (Polygon plg in proxy.polygons)
                {
                    if (!plg.visible) continue;
                    //if (proxy.polygons.IndexOf(plg) != 1)
                    //    continue;
                    plg.ComputeLocalFrame();

                    Vector2d[] coords = plg.points3
                        .Select(p3 => plg.localframe.GetPointLocalCoord(p3.pos).ToVector2d())
                        .ToArray();

                    int polyindex = 0;
                    foreach (Polygon rayplg in raypolygons)
                    {
                        Segment seg = IntersectionHelper.ConvexPolygonsIntersection3D(rayplg, plg);
                        if (seg != null)
                        {
                            Vector3d sp1 = seg.p1;
                            Vector3d sp2 = seg.p2;
                            Vector3d srcp1 = srcpoints[polyindex];
                            Vector3d srcp2 = srcpoints[(polyindex + 1) % srcpoints.Length];
                            Vector2d coordsp1 = rayplg.localframe.GetPointLocalCoord(sp1).ToVector2d();
                            Vector2d coordsp2 = rayplg.localframe.GetPointLocalCoord(sp2).ToVector2d();
                            Vector2d coordsrcp1 = rayplg.localframe.GetPointLocalCoord(srcp1).ToVector2d();
                            Vector2d coordsrcp2 = rayplg.localframe.GetPointLocalCoord(srcp2).ToVector2d();
                            Vector2d coordlp = rayplg.localframe.GetPointLocalCoord(lp).ToVector2d();

                            Vector2d interpt1, interpt2;
                            IntersectionHelper.LinesIntersection(coordlp, coordsp1, coordsrcp1, coordsrcp2, out interpt1);
                            IntersectionHelper.LinesIntersection(coordlp, coordsp2, coordsrcp1, coordsrcp2, out interpt2);
                            Vector3d projp1 = rayplg.localframe.PointAtCoord(interpt1);
                            Vector3d projp2 = rayplg.localframe.PointAtCoord(interpt2);
                            if ((projp1 - lp).Length() < (sp1 - lp).Length() && (projp2 - lp).Length() < (sp2 - lp).Length())
                            {
                                Vector2d p1 = this.cameracalibrator.ComputePointImgPos(seg.p1);
                                Vector2d p2 = this.cameracalibrator.ComputePointImgPos(seg.p2);
                                Point center = Vector2Point((p1 + p2) / 2);
                                if (this.polylabelmap[center.Y, center.X] == this.polyids[plg])
                                    allintersegments.Add(new LineSegment2D(Vector2Point(p1), Vector2Point(p2)));
                            }
                        }
                        polyindex++;

                        //Vector3d interpt1, interpt2;
                        //bool isintersect1 = line1.Intersect(plg, out interpt1);
                        //bool isintersect2 = line2.Intersect(plg, out interpt2);
                        //Vector2d imposinterpt1 = this.cameracalibrator.ComputePointImgPos(interpt1);
                        //Vector2d imposinterpt2 = this.cameracalibrator.ComputePointImgPos(interpt2);

                        //if (isintersect1 && isintersect2)
                        //{
                        //    Vector2d coord1 = plg.localframe.GetPointLocalCoord(interpt1).ToVector2d();
                        //    Vector2d coord2 = plg.localframe.GetPointLocalCoord(interpt2).ToVector2d();
                        //    Vector2d[] interpts = IntersectionHelper.SegmentPolygonIntersection(coord1, coord2, coords);
                        //    Vector2d[] imposinterpts = interpts
                        //        .Select(p => this.cameracalibrator.ComputePointImgPos(plg.localframe.PointAtCoord(p)))
                        //        .ToArray();
                        //    if (interpts.Length == 0)
                        //    {
                        //        if (Shape2D.PointInPoly(coord1, coords))
                        //            intersegments.Add(new LineSegment2D(Vector2Point(imposinterpt1), Vector2Point(imposinterpt2)));
                        //    }
                        //    else if (interpts.Length == 1)
                        //    {
                        //        if (Shape2D.PointInPoly(coord1, coords))
                        //            intersegments.Add(new LineSegment2D(Vector2Point(imposinterpt1), Vector2Point(imposinterpts[0])));
                        //        else
                        //            intersegments.Add(new LineSegment2D(Vector2Point(imposinterpt2), Vector2Point(imposinterpts[0])));
                        //    }
                        //    else // 2
                        //    {
                        //        if ((interpts[0] - interpts[1]).Length() < 1e-5)
                        //        {
                        //            if (Shape2D.PointInPoly(coord1, coords))
                        //                intersegments.Add(new LineSegment2D(Vector2Point(imposinterpt1), Vector2Point(imposinterpts[0])));
                        //            else
                        //                intersegments.Add(new LineSegment2D(Vector2Point(imposinterpt2), Vector2Point(imposinterpts[0])));
                        //        }
                        //        else
                        //        {
                        //            intersegments.Add(new LineSegment2D(Vector2Point(imposinterpts[0]), Vector2Point(imposinterpts[1])));
                        //        }
                        //    }
                        //}
                    }
                    plg.localframe = null;
                }
            }
            return allintersegments.ToArray();
        }

        private Image<Bgra, byte> TextureNormalization(Image<Bgra, byte> tex, Polygon plg)
        {
            if (!this.islightinfoloaded)
                return tex;
            
            Vector3d n = plg.normal.Normalize();
            Vector3d lp = this.optimlightposition;

            plg.ComputeLocalFrame();

            for (int y = 0; y < tex.Height; ++y)
            {
                double ycoord = (y - 4) / (tex.Height - 8);
                for (int x = 0; x < tex.Width; ++x)
                {
                    double xcoord = (x - 4) / (tex.Width - 8);
                    if (tex[y, x].Alpha == 0) continue;
                    Vector3d p = plg.localframe.PointAtCoord(new Vector2d(xcoord, ycoord));
                    double cos = ((lp - p).Normalize()).Dot(n);
                    if (cos > 0)
                    {
                        double scale = 1 / cos;
                        Bgra c = tex[y, x];
                        c.Blue /= scale;
                        c.Green /= scale;
                        c.Red /= scale;
                        tex[y, x] = c;
                    }
                }
            }
            return tex;
        }

    }
}
