﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using MyGeometry;
using System.Threading;

namespace i3_ImageManipulator
{
	public class IntersectionHelper
	{
		public static bool IsSegmentIntersect(Vector2d p1, Vector2d p2, Vector2d p3, Vector2d p4)
		{
			Vector3d p31 = new Vector3d(p1, 1);
			Vector3d p32 = new Vector3d(p2, 1);
			Vector3d p33 = new Vector3d(p3, 1);
			Vector3d p34 = new Vector3d(p4, 1);
			Vector2d interpt = (p31.Cross(p32)).Cross(p33.Cross(p34)).HomogenousNormalize().ToVector2d();
			double dist1 = (interpt - p1).Dot((p2 - p1).Normalize());
			double dist2 = (interpt - p3).Dot((p4 - p3).Normalize());
			return dist1 >= 0 && dist1 <= (p2 - p1).Length() && dist2 >= 0 && dist2 <= (p4 - p3).Length();
		}
		public static bool IsPolygonIntersect(Vector2d[] pts1, Vector2d[] pts2)
		{
			foreach (Vector2d p in pts1)
				if (Shape2D.PointInPoly(p, pts2))
					return true;
			foreach (Vector2d p in pts2)
				if (Shape2D.PointInPoly(p, pts1))
					return true;
			for (int i = 0; i < pts1.Length; ++i)
				for (int j = 0; j < pts2.Length; ++j)
					if (IsSegmentIntersect(pts1[i], pts1[(i + 1) % pts1.Length], pts2[j], pts2[(j + 1) % pts2.Length]))
						return true;
			return false;
		}
		public static bool IsPolygonInside(Vector2d[] pts1, Vector2d[] pts2)
		{
			foreach (Vector2d p in pts1)
				if (!Shape2D.PointInPoly(p, pts2))
					return false;
			return true;
		}
		public static bool SegmentLineIntersection(Vector2d sp1, Vector2d sp2, Vector2d lp1, Vector2d lp2, out Vector2d interpt)
		{
			Vector3d u = new Vector3d(sp1, 1);
			Vector3d v = new Vector3d(sp2, 1);
			Vector3d segline = u.Cross(v);
			Vector3d m = new Vector3d(lp1, 1);
			Vector3d n = new Vector3d(lp2, 1);
			Vector3d line = m.Cross(n);
			interpt = segline.Cross(line).HomogenousNormalize().ToVector2d();
			double len = (interpt - sp1).Dot((sp2 - sp1).Normalize());
			return (len >= 0 && len <= (sp2 - sp1).Length());
		}

		public static bool SegmentLineIntersectionDir(Vector2d sp1, Vector2d sp2, Vector2d lp, Vector2d ldir, out Vector2d interpt)
		{
			Vector2d lp2 = lp + ldir;
			return SegmentLineIntersection(sp1, sp2, lp, lp2, out interpt);
		}

		public static Vector2d[] LinePolygonIntersection(Vector2d lp, Vector2d ldir, Vector2d[] poly, out int[] segindices)
		{
			List<Vector2d> interpts = new List<Vector2d>();
			List<int> segindexlist = new List<int>();
			for (int i = 0; i < poly.Length; ++i)
			{
				Vector2d interpt;
				if (SegmentLineIntersectionDir(poly[i], poly[(i + 1) % poly.Length], lp, ldir, out interpt))
				{
					interpts.Add(interpt);
					segindexlist.Add(i);
				}
			}
			segindices = segindexlist.ToArray();
			return interpts.ToArray();
		}

		public static Vector2d[] LinePolygonIntersection(Vector2d lp1, Vector2d lp2, Vector2d[] poly)
		{
			List<Vector2d> interpts = new List<Vector2d>();
			for (int i = 0; i < poly.Length; ++i)
			{
				Vector2d interpt;
				if (SegmentLineIntersection(poly[i], poly[(i + 1) % poly.Length], lp1, lp2, out interpt))
				{
					interpts.Add(interpt);
				}
			}
			return interpts.ToArray();
		}

		public static Segment ConvexPolygonsIntersection3D(Polygon plg1, Polygon plg2)
		{
			// plane1: r*n1 = h1 = r10*n1
			// plane2: r*n2 = h2 = r20*n2
			// r = (c1*n1+c2*n2)+l(n1xn2)
			// c1 = (h1 - h2(n1*n2)) / (1 - (n1*n2)^2)
			// c2 = (h2 - h1(n1*n2)) / (1 - (n1*n2)^2)

			Vector3d n1 = plg1.normal;
			Vector3d n2 = plg2.normal;
			Vector3d linedir = n1.Cross(n2);
			if (linedir.Length() < 1e-7)
				return null;

			double h1 = plg1.points3[0].pos.Dot(n1);
			double h2 = plg2.points3[0].pos.Dot(n2);
			double c1 = (h1 - h2 * (n1.Dot(n2))) / (1 - Math.Pow(n1.Dot(n2), 2));
			double c2 = (h2 - h1 * (n1.Dot(n2))) / (1 - Math.Pow(n1.Dot(n2), 2));
			Vector3d pos = c1 * n1 + c2 * n2;
			Vector3d dir = linedir.Normalize();

			Vector3d interpt1 = pos;
			Vector3d interpt2 = pos + 10 * dir;

			//return new Segment(pos - dir, pos + dir);


			Vector2d[] coords1 = plg1.points3
			.Select(p3 => plg1.localframe.GetPointLocalCoord(p3.pos).ToVector2d())
			.ToArray();
			Vector2d[] coords2 = plg2.points3
			.Select(p3 => plg2.localframe.GetPointLocalCoord(p3.pos).ToVector2d())
			.ToArray();

			Vector2d coord1 = plg1.localframe.GetPointLocalCoord(interpt1).ToVector2d();
			Vector2d coord2 = plg1.localframe.GetPointLocalCoord(interpt2).ToVector2d();
			Vector2d[] interpts1 = IntersectionHelper.LinePolygonIntersection(coord1, coord2, coords1);
			Vector2d coord3 = plg2.localframe.GetPointLocalCoord(interpt1).ToVector2d();
			Vector2d coord4 = plg2.localframe.GetPointLocalCoord(interpt2).ToVector2d();
			Vector2d[] interpts2 = IntersectionHelper.LinePolygonIntersection(coord3, coord4, coords2);

			Vector3d[] interpts3d1 = interpts1.Select(p2 => plg1.localframe.PointAtCoord(p2)).ToArray();
			Vector3d[] interpts3d2 = interpts2.Select(p2 => plg2.localframe.PointAtCoord(p2)).ToArray();
			Vector2d[] interpts1coord2 = interpts3d1.Select(p3 => plg2.localframe.GetPointLocalCoord(p3).ToVector2d()).ToArray();
			Vector2d[] interpts2coord1 = interpts3d2.Select(p3 => plg1.localframe.GetPointLocalCoord(p3).ToVector2d()).ToArray();

			if (interpts1.Length != 2 || interpts2.Length != 2)
				return null;

			if (Shape2D.PointInPoly(interpts2coord1[0], coords1))
			{
				if (Shape2D.PointInPoly(interpts2coord1[1], coords1))
				{
					return new Segment(interpts3d2[0], interpts3d2[1]);
				}
				else
				{
					double dot1 = (interpts1[0] - interpts2coord1[0]).Dot(interpts2coord1[1] - interpts2coord1[0]);
					double dot2 = (interpts1[1] - interpts2coord1[0]).Dot(interpts2coord1[1] - interpts2coord1[0]);
					if (dot1 < dot2)
						return new Segment(interpts3d2[0], interpts3d1[1]);
					else
						return new Segment(interpts3d2[0], interpts3d1[0]);
				}
			}
			else if (Shape2D.PointInPoly(interpts2coord1[1], coords1))
			{
				double dot1 = (interpts1[0] - interpts2coord1[1]).Dot(interpts2coord1[0] - interpts2coord1[1]);
				double dot2 = (interpts1[1] - interpts2coord1[1]).Dot(interpts2coord1[0] - interpts2coord1[1]);
				if (dot1 < dot2)
					return new Segment(interpts3d2[1], interpts3d1[1]);
				else
					return new Segment(interpts3d2[1], interpts3d1[0]);
			}
			else if (Shape2D.PointInPoly(interpts1coord2[0], coords2) && Shape2D.PointInPoly(interpts1coord2[1], coords2))
			{
				return new Segment(interpts3d1[0], interpts3d1[1]);
			}
			else
			{
				return null;
			}
		}

		public static Vector2d ComputeOutVectorRoom(PolyProxy proxy, PolyProxy roombox, out Vector2d outpt, out double rad)
		{
			Vector2d outvec = new Vector2d();
			outpt = new Vector2d();
			rad = 0;

			Polygon poly1 = proxy.polygons[0];
			Polygon poly2 = roombox.polygons[0];
			Vector2d[] pts1 = new Vector2d[poly1.points3.Count];
			for (int i = 0; i < pts1.Length; ++i)
			{
				pts1[i] = poly1.points3[i].pos.ToVector2d();
			}
			Vector2d[] pts2 = new Vector2d[poly2.points3.Count];
			for (int i = 0; i < pts1.Length; ++i)
			{
				pts2[i] = poly2.points3[i].pos.ToVector2d();
			}

			Vector2d center1 = proxy.center.pos.ToVector2d();
			Vector2d center2 = roombox.center.pos.ToVector2d();
			Vector2d centerdir = (center2 - center1).Normalize();

			Vector2d outdir = new Vector2d();
			double mindist = double.MaxValue;
			Vector2d minoutvec = new Vector2d();

			outdir = centerdir;

			Vector2d tempoutpt;
			double temprad;
			outvec = ComputeOutVectorRoomDir(pts1, pts2, outdir, out tempoutpt, out temprad);
			if (!outvec.Equals(Vector2d.Zero) && outvec.Length() < mindist)
			{
				minoutvec = outvec;
				outpt = tempoutpt;
				rad = temprad;
			}
			//}
			return minoutvec;
		}

		private static Vector2d ComputeOutVectorRoomDir(Vector2d[] pts1, Vector2d[] pts2, Vector2d outdir, out Vector2d outpt, out double rad)
		{
			Vector2d outvec = new Vector2d();
			outpt = new Vector2d();
			rad = 0;
			for (int i = 0; i < pts1.Length; ++i)
			{
				Vector2d p = pts1[i];
				if (Shape2D.PointInPoly(p, pts2))
					continue;
				int[] segindices;
				Vector2d[] interpts = IntersectionHelper.LinePolygonIntersection(p, outdir, pts2, out segindices);
				for (int j = 0; j < interpts.Length; ++j)
				{
					Vector2d pt = interpts[j];
					Vector2d off = pt - p;
					if (off.Dot(outdir) < 0) continue;
					off += outdir * 1e-7;
					Vector2d[] pts = new Vector2d[pts1.Length];
					for (int k = 0; k < pts.Length; ++k)
						pts[k] = pts1[k] + off;
					if (IsPolygonInside(pts, pts2))
					{
						outpt = pt;
						outvec = pt - p;
						Vector2d dir1 = (pts1[i] - pts1[(i + 1) % pts1.Length]).Normalize();
						Vector2d dir2 = (pts1[i] - pts1[(i - 1 + pts1.Length) % pts1.Length]).Normalize();
						int segindex = segindices[j];
						Vector2d dir3 = (pts2[segindex] - pts2[(segindex + 1) % pts2.Length]).Normalize();
						double dot1 = Math.Abs(dir1.Dot(dir3));
						double dot2 = Math.Abs(dir2.Dot(dir3));
						rad = (dot1 > dot2) ? -Math.Acos(dot1) : Math.Acos(dot2);
						return outvec;
					}
				}
			}
			return outvec;
		}

		public static Vector2d ComputeOutVector(PolyProxy proxy1, PolyProxy proxy2, out Vector2d outpt, out double rad)
		{
			Vector2d outvec = new Vector2d();
			outpt = new Vector2d();
			rad = 0;

			Polygon poly1 = proxy1.polygons[0];
			Polygon poly2 = proxy2.polygons[0];
			Vector2d[] pts1 = new Vector2d[poly1.points3.Count];
			for (int i = 0; i < pts1.Length; ++i)
			{
				pts1[i] = poly1.points3[i].pos.ToVector2d();
			}
			Vector2d[] pts2 = new Vector2d[poly2.points3.Count];
			for (int i = 0; i < pts1.Length; ++i)
			{
				pts2[i] = poly2.points3[i].pos.ToVector2d();
			}

			Vector2d center1 = proxy1.center.pos.ToVector2d();
			Vector2d center2 = proxy2.center.pos.ToVector2d();
			Vector2d centerdir;
			if (center1.Equals(center2))
				centerdir = (proxy2.polygons[0].linesegments[0].u.pos - proxy2.polygons[0].linesegments[0].v.pos).Normalize();
			else
				centerdir = (center1 - center2).Normalize();

			double mindist = double.MaxValue;
			Vector2d minoutvec = new Vector2d();
			Vector2d outdir = centerdir;
			Vector2d tempoutpt;
			double temprad;
			outvec = ComputeOutVectorDir(pts1, pts2, outdir, out tempoutpt, out temprad);
			if (!outvec.Equals(Vector2d.Zero) && outvec.Length() < mindist)
			{
				minoutvec = outvec;
				outpt = tempoutpt;
				rad = temprad;
			}
			//}
			return minoutvec;
		}

		private static Vector2d ComputeOutVectorDir(Vector2d[] pts1, Vector2d[] pts2, Vector2d outdir, out Vector2d outpt, out double rad)
		{
			Vector2d outvec = Vector2d.Zero;
			outpt = new Vector2d();
			rad = 0;
			for (int i = 0; i < pts1.Length; ++i)
			{
				Vector2d p = pts1[i];
				int[] segindices;
				Vector2d[] interpts = IntersectionHelper.LinePolygonIntersection(p, outdir, pts2, out segindices);
				for (int j = 0; j < interpts.Length; ++j)
				{
					Vector2d pt = interpts[j];
					Vector2d off = pt - p;
					if (off.Dot(outdir) < 0) continue;
					off += outdir * 1e-7;
					Vector2d[] pts = new Vector2d[pts1.Length];
					for (int k = 0; k < pts.Length; ++k)
						pts[k] = pts1[k] + off;
					if (!IsPolygonIntersect(pts, pts2))
					{
						outpt = pt;
						outvec = pt - p;
						Vector2d dir1 = (pts1[i] - pts1[(i + 1) % pts1.Length]).Normalize();
						Vector2d dir2 = (pts1[i] - pts1[(i - 1 + pts1.Length) % pts1.Length]).Normalize();
						int segindex = segindices[j];
						Vector2d dir3 = (pts2[segindex] - pts2[(segindex + 1) % pts2.Length]).Normalize();
						double dot1 = Math.Abs(dir1.Dot(dir3));
						double dot2 = Math.Abs(dir2.Dot(dir3));
						rad = (dot1 > dot2) ? -Math.Acos(dot1) : Math.Acos(dot2);
						return outvec;
					}
				}
			}

			return outvec;
		}

		public static bool LinesIntersection(Vector2d lp1, Vector2d lp2, Vector2d lp3, Vector2d lp4, out Vector2d interpt)
		{
			double dot = ((lp2 - lp1).Normalize()).Dot((lp4 - lp3).Normalize());
			if (Math.Abs((Math.Abs(dot) - 1)) < 1e-7)
			{
				interpt = Vector2d.Zero;
				return false;
			}
			Vector3d u = new Vector3d(lp1, 1);
			Vector3d v = new Vector3d(lp2, 1);
			Vector3d line1 = u.Cross(v);
			Vector3d m = new Vector3d(lp3, 1);
			Vector3d n = new Vector3d(lp4, 1);
			Vector3d line2 = m.Cross(n);
			interpt = line1.Cross(line2).HomogenousNormalize().ToVector2d();
			return true;
		}

	}
	public class NonLocalEditor
	{
		public static double cos5 = Math.Cos(1.0 / 36 * Math.PI);
		public static double cos3 = Math.Cos(1.0 / 60 * Math.PI);
		
		public int numpolygons = 0;

		public List<PolyProxy> proxies = null;
		public List<PolyProxy> fixedproxies = null;
		public List<ProxyGroup> proxygroups = null;
		public List<PolygonGroup> polygroups = new List<PolygonGroup>();
		private PolyProxy handleproxy = null;

		public NonLocalEditor(List<PolyProxy> prxs, List<ProxyGroup> pxgrups, RepContainer bigbox)
		{
			this.proxygroups = pxgrups;
			this.proxies = new List<PolyProxy>();
			this.proxies.AddRange(prxs);
			
			// if bigbox exist, means repetition editing, exclude inside proxies
			// from propagation
			if (bigbox != null)
			{
				this.proxies.Add(bigbox);
				foreach (PolyProxy proxy in bigbox.inobjects)
				{
					this.proxies.Remove(proxy);
				}
			}
			this.FindRelations();
			this.PreparePropagation();
		}

		public List<Polygon> propagatingpolygons = new List<Polygon>();
		public List<Polygon> propagatingtracingpolygons = new List<Polygon>();
		public Polygon GetPropagatingPolygon(int index)
		{
			if (index < 0) return null;
			int n = this.propagatingtracingpolygons.Count;
			return this.propagatingtracingpolygons[index % n];
		}
		public PolygonGroup GetPolygonGroup(int index)
		{
			if (index < 0) return null;
			int n = this.polygroups.Count;
			return this.polygroups[index % n];
		}
		public void AddHeightPreserveGroup(List<Polygon> plgs)
		{
			int gid = this.polygroups.Count;
			List<Polygon> polygons = new List<Polygon>();
			polygons.AddRange(plgs);
			PolygonGroup g = new PolygonGroup(polygons, gid);
			foreach (Polygon plg in plgs)
			{
				plg.groupindex.Add(gid);
			}
			g.groupstyle = MutualRelation.HEIGHTPRESERVE;
			this.polygroups.Add(g);
		}
		public void AddCoplanarGroup(List<Polygon> plgs)
		{
			int gid = this.polygroups.Count;
			List<Polygon> polygons = new List<Polygon>();
			polygons.AddRange(plgs);
			PolygonGroup g = new PolygonGroup(polygons, gid);
			foreach (Polygon plg in plgs)
			{
				plg.groupindex.Add(gid);
			}
			g.groupstyle = MutualRelation.COPLANAR;
			this.polygroups.Add(g);
		}

		//----------------------------------------------------------------------------------
		//- entries
		public void Init()
		{
			foreach (PolyProxy proxy in this.proxies)
				foreach (Polygon p in proxy.polygons)
					p.visited = false;
			foreach (PolygonGroup g in this.polygroups)
				g.visited = false;
			foreach (ProxyGroup g in this.proxygroups)
				g.visited = false;
		}
		

		// inter- and intra- proxy level propagation
		public void Propagate(PolyProxy handle, Polygon handlepolygon)
		{
			if (handle == null) return;

			if (!this.Check())
			{
				Program.OutputText("@Propagation failed. ", true);
				return;
			}

			this.handleproxy = handle;
			this.propagatingpolygons.Clear();
			this.HandleFixedProxies();
			this.IntraObjectPropagation(handle, handlepolygon);
			this.InterObjectPropagation();
			this.HandleRepetitions();
		}


		private bool Check()
		{
			foreach (PolyProxy px in this.proxies)
			{
				foreach (Polygon plg in px.polygons)
				{
					if (plg.propagatepoints3.Count == 0)
					{
						Program.OutputText("@Propagation Check: polygon has null propagating points", true);
						return false;
					}
				}
			}
			return true;
		}


		private void HandleFixedProxies()
		{
			// find those fixed polygons
			this.fixedproxies = new List<PolyProxy>();
			foreach (PolyProxy px in this.proxies)
			{
				if (px.isfixed)
				{
					this.fixedproxies.Add(px);
				}
			}
			this.fixedproxies.Add(this.handleproxy);
			foreach (PolyProxy px in fixedproxies)
			{
				foreach (Polygon ply in px.polygons)
				{
					ply.PropagationUpdate(); // propagation points position,
					ply.visited = true;
				}
			}
		}
		private void IntraObjectPropagation(PolyProxy handle, Polygon handlepolygon)
		{
			// find polygons to propagate
			ProxyGroup group = this.proxygroups[handle.groupindex];
			foreach (PolyProxy px in group.proxies)
				px.visited = true;
			this.PropagateGroup(group, handlepolygon);
		}
		private void InterObjectPropagation()
		{
			// set groups which has unhandled polygons
			foreach (PolygonGroup g in this.polygroups)
			{
				foreach (Polygon plg in g.polygons)
				{
					if (!plg.visited)
					{
						g.visited = false;
						break;
					}
				}
			}

			// find already treated polygons
			List<Polygon> handledplgs = new List<Polygon>();
			foreach (PolyProxy px in this.proxies)
			{
				foreach (Polygon plg in px.polygons)
				{
					if (plg.visited)
					{
						handledplgs.Add(plg);
					}
				}
			}
			
			// align co-planarity / height preserve
			foreach (Polygon plg in handledplgs)
			{
				foreach (int gid in plg.groupindex)
				{
					PolygonGroup g = this.polygroups[gid];
					if (g.groupstyle == MutualRelation.COPLANAR || g.groupstyle == MutualRelation.HEIGHTPRESERVE)
					{
						foreach (Polygon plg2 in g.polygons)
						{
							if (!plg2.visited)
							{
								foreach (Polygon pg in g.polygons)
								{
									if (pg.visited)
									{
										this.Align(plg2, pg, g.groupstyle);
										break;
									}
								}

								//		this.UpdateSinglePolygon(plg2, handledplgs);
								
								plg2.PropagationUpdate();
								this.propagatingpolygons.Add(plg2);
								plg2.visited = true;
							}
						}
					}
				}
			}

			// propagates to other groups by proximity
			List<ProxyGroup> handledgroups = new List<ProxyGroup>();
			foreach (ProxyGroup g in this.proxygroups)
			{
				if (g.visited)
					handledgroups.Add(g);
			}
			while (true)
			{
				// find the next group to handle
				ProxyGroup next = this.FindClosestUnhandledProxyGroup(handledgroups);
				if (next == null) break;

				this.PropagateGroup(next, null);

			}
		}
		private void HandleRepetitions()
		{
			// find the group where the handle lies
			ProxyGroup handlegroup = null;
			foreach (ProxyGroup g in this.proxygroups)
			{
				if (g.proxies.Contains(this.handleproxy))
				{
					handlegroup = g;
					break;
				}
			}

			// handle repetitions
			if (handlegroup != null && handlegroup.reps != null)
			{
				foreach (ProxyGroup g2 in handlegroup.reps)
				{
					this.CopyRepetition(handlegroup, g2);
				}
				// 
				this.FindRelations();
			}
		}
		private void PropagateGroup(ProxyGroup pxgrup, Polygon handlepolygon)
		{
			// this is a simplified version of iWires propagation, for algorithm details,
			// please refer to the iWires paper. ;)
			pxgrup.visited = true;

			List<Polygon> handlers = new List<Polygon>();
			foreach (PolyProxy px in pxgrup.proxies)
			{
				foreach (Polygon plg in px.polygons)
				{
					if (plg.visited)
					{
						handlers.Add(plg);
					}
				}
			}

			if (handlers.Count == 0) return;


			// find polygons to propagate
			List<PolyProxy> obj = pxgrup.proxies;
			List<Polygon> plgs2handle = new List<Polygon>(); // polygons to handle
			foreach (PolyProxy px in obj)
			{
				plgs2handle.AddRange(px.polygons);
			}

			Program.OutputText("@PropagatingGroup: polygons to handle = " + plgs2handle.Count, true);


			// proximity-based group propagation
			List<Polygon> plgshandled = new List<Polygon>();
			plgshandled.AddRange(handlers);

			if (handlepolygon != null)
			{
				plgshandled.Remove(handlepolygon);
				plgshandled.Insert(0, handlepolygon);
			}

			// symmetry
			List<Polygon> symmetries = new List<Polygon>();
			foreach (Polygon ply in plgshandled)
			{
				if (UpdateSymmetry(ply))
				{
					foreach (Polygon pl in ply.mirrorsymmetries)
					{
						pl.PropagationUpdate(); // propagation points position, polygon center
						pl.visited = true;
					}
					symmetries.AddRange(ply.mirrorsymmetries);
					foreach (Polygon pl in ply.mirrorsymmetries)
						pl.visited = true;
				}
			}
			plgshandled.AddRange(symmetries);
			symmetries.Clear();

			while (true)
			{
				// find the cloest group
				double mindis = double.MaxValue; Polygon closest = null;
				foreach (Polygon plg in plgs2handle)
				{
					if (plg.visited) continue;
					double dist = this.FindDist(plg, plgshandled);
					if (dist < mindis)
					{
						mindis = dist;
						closest = plg;
					}
				}

				// update/optimize the group
				if (closest != null)
				{
					this.UpdateSinglePolygon(closest, plgshandled);
					this.propagatingpolygons.Add(closest);

					closest.PropagationUpdate();
					if (UpdateSymmetry(closest))
					{
						// symetry
						this.propagatingpolygons.AddRange(closest.mirrorsymmetries);
						plgshandled.AddRange(closest.mirrorsymmetries);
						foreach (Polygon pl in closest.mirrorsymmetries)
						{
							pl.PropagationUpdate();
							pl.visited = true;
						}
					}

					plgshandled.Add(closest);
					closest.visited = true;
				}
				else
					break;
			}


			// re-construct each proxy
			foreach (PolyProxy box in obj)
			{
				if (!this.fixedproxies.Contains(box))
					this.Optimize(box);
			}

			// find no. of unhandled polygons
			int count = 0;
			foreach (PolyProxy px in this.proxies)
			{
				foreach (Polygon plg in px.polygons)
				{
					if (!plg.visited)
					{
						count++;
					}
				}
			}
			Program.OutputText("no of rest polygons = " + count, true);
		}
		public double FindDist(Polygon plg, List<Polygon> refpolys)
		{
			double mindis = double.MaxValue;
			foreach (Polygon q in refpolys)
			{
				double dis = (q.center.pos - plg.center.pos).Length();
				if (dis < mindis)
				{
					mindis = dis;
				}
			}
			return mindis;
		}
		private ProxyGroup FindClosestUnhandledProxyGroup(List<ProxyGroup> handledgroups)
		{
			ProxyGroup closest = null;
			double mindis = double.MaxValue;
			foreach (ProxyGroup g in this.proxygroups)
			{
				if (!g.visited)
				{
					foreach (ProxyGroup g2 in handledgroups)
					{
						double dis = this.FindGroupDistance(g, g2);
						if (dis < mindis)
						{
							mindis = dis;
							closest = g;
						}
					}
				}
			}
			return closest;
		}
		private double FindGroupDistance(ProxyGroup g1, ProxyGroup g2)
		{
			double min = 1e8;
			foreach (PolyProxy p in g1.proxies)
			{
				double mindis = double.MaxValue;
				foreach (PolyProxy q in g2.proxies)
				{
					double dis = FindDistance(p, q);
					if (dis < mindis)
					{
						mindis = dis;
					}
				}
				if (min > mindis)
				{
					min = mindis;
				}
			}
			return min;
		}
		private void PropagateProxy(PolyProxy px, List<Polygon> refplgs)
		{
			foreach (Polygon plg in px.polygons)
			{
				if (!plg.visited)
				{
					this.UpdateSinglePolygon(plg, refplgs);
					plg.visited = true;
					plg.PropagationUpdate();
				}
			}
		}

		//----------------------------------------------------------------------------------
		// repetitions, extra..
		private void CopyRepetition(ProxyGroup g1, ProxyGroup g2)
		{
			// copy g1 to g2
			Vector3d center1, axis1;
			this.FindGroupGroundOrentationInfo(g1, out center1, out axis1);
			Vector3d center2, axis2;
			this.FindGroupGroundOrentationInfo(g2, out center2, out axis2);

			ImageManipulator.userballs.Add(center1);
			ImageManipulator.userballs.Add(center2);


			Vector3d t = center2 - center1;
			axis1 = axis1.Normalize();
			axis2 = axis2.Normalize();
			Vector3d rotaxis = (axis1.Cross(axis2)).Normalize();
			if (double.IsNaN(rotaxis.x)) rotaxis = new Vector3d(0, 0, 1);

			double cos = Math.Abs(axis1.Dot(axis2));
			if (cos > 1) cos = 1;
			double rotangle = Math.Acos(cos);
			Matrix4d T = Matrix4d.TranslationMatrix(t);
			Matrix4d R = Matrix4d.RotationMatrix(rotaxis, rotangle);
			Matrix4d Q = T * R;
			// do clone
			List<PolyProxy> newproxies = new List<PolyProxy>();
			foreach (PolyProxy px in g1.proxies)
			{
				PolyProxy qx = px.Clone() as PolyProxy;
				ImageManipulator.TruelyDeformProxy(qx, Q, center1, 1);
				newproxies.Add(qx);
			}

			// update propagation points
			foreach (PolyProxy px in newproxies)
			{
				foreach (Polygon plg in px.polygons)
				{
					plg.PropagationUpdate();
				}
			}


			// update the proxies
			int id = this.proxies.IndexOf(g2.proxies[0]);
			List<PolyProxy> allproxies = new List<PolyProxy>();
			for (int i = 0; i < id; ++i)
			{
				allproxies.Add(this.proxies[i]);
			}
			allproxies.AddRange(newproxies);
			for (int i = id + g2.proxies.Count; i < this.proxies.Count; ++i)
			{
				allproxies.Add(this.proxies[i]);
			}
			g2.proxies = newproxies;
			this.proxies = allproxies;
		}
		private void FindGroupGroundOrentationInfo(ProxyGroup g, out Vector3d center, out Vector3d axis)
		{
			CoordinateFrame frame = g.proxies[0].frame;
			Vector3d[] bpts = ImageManipulator.GetBoundingBoxPoints(g.proxies);
			center = (bpts[0] + bpts[6]) / 2;
			center.z = 0;
			axis = frame.x;
		}

		// members
		private void FindRelations()
		{
			this.polygroups = new List<PolygonGroup>();
			foreach (PolyProxy px in this.proxies)
			{
				foreach (Polygon plg in px.polygons)
				{
					plg.groupindex.Clear();
				}
			}

			List<Polygon> allpolys = new List<Polygon>();
			foreach (PolyProxy proxy in this.proxies)
				allpolys.AddRange(proxy.polygons);
			int index = 0;
			// concentric.
			foreach (Polygon p in allpolys) p.visited = false;
			foreach (Polygon p in allpolys)
			{
				List<Polygon> polys = new List<Polygon>();
				polys.Add(p);
				if (p.visited) continue;
				p.visited = true;
				foreach (Polygon q in allpolys)
				{
					if (q == p) continue;
					if (!q.visited && IsConcentric(p, q))
					{
						polys.Add(q);
						q.visited = true;
					}
				}
				if (polys.Count > 2)
				{
		//			Program.OutputText("concentric group " + index + " := " + polys.Count, true);
					PolygonGroup g = new PolygonGroup(polys, index++);
					g.groupstyle = MutualRelation.CONTAINER;
					this.polygroups.Add(g);
				}
			}
			// coplanar.
			foreach (Polygon p in allpolys) p.visited = false;
			foreach (Polygon p in allpolys) {
				List<Polygon> polys = new List<Polygon>();
				if (p.visited) continue;
				p.visited = true;
				polys.Add(p);
				foreach (Polygon q in allpolys) {
					if (q == p) continue;
					if (!q.visited && IsCoplanar(p, q))
					{
						polys.Add(q);
						q.visited = true;
					}
				}
				if (polys.Count > 1) {
		//			Program.OutputText("coplanar group " + index + " := " + polys.Count,true);
					PolygonGroup g = new PolygonGroup(polys, index++);
					g.groupstyle = MutualRelation.COPLANAR;
					this.polygroups.Add(g);
				}
			}
			// parallel.
			foreach (Polygon p in allpolys) p.visited = false;
			foreach (Polygon p in allpolys)
			{
				List<Polygon> polys = new List<Polygon>();
				if (p.visited) continue;
				polys.Add(p);
				p.visited = true;
				foreach (Polygon q in allpolys)
				{
					if (q == p) continue;
					if (!q.visited && IsParallel(p, q))
					{
						polys.Add(q);
						q.visited = true;
					}
				}
				if (polys.Count > 1)
				{
			//		Program.OutputText("parallel group " + index + " := " + polys.Count, true);
					PolygonGroup g = new PolygonGroup(polys, index++);
					g.groupstyle = MutualRelation.PARALLEL;
					this.polygroups.Add(g);
				}
			}
			// assign group index
			foreach (PolygonGroup g in this.polygroups) {
				foreach (Polygon ply in g.polygons)
					ply.groupindex.Add(g.index);
				this.numpolygons += g.polygons.Count;
			}
		}
		private void PreparePropagation()
		{
			// create propagating point3
			foreach (PolyProxy proxy in this.proxies)
			{
				foreach (Polygon plg in proxy.polygons)
				{
					foreach (Point3 p in plg.propagatepoints3)
					{
						p.R = Matrix3d.IdentityMatrix();
					}
				}
			}
		}
		private void UpdateSinglePolygon(Polygon p, List<Polygon> polys)
		{
			this.propagatingtracingpolygons.Add(p.Clone() as Polygon);

			List<Point3> refpoints = new List<Point3>();
			foreach (Polygon q in polys) {
				if (q.hostproxy != p.hostproxy)
					refpoints.AddRange(q.propagatepoints3);
			}
			if (refpoints.Count == 0)
			{
				foreach (Polygon q in polys)
				{
					refpoints.AddRange(q.propagatepoints3);
				}
			}
			foreach (Point3 s in p.points3) {
				double mindis = 1e8; Point3 t = null;
				foreach (Point3 q in refpoints) {
					double d = (q.oldpos - s.oldpos).Length();
					if (d < mindis)
					{
						mindis = d;
						t = q;
					}
				}
				if (t != null) {
					s.pos = t.pos + t.R * (s.oldpos - t.oldpos);
					s.R = t.R;
				}
			}
			this.Optimize(p);

			this.propagatingtracingpolygons.Add(p.Clone() as Polygon);

	//		p.UpdatePropagatingPoints();
		}
		private bool UpdateSymmetry(Polygon p)
		{
			// p is a symmetry polygon with some other polygon q
			// update q by symmetry, requiring p is already updated
			if (p.mirrorsymmetries == null) return false;
			List<Symmetry> sms = p.symmelements; int index = 0;
			foreach (Polygon plg in p.mirrorsymmetries)
			{
				if (plg.visited) continue;
				Symmetry sm = sms[index];
				foreach (Point3 pt in p.points3)
				{
					pt.symmpoints[index].pos = ImageManipulator.GetMirrorSymmetryPoint(pt.pos, sm.axis, sm.center);
				}
				index++;
			}
			foreach (Polygon plg in p.mirrorsymmetries)
				this.Optimize(plg);
			return true;
		}
		private void GroupOptimization(PolygonGroup g)
		{

		}
		private void Optimize(PolyProxy box)
		{
			if (box.polygons.Count < 2) return;

			Vector3d x1 = box.polygons[2].center.pos;
			Vector3d x2 = box.polygons[4].center.pos;
			Vector3d y1 = box.polygons[3].center.pos;
			Vector3d y2 = box.polygons[5].center.pos;
			Vector3d z1 = box.polygons[0].center.pos;
			Vector3d z2 = box.polygons[1].center.pos;
			Vector3d xdir = x1 - x2;
			Vector3d ydir = y1 - y2;
			Vector3d zdir = z2 - z1;

			int sign = box.polygons[2].normal.Dot(box.polygons[4].normal) < 0 ? -1 : 1;
			Vector3d xaxis = (box.polygons[2].normal + box.polygons[4].normal * sign).Normalize();
			if (xaxis.Dot(xdir) < 0) xaxis = new Vector3d() - xaxis;
			sign = box.polygons[3].normal.Dot(box.polygons[5].normal) < 0 ? -1 : 1;
			Vector3d yaxis = (box.polygons[3].normal + box.polygons[5].normal * sign).Normalize();
			if (yaxis.Dot(ydir) < 0) yaxis = new Vector3d() - yaxis;
			sign = box.polygons[1].normal.Dot(box.polygons[0].normal) < 0 ? -1 : 1;
			Vector3d zaxis = (box.polygons[1].normal + box.polygons[0].normal * sign).Normalize();
			if (zaxis.Dot(zdir) < 0) zaxis = new Vector3d() - zaxis;

			double x = (x1.Dot(xaxis) + x2.Dot(xaxis))/2;
			double y = (y1.Dot(yaxis) + y2.Dot(yaxis))/2;
			double z = (z1.Dot(zaxis) + z2.Dot(zaxis))/2;


			Vector3d c = x * xaxis + y * yaxis + z * zaxis;


			double xlen = Math.Abs(xdir.Dot(xaxis)) / 2;
			double ylen = Math.Abs(ydir.Dot(yaxis)) / 2;
			double zlen = Math.Abs(zdir.Dot(zaxis)) / 2;
			Matrix3d R = new Matrix3d(xaxis, yaxis, zaxis);
			box.points3[0].pos = c + R * new Vector3d(xlen, -ylen, -zlen);
			box.points3[1].pos = c + R * new Vector3d(xlen, ylen, -zlen);
			box.points3[2].pos = c + R * new Vector3d(-xlen, ylen, -zlen);
			box.points3[3].pos = c + R * new Vector3d(-xlen, -ylen, -zlen);
			box.points3[4].pos = c + R * new Vector3d(xlen, -ylen, zlen);
			box.points3[5].pos = c + R * new Vector3d(xlen, ylen, zlen);
			box.points3[6].pos = c + R * new Vector3d(-xlen, ylen, zlen);
			box.points3[7].pos = c + R * new Vector3d(-xlen, -ylen, zlen);
		}
		private void Optimize(Polygon p)
		{
			List<Point3> points = p.points3;
			int num = points.Count;
			Vector3d c = new Vector3d(); Vector3d n = new Vector3d();
			foreach (Point3 pt in points)
				c += pt.pos;
			c /= num;
			Vector3d o = points[0].pos;
			for (int i = 1; i < num - 1; ++i) {
				Vector3d u = points[i].pos, v = points[i + 1].pos;
				Vector3d norm = (u - o).Cross(v - u).Normalize();
				n += norm;
			}
			n = n.Normalize();
			p.center.pos = c;
			p.normal = n;

			this.Align(p);

			foreach (Point3 pt in points) {
				Vector3d pc = pt.pos - p.center.pos;
				pt.pos = pt.pos - pc.Dot(p.normal) * p.normal;
			}
		}
		private bool Align(Polygon p)
		{
			// align the polygon accordingl to the relations
			bool aligned = false;
			foreach (int gid in p.groupindex)
			{
				List<Polygon> plgs = new List<Polygon>();
				PolygonGroup group = this.polygroups[gid];
				foreach (Polygon q in group.polygons)
				{
					if (q.visited) plgs.Add(q);
				}
				double min = double.MaxValue; Polygon referee = null;
				foreach (Polygon q in plgs)
				{
					if (q.hostproxy == p.hostproxy) continue;
					double d = (p.center.pos - q.center.pos).Length();
					if (d < min)
					{
						referee = q;
						min = d;
					}
				}
				if (referee != null)
				{
					this.Align(p, referee, group.groupstyle);
					aligned = true;
				}
			}
			return aligned;
		}
		private void Align(Polygon p, Polygon q, MutualRelation r)
		{
			Vector3d o = q.center.pos, normal = q.normal;
			switch (r)
			{
				case MutualRelation.PARALLEL:
		//			p.normal = normal;
					break;
				case MutualRelation.COPLANAR:
					{
						this.AlignCoplanar(p, q);
				//		p.normal = normal;
				//		p.center.pos = p.center.pos - (p.center.pos - o).Dot(normal) * normal;
					}
					break;
				case MutualRelation.HEIGHTPRESERVE:
					{
						this.AlignHeightPreserve(p,q);
					}
					break;
				case MutualRelation.CONTAINER:
					{
						p.normal = normal;
						p.center.pos = o + (p.center.pos - o).Dot(normal) * normal;
					}
					break;
			}
		}
		private void AlignCoplanar(Polygon p, Polygon q)
		{
			Vector3d o = q.center.pos, normal = q.normal;
			Vector3d t = (o - p.center.pos).Dot(normal) * normal;
			p.center.pos += t;
			p.normal = normal;
			foreach (Point3 pt in p.points3)
			{
				pt.pos += t;
			}
		}
		private void AlignHeightPreserve(Polygon p, Polygon q)
		{
			// align p, 
			Vector3d axis = Vector3d.Zaxis;
			Vector3d t = new Vector3d(0,0, q.center.pos.z - q.center.oldpos.z);
			p.center.pos = p.center.oldpos + t;
			foreach (Point3 pt in p.points3)
			{
				pt.pos = pt.oldpos + t;
			}
		}
		private bool IsCoplanar2(Polygon p, Polygon q)
		{
			if (p.hostproxy != q.hostproxy && p.hostproxy.segindex != q.hostproxy.segindex)
			{
				if (Math.Abs(p.normal.Dot(Vector3d.Zaxis)) < 0.5) 
					return false;
			}
			if (Math.Abs(p.normal.Dot(q.normal)) < cos3) return false;
			Vector3d t = (p.center.pos - q.center.pos).Normalize();
			if (double.IsNaN(t.x)) return true; // the same face
			double cos = t.Dot(p.normal);
			return Math.Abs(cos) < 1e-3;
		}
		private bool IsCoplanar(Polygon p, Polygon q)
		{
			if (p.hostproxy != q.hostproxy && p.hostproxy.segindex != q.hostproxy.segindex)
			{
				if (Math.Abs(p.normal.Dot(Vector3d.Zaxis)) < 0.5)
					return false;
			}
			if (Math.Abs(p.normal.Dot(q.normal)) < cos3) return false;
			double l1 = p.hostproxy.DiagLength();
			double l2 = q.hostproxy.DiagLength();
			double maxL = l1 > l2 ? l1 : l2;
			double dh = Math.Abs((p.center.pos - q.center.pos).Dot(p.normal));
			return dh / (maxL+1e-8) < 1e-3;
		}
		private bool IsConcentric(Polygon p, Polygon q)
		{
			if (Math.Abs(p.normal.Dot(q.normal)) < cos5) return false;
			double cos = (p.center.pos - q.center.pos).Normalize().Dot(p.normal);
			return Math.Abs(cos) > cos5;
		}
		private bool IsParallel(Polygon p, Polygon q)
		{
			return Math.Abs(p.normal.Dot(q.normal)) > cos5;
		}
		private bool IsHeightPreserve(Polygon p, Polygon q)
		{
			if (p.hostproxy == q.hostproxy || p.hostproxy.segindex == q.hostproxy.segindex)
			{
				return false;
			}
			if (Math.Abs(p.normal.Dot(Vector3d.Zaxis)) < 0.8 ||
				Math.Abs(q.normal.Dot(Vector3d.Zaxis)) < 0.8)
				return false;
			if (Math.Abs(p.normal.Dot(q.normal)) < cos3) return false;
			Vector3d t = (p.center.pos - q.center.pos).Normalize();
			if (double.IsNaN(t.x)) return true; // the same face
			double cos = t.Dot(p.normal);
			return Math.Abs(cos) < 1e-2;
		}
		
		private double FindGroupDist(PolygonGroup g, List<Polygon> polys)
		{
			double dist = 0;
			foreach (Polygon p in g.polygons)
			{
				double mindis = 0;
				foreach (Polygon q in polys)
				{
					double d = (q.center.pos - p.center.pos).Length();
					if (d < mindis) mindis = d;
				}
				dist += mindis;
			}
			return dist / g.polygons.Count;
		}
		private double FindDistance(Polygon p, List<Polygon> polys)
		{
			double mindis = double.MaxValue;
			foreach (Polygon q in polys)
			{
				double d = (q.center.pos - p.center.pos).Length();
				if (d < mindis) mindis = d;
			}
			return mindis;
		}
		private double FindDistance(PolyProxy p, PolyProxy q)
		{
			return (p.center.pos - q.center.pos).Length();
		}
		
	}
}
