﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using MyGeometry;

namespace i3_ImageManipulator
{
    public class LightPositionOptimizer
    {
        /// <summary>
        /// h: proxy height.
        /// bc: center of proxy's bottom polygon.
        /// sc: center of shadow polygon (the rectangle area).
        /// </summary>
        public const double OutWeight = 1;

        private Vector3d[] lps;
        private double[] hs;
        private double[][] weights;
        private double[][] lambdas;
        private int[][] lambdaindices;
        private Vector2d[][] sprofiles, spolygons;

        private List<Vector3d[]> optimizedpositions = new List<Vector3d[]>();

        public Vector3d[][] OptimizationProcessPositions 
        { get { return this.optimizedpositions.ToArray(); } }

        // shadow polygon {c1, p1, p2, p3, c3, c2}.
        // map pi => cj.
        private int[] spolyinds = { -1/*no use*/, 0, 5, 4 };


        public void function1_fvec(double[] xarr, double[] fi, object obj)
        {
            //this.ComputeLambda(xarr);

            double x = xarr[0];
            double y = xarr[1];
            double z = xarr[2];

            // update polygon.
            Vector2d lg = new Vector2d(x, y);
            double lh = z;
            for (int i = 0; i < hs.Length; ++i)
            {
                double h = this.hs[i];
                for (int j = 1; j <= 3; ++j)
                {
                    int index = this.spolyinds[j];
                    Vector2d c = this.spolygons[i][index];
                    this.spolygons[i][j] = (c - lg) * h / (lh - h) + c;
                }
            }

            // fi values.
            int findex = 0;
            for (int i = 0; i < hs.Length; ++i)
            {
                for (int j = 0; j < this.sprofiles[i].Length; ++j)
                {
                    Vector2d p = sprofiles[i][j];
                    int index = this.lambdaindices[i][j];
                    int next = (index + 1) % spolygons[i].Length;
                    double lambda = this.lambdas[i][j];
                    Vector2d q1 = spolygons[i][index];
                    Vector2d q2 = spolygons[i][next];
                    double xpart = p.x - ((1 - lambda) * q1.x + lambda * q2.x);
                    double ypart = p.y - ((1 - lambda) * q1.y + lambda * q2.y);
                    double weight = weights[i][j];
                    fi[findex++] = xpart * weight;
                    fi[findex++] = ypart * weight;
                }
            }
        }

        public void function1_jac(double[] xarr, double[] fi, double[,] jac, object obj)
        {
            //this.ComputeLambda(xarr);

            double x = xarr[0];
            double y = xarr[1];
            double z = xarr[2];

            // update polygon.
            Vector2d lg = new Vector2d(x, y);
            double lh = z;
            for (int i = 0; i < hs.Length; ++i)
            {
                double h = this.hs[i];
                for (int j = 1; j <= 3; ++j)
                {
                    int index = this.spolyinds[j];
                    Vector2d c = this.spolygons[i][index];
                    this.spolygons[i][j] = (c - lg) * h / (lh - h) + c;
                }
            }

            // fi & jac values.
            int findex = 0;
            for (int i = 0; i < hs.Length; ++i)
            {
                double h = this.hs[i];
                for (int j = 0; j < this.sprofiles[i].Length; ++j)
                {
                    Vector2d p = sprofiles[i][j];
                    int index = this.lambdaindices[i][j];
                    int next = (index + 1) % spolygons[i].Length;
                    double lambda = this.lambdas[i][j];
                    Vector2d q1 = spolygons[i][index];
                    Vector2d q2 = spolygons[i][next];
                    double xpart = p.x - ((1 - lambda) * q1.x + lambda * q2.x);
                    double ypart = p.y - ((1 - lambda) * q1.y + lambda * q2.y);
                    double weight = weights[i][j];

                    fi[findex] = xpart * weight;
                    jac[findex, 0] = 0; // x.
                    jac[findex, 1] = 0; // y.
                    jac[findex, 2] = 0; // z.
                    if (index >= 1 && index <= 3)
                    {
                        int pind = this.spolyinds[index];
                        Vector2d c = spolygons[i][pind];
                        jac[findex, 0] += (-(1 - lambda) * h / (h - z)) * weight; // x.
                        jac[findex, 2] += (-(1 - lambda) * h * (x - c.x) / ((z - h) * (z - h))) * weight; // z.
                    }
                    if (next >= 1 && next <= 3)
                    {
                        int pind = this.spolyinds[next];
                        Vector2d c = spolygons[i][pind];
                        jac[findex, 0] += (-lambda * h / (h - z)) * weight; // x.
                        jac[findex, 2] += (-lambda * h * (x - c.x) / ((z - h) * (z - h))) * weight; // z.
                    }
                    findex++;

                    fi[findex] = ypart * weight;
                    jac[findex, 0] = 0; // x.
                    jac[findex, 1] = 0; // y.
                    jac[findex, 2] = 0; // z.
                    if (index >= 1 && index <= 3)
                    {
                        int pind = this.spolyinds[index];
                        Vector2d c = spolygons[i][pind];
                        jac[findex, 1] += (-(1 - lambda) * h / (h - z)) * weight; // y.
                        jac[findex, 2] += (-(1 - lambda) * h * (y - c.y) / ((z - h) * (z - h))) * weight; // z.
                    }
                    if (next >= 1 && next <= 3)
                    {
                        int pind = this.spolyinds[next];
                        Vector2d c = spolygons[i][pind];
                        jac[findex, 1] += (-lambda * h / (h - z)) * weight; // y.
                        jac[findex, 2] += (-lambda * h * (y - c.y) / ((z - h) * (z - h))) * weight; // z.
                    }
                    findex++;
                }
            }
        }

        public Vector3d Optimize(ShadowOptimizationData data)
        {
            this.hs = data.heights;
            this.lps = data.lightpositions;
            this.sprofiles = data.shadowprofiles;
            this.spolygons = data.shadowpolygons;

            double mincost = double.MaxValue;
            Vector3d minlp = new Vector3d();
            for (int i = 0; i < this.lps.Length; ++i)
            {
                List<Vector3d> optimpositions = new List<Vector3d>();

                Vector3d lp = this.lps[i];
                double[] x = new double[] { lp.x, lp.y, lp.z };
                double epsg = 0.0000000001;
                double epsf = 0;
                double epsx = 0;
                int maxits = 0;

                double err = this.ComputeLambda(x);
                Program.OutputText(string.Format("{0}", err), true);
                int numpts = this.sprofiles.Sum(pts => pts.Length);

                double[] xinitial = (double[])x.Clone();
                double[] finitial = new double[numpts * 2];
                function1_fvec(xinitial, finitial, null);
                double initialsum = finitial.Sum(val => val * val);
                Program.OutputText(string.Format("{0}", initialsum), true);
                Program.OutputText("", true);

                optimpositions.Add(lp);

                // first run
                alglib.minlmstate state;
                alglib.minlmreport rep;

                alglib.minlmcreatevj(numpts * 2, x, out state);
                alglib.minlmsetcond(state, epsg, epsf, epsx, maxits);
                alglib.minlmoptimize(state, function1_fvec, function1_jac, null, null);
                alglib.minlmresults(state, out x, out rep);

                optimpositions.Add(new Vector3d(x[0], x[1], x[2]));


                int iters = 0;
                double dist = 0;
                double oldf = double.MinValue;
                double newf = double.MinValue;
                do
                {
                    double[] xtemp = (double[])x.Clone();
                    double[] fitemp = new double[numpts * 2];
                    function1_fvec(xtemp, fitemp, null);
                    newf = fitemp.Sum(val => val * val);
                    dist = (oldf == double.MinValue) ? 100 : Math.Abs(newf - oldf);
                    oldf = newf;
                    iters++;

                    //Program.OutputText(string.Format("{0}", rep.terminationtype), true); // EXPECTED: 4
                    //Program.OutputText(string.Format("{0}", alglib.ap.format(x, 2)), true);
                    //Program.OutputText(string.Format("{0}", newf), true);
                    //Program.OutputText(string.Format("{0}", iters), true);

                    this.ComputeLambda(x);

                    alglib.minlmrestartfrom(state, x);
                    alglib.minlmoptimize(state, function1_fvec, function1_jac, null, null);
                    alglib.minlmresults(state, out x, out rep);

                    optimpositions.Add(new Vector3d(x[0], x[1], x[2]));


                    fitemp = new double[numpts * 2];
                    function1_fvec(xtemp, fitemp, null);
                    newf = fitemp.Sum(val => val * val);
                    dist = (oldf == double.MinValue) ? 100 : Math.Abs(newf - oldf);
                    oldf = newf;

                    //Program.OutputText(string.Format("{0}", newf), true);
                    //Program.OutputText("", true);

                } while (dist > 1e-20 && iters < 200);

                double cost = newf;
                if (cost < mincost)
                {
                    mincost = cost;
                    minlp = new Vector3d(x[0], x[1], x[2]);
                }


                this.optimizedpositions.Add(optimpositions.ToArray());
            }

            return minlp;
        }

        private double ComputeLambda(double[] xarr)
        {
            double errsum = 0;

            double x = xarr[0];
            double y = xarr[1];
            double z = xarr[2];

            // update polygon.
            Vector2d lg = new Vector2d(x, y);
            double lh = z;
            for (int i = 0; i < hs.Length; ++i)
            {
                double h = this.hs[i];
                for (int j = 1; j <= 3; ++j)
                {
                    int index = this.spolyinds[j];
                    Vector2d c = this.spolygons[i][index];
                    this.spolygons[i][j] = (c - lg) * h / (lh - h) + c;
                }
            }

            // weigths, lambdas, lambda indices.
            this.weights = new double[this.sprofiles.Length][];
            this.lambdas = new double[this.sprofiles.Length][];
            this.lambdaindices = new int[this.sprofiles.Length][];
            for (int i = 0; i < this.sprofiles.Length; ++i)
            {
                Vector2d[] profile = this.sprofiles[i];
                Vector2d[] polygon = this.spolygons[i];
                this.weights[i] = new double[profile.Length];
                this.lambdas[i] = new double[profile.Length];
                this.lambdaindices[i] = new int[profile.Length];
                for (int j = 0; j < profile.Length; ++j)
                {
                    Vector2d p = profile[j];
                    bool inpoly = Shape2D.PointInPoly(p, polygon);
                    this.weights[i][j] = inpoly ? 1 : OutWeight;
                    int segindex;
                    double lambda;
                    double dist = Shape2D.Point2PolyBoundaryDist(p, polygon, out segindex, out lambda);
                    errsum += dist * dist * this.weights[i][j] * this.weights[i][j];
                    if (lambda > 1) lambda = 0.999999; else if (lambda < 0) lambda = 0.000001;
                    this.lambdas[i][j] = lambda;
                    this.lambdaindices[i][j] = segindex;
                }
            }
            return errsum;
        }
    }
}
