﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Drawing;
using System.IO;
using System.Linq;
using System.Text;

using MyGeometry;
using Tao.OpenGl;
using Tao.Platform.Windows;

using Emgu.CV;
using Emgu.CV.Structure;
using Emgu.CV.UI;
using System.Drawing.Imaging;

namespace i3_ImageManipulator
{
	public unsafe partial class ImageManipulator : IDisposable
	{
		// ----------------------------------------------------------------------------
		// 

		// load and save ultilities
		public void ReadImage(string filename)
		{
			this.image = new Image<Bgr, byte>(filename);
			this.imgwidth = this.image.Width;
			this.imgheight = this.image.Height;
			this.imageview = new ImageViewer();
			this.imagelabel = new int[this.imgheight, this.imgwidth];
			this.CreateGLTxture(this.image, layertxtid);
			this.parentview.SetBallBounds(this.imgwidth, this.imgheight);
			this.SetGLViewPort(this.imgwidth, this.imgheight);
		}
		public void SaveImage(string filename)
		{
			this.image.Save(filename);
		}


		// this function generates gl textures for proxies when downloaded, possibly with alpha matting
		private void GenerateGLTextures()
		{
			foreach (Polygon plg in this.allpolygons)
			{
				Image<Bgra, byte> img = plg.texture;
				
				if (img == null) continue;

				byte[] txture = this.Image2Texture(img);
				uint[] texid = new uint[1];
				Gl.glGenTextures(1, texid);	// Create The Texture
				
				// Typical Texture Generation Using Data From The Bitmap
				Gl.glBindTexture(Gl.GL_TEXTURE_2D, texid[0]);
				Gl.glTexImage2D(Gl.GL_TEXTURE_2D, 0, 4, (int)img.Width, (int)img.Height, 0, Gl.GL_RGBA, Gl.GL_UNSIGNED_BYTE, txture);
				Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_MIN_FILTER, Gl.GL_LINEAR);
				Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_MAG_FILTER, Gl.GL_LINEAR);

				Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_WRAP_S, Gl.GL_CLAMP_TO_EDGE);
				Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_WRAP_T, Gl.GL_CLAMP_TO_EDGE);
				Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_WRAP_R, Gl.GL_CLAMP_TO_EDGE);

				plg.textid = texid;
			}
		}
		// this function generates gl textures for proxies online, with alpha
		public void ProjectTexture2Proxies()
		{
			foreach (Polygon plg in this.allpolygons)
			{
				Image<Bgra, byte> txt = this.WarpPolygonRegion2AlphaTexture(this.image, plg);

				plg.texture = txt;

				byte[] txture = this.Image2Texture(txt);
				uint[] texid = new uint[1];
				
				Gl.glGenTextures(1, texid);	// Create The Texture
				// Typical Texture Generation Using Data From The Bitmap
				Gl.glBindTexture(Gl.GL_TEXTURE_2D, texid[0]);
				Gl.glTexImage2D(Gl.GL_TEXTURE_2D, 0, 4, (int)txt.Width, (int)txt.Height, 0, Gl.GL_RGBA, Gl.GL_UNSIGNED_BYTE, txture);
				Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_MIN_FILTER, Gl.GL_LINEAR);
				Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_MAG_FILTER, Gl.GL_LINEAR);

				Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_WRAP_S, Gl.GL_CLAMP_TO_EDGE);
				Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_WRAP_T, Gl.GL_CLAMP_TO_EDGE);
				Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_WRAP_R, Gl.GL_CLAMP_TO_EDGE);

				plg.textid = texid;

			}
		}


		// initial entry
		public void Init()
		{
			this.UnpackAllData();
			return;
		}

		private void CameraReady()
		{
			Program.OutputText("Loaded camera file successful ...", true);
			this.cameracalibrator.ComputeParameters();
			this.cameracalibrator.GetGLMatrices(out this.glprojectionmatrix, out this.glmodelviewmatrix,
				this.glviewportw, this.glviewporth, 0.01, 100);
			this.vp2d = this.cameracalibrator.GetVPs();
		}
		private void SegmentationReady()
		{
			this.FindSegmentsPolygonHulls();
			this.GetSegmentsPixelPos();
		}
		private void ProcessSegmentation(Image<Gray, byte> img)
		{
			using (MemStorage storage = new MemStorage())
			{
				List<Point[]> contourpoints = new List<Point[]>();
				List<List<Point2>> polygons = new List<List<Point2>>();
				for (Contour<Point> contours = img.FindContours(); contours != null;
					contours = contours.HNext)
				{
					contourpoints.Add(contours.ToArray());

					// get the approximating polygon
					Contour<Point> polygon = contours.ApproxPoly(5);
					Point[] polypoints = polygon.ToArray();
					List<Point2> poly = new List<Point2>();
					foreach (Point p in polypoints)
					{
						poly.Add(new Point2(Point2Vecter2d(p)));
					}
					polygons.Add(poly);
				}
				for (int i = 0; i < contourpoints.Count; ++i)
				{
					Point[] contour = contourpoints[i];
					contourpoints[i] = contour;
				}

				// get object approximate polygons, the largest one
				List<Point2> largest = null;
				foreach (List<Point2> poly1 in polygons)
				{
					bool find = true;
					foreach (Point2 pt in poly1)
					{
						foreach (List<Point2> poly2 in polygons)
						{
							if (poly2 == poly1) continue;
							if (Shape2D.PointInPoly(pt.pos, poly2))
							{
								find = false;
								continue;
							}
						}
						if (find == false)
							continue;
					}
					if (find)
						largest = poly1;
				}
				if (largest == null)
				{
					int n = 0;
					foreach (List<Point2> poly in polygons)
					{
						if (n < poly.Count)
						{
							n = poly.Count;
							largest = poly;
						}
					}
				}

				bool needallpolygons = true;
				if (!needallpolygons)
				{

					Polygon plg = new Polygon(largest);
					this.objectpolygons.Add(new Polygon[1] { plg });
				}
				else
				{
					int index = polygons.IndexOf(largest);
					List<Point2> tmp = polygons[0];
					polygons[0] = largest;
					polygons[index] = tmp;
					List<Polygon> polys = new List<Polygon>();
					foreach (List<Point2> poly1 in polygons)
					{
						Polygon plg = new Polygon(poly1);
						polys.Add(plg);
					}
					this.objectpolygons.Add(polys.ToArray());
				}

				// get object profiles
				List<Point2[]> contrs = new List<Point2[]>();
				foreach (Point[] points in contourpoints)
				{
					List<Point2> pts2 = new List<Point2>();
					foreach (Point p in points)
					{
						pts2.Add(new Point2(Point2Vecter2d(p)));
					}
					contrs.Add(pts2.ToArray());
				}
				this.objectprofiles.Add(contrs);
				this.AssignImagePixelLabel(img, this.objectprofiles.Count);
			}
		}
		private void SceneReady()
		{
			Program.OutputText("Loaded room configuration file successful ...", true);
			foreach (PolyProxy proxy in this.polyproxies)
			{
				this.FinalizeProxy(proxy);
			}
			this.GetAllPolygons();
			this.ProjectPixels2Proxies(); // get txtquad
			this.SetTranslationCenter();
			this.CreateProxyEditMetaphors();
			this.CreatePropagationPoints();
			this.GroupProxies();
			this.GetEnvironmentObjAvgSize(); // for insertion and replacement
			this.CreateNonLocalEditor();
			this.ComputeProxyRenderSize();
		}
		private void LayerReady()
		{
			this.resultimage = this.imlayer;
			if (this.resultimage == null)
				this.resultimage = this.image;
			this.CreateGLTxture(this.resultimage, layertxtid);
		}
		private void ShadowReady()
		{
			Program.OutputText("Loaded light file successful ...", true);
			this.islightinfoloaded = true;
			this.InitShadowParameters();
			this.ShadowCalculationWithLoadedLightInfo();
		}
		private void TextureReady()
		{
			Program.OutputText("Loaded texture files successful ...", true);
			this.GenerateGLTextures();
			this.RenderBoxWithTexutreMap = true;
		}
		private void TextureNotReady()
		{
			Program.OutputText("--- fail to load texture files ..., will map gl textures", true);
			if (has3dobjects)
			{
				this.ProjectTexture2Proxies();
				this.RenderBoxWithTexutreMap = true;
			}
		}


		
		private void GroupProxies()
		{
			int n = this.polyproxies.Count;
			bool[] tag = new bool[n]; int gid = 0;
			for (int i = 0; i < n; ++i)
			{
				if (tag[i] == true) continue;

				PolyProxy px = this.polyproxies[i];
				int index = px.segindex;
				List<PolyProxy> group = new List<PolyProxy>();
				group.Add(px);
				tag[i] = true;
				for (int j = i + 1; j < n; ++j)
				{
					PolyProxy px2 = this.polyproxies[j];
					if (px2.segindex == index)
					{
						group.Add(px2);
						tag[j] = true;
					}
				}
				ProxyGroup g = new ProxyGroup(group, gid++);
				foreach (PolyProxy proxy in g.proxies)
					proxy.groupindex = g.groupindex;
				this.proxygroups.Add(g);
			}
		}



		private List<PolyProxy> oldproxies = null;
		public void StoreScene()
		{
			this.oldproxies = new List<PolyProxy>();
			foreach (PolyProxy px in this.polyproxies)
			{
				PolyProxy qx = px.Clone() as PolyProxy;
				this.FinalizeProxy(qx);
				this.oldproxies.Add(qx);
			}
		}
		public void RestoreScene()
		{
			this.polyproxies = new List<PolyProxy>();
			foreach (PolyProxy px in this.oldproxies)
			{
				PolyProxy qx = px.Clone() as PolyProxy;
				this.FinalizeProxy(qx);
				this.polyproxies.Add(qx);
			}

			this.GetAllPolygons();
			this.CreateProxyEditMetaphors();
			this.CreatePropagationPoints();
			this.GroupProxies();
			this.CreateNonLocalEditor();
			this.resultimage = this.imlayer.Copy();
		}

		
		public void UnpackAllData()
		{
			string dir = Path.GetDirectoryName(imagename);
			string name = Path.GetFileNameWithoutExtension(imagename);
			bool succeed = true;
			string datapath = Path.Combine(dir, "data\\" + name + ".data");
			this.vp2d = new Vector2d[3];
			if (this.cameracalibrator == null) this.cameracalibrator = new CameraCalibrator();
			if (File.Exists(datapath))
			{
				FileStream fstream = new FileStream(datapath, FileMode.Open);
				BinaryReader br = new BinaryReader(fstream);
				try
				{
					// 1. camera
					this.UnpackCameraData(br);
					this.CameraReady();
					// 2. segments.
					this.UnpackSegmentation(br);
					this.SegmentationReady();
					// 2. config
					this.UnpackConfiguration(br);
					this.SceneReady();
					// 3. layer
					this.UnpackImageLayer(br);
					this.LayerReady();
					// 4. textures
					this.UnpackTextures(br);
					this.TextureReady();
					// 5. light
					this.UnpackLight(br);
					this.ShadowReady();

					// 6.
					this.StoreScene();
				}
				catch (Exception ex)
				{
					Program.OutputText(ex.Message, true);
					succeed = false;
				}
				finally
				{
					br.Close();
				}
			}
			else
			{
				succeed = false;
			}

			if (succeed)
				Program.OutputText("Unpacking all data succeedeed ...", true);
			else
				Program.OutputText("Unpacking all data failed ...", true);
		}
		private void UnpackLight(BinaryReader br)
		{
			double x = br.ReadDouble();
			double y = br.ReadDouble();
			double z = br.ReadDouble();
			this.optimlightposition = new Vector3d(x, y, z);
			this.LightSize = optimlightposition.z / 10;
			Program.OutputText("Unpacking light succeedeed ...", true);
		}
		private static Image<Bgr, byte> Bytes2ImageBgr(byte[] buffer)
		{
			MemoryStream stream = new MemoryStream(buffer);
			Bitmap bmp = new Bitmap(stream);
			Image<Bgr, byte> image = new Image<Bgr, byte>(bmp);
			bmp.Dispose();
			return image;
		}
		private static Image<Bgra, byte> Bytes2ImageBgra(byte[] buffer)
		{
			MemoryStream stream = new MemoryStream(buffer);
			Bitmap bmp = new Bitmap(stream);
			Image<Bgra, byte> image = new Image<Bgra, byte>(bmp);
			bmp.Dispose();
			return image;
		}
		private static Image<Gray, byte> Bytes2ImageGray(byte[] buffer)
		{
			MemoryStream stream = new MemoryStream(buffer);
			Bitmap bmp = new Bitmap(stream);
			Image<Gray, byte> image = new Image<Gray, byte>(bmp);
			bmp.Dispose();
			return image;
		}
		private void UnpackTextures(BinaryReader br)
		{
			bool istextureread = true;
			try
			{
				foreach (PolyProxy proxy in this.polyproxies)
				{
					foreach (Polygon plg in proxy.polygons)
					{
						int num = br.ReadInt32();
						byte[] buffer = br.ReadBytes(num);
						plg.texture = Bytes2ImageBgra(buffer);
					}
				}
			}
			catch (IndexOutOfRangeException ex)
			{
				Program.OutputText(ex.Message, true);
				istextureread = false;
			}

			if (istextureread)
				Program.OutputText("Read textures successful ...", true);
			else
				Program.OutputText("Read textures failed!!!", true);
		}
		private void UnpackImageLayer(BinaryReader br)
		{
			int num = br.ReadInt32();
			byte[] buffer = br.ReadBytes(num);
			this.imlayer = Bytes2ImageBgr(buffer);
		}
		private void UnpackConfiguration(BinaryReader br)
		{
			int totallength = br.ReadInt32();
			this.finalized = br.ReadBoolean();

			// proxies.
			int proxycount = br.ReadInt32();
			for (int index = 0; index < proxycount; ++index)
			{
				int n = br.ReadInt32();
				double depth = br.ReadDouble();
				int segindex = br.ReadInt32();
				List<Point2> pts2 = new List<Point2>();
				List<Point3> pts3 = new List<Point3>();
				for (int i = 0; i < n; ++i)
				{
					double x = br.ReadDouble();
					double y = br.ReadDouble();
					Point2 p = new Point2(new Vector2d(x, y));
					pts2.Add(p);
				}
				for (int i = 0, j = 0; i < n; ++i, j += 3)
				{
					double x = br.ReadDouble();
					double y = br.ReadDouble();
					double z = br.ReadDouble();
					Point3 p = new Point3(new Vector3d(x, y, z));
					pts3.Add(p);
				}
				PolyProxy proxy = new PolyProxy(pts2, pts3);
				proxy.depth = depth;
				proxy.segindex = segindex;
				if (proxy.MakeBox() == false && proxy.MakePlanarQuad() == false)
				{
					Program.OutputText("@ReadRoomConf::unknow proxy type!", true);
				}
				if (index == 0)
				{
					
				}
				else
				{
					this.polyproxies.Add(proxy);
				}
			}

			// on-objects relations.
			int onobjectcount = br.ReadInt32();
			for (int index = 0; index < onobjectcount; ++index)
			{
				int k = br.ReadInt32();
				int j = br.ReadInt32();
				List<PolyProxy> objs = new List<PolyProxy>();
				for (int i = 0; i < k; ++i)
				{
					int l = br.ReadInt32();
					objs.Add(this.polyproxies[l - 1]);
				}
				this.polyproxies[j - 1].onobjects = objs;
			}

			// joints.
			int jointcount = br.ReadInt32();
			for (int index = 0; index < jointcount; ++index)
			{
				int k = br.ReadInt32();
				int j = br.ReadInt32();
				int faceindex1 = br.ReadInt32();
				int lineindex1 = br.ReadInt32();
				int faceindex2 = br.ReadInt32();
				int lineindex2 = br.ReadInt32();
				PolyProxy p1 = this.polyproxies[k];
				PolyProxy p2 = this.polyproxies[j];
				p1.jointproxy = p2;
				p2.jointproxy = p1;
				p1.isjointhoster = true;
				p2.isjointhoster = false;
				p1.jointline = p1.polygons[faceindex1].linesegments[lineindex1];
				p2.jointline = p2.polygons[faceindex2].linesegments[lineindex2];
				p1.type = p2.type = ProxyType.JOINT;
			}

			this.AssignProxyIndex();
			this.has3dobjects = this.polyproxies.Count > 0;
			this.RenderSolidProxies = false;
			Program.OutputText("Unpacking room configuration data succeeded ...", true);
		}
		private void UnpackCameraData(BinaryReader br)
		{
			this.cameracalibrator.UnpackCameraData(br);
		}
		private void UnpackSegmentation(BinaryReader br)
		{
			int n = br.ReadInt32();
			List<Image<Gray, byte>> segments = new List<Image<Gray, byte>>();
			for (int i = 0; i < n; ++i)
			{
				int num = br.ReadInt32();
				byte[] buffer = br.ReadBytes(num);
				segments.Add(Bytes2ImageGray(buffer));
			}
			foreach (Image<Gray, byte> seg in segments)
			{
				this.ProcessSegmentation(seg);
			}
		}

		private static byte[] Image2Bytes(Image<Bgr, byte> image)
		{
			MemoryStream stream = new MemoryStream();
			image.Bitmap.Save(stream, ImageFormat.Png);
			byte[] bytes = stream.GetBuffer();
			return bytes;
		}
		private static byte[] Image2Bytes(Image<Bgra, byte> image)
		{
			MemoryStream stream = new MemoryStream();
			image.Bitmap.Save(stream, ImageFormat.Png);
			byte[] bytes = stream.GetBuffer();
			return bytes;
		}
		private static byte[] Image2Bytes(Image<Gray, byte> image)
		{
			MemoryStream stream = new MemoryStream();
			image.Bitmap.Save(stream, ImageFormat.Png);
			byte[] bytes = stream.GetBuffer();
			return bytes;
		}

	}
}