﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Linq;
using System.Collections;	// -- hashtable 
using System.ComponentModel;
using Tao.OpenGl;
using Tao.Platform.Windows;
using MyGeometry;
using System.Drawing;
using System.IO;
using System.Diagnostics;

using System.Drawing.Imaging;

using Emgu.CV;
using Emgu.CV.Structure;
using Emgu.CV.UI;


namespace i3_ImageManipulator
{
    [TypeConverterAttribute(typeof(DeformerConverter))]
    public unsafe partial class ImageManipulator : IDisposable
    {
		// variables, members
		public byte TabIndex = 0;
		public ImageView parentview = null;
		public CameraCalibrator cameracalibrator = null;
		public Color BoxSelectColor { get; set; }
		public Color ProxyColor { get; set; }
		public bool renderLayerImage = true;
		public bool RenderLayerImage {
			get { return renderLayerImage; }
			set {
				renderLayerImage = value; 
				if (parentview != null) 
					parentview.Refresh(); 
			} 
		}
		public bool RenderBoxWithTexutreMap { get; set; }
		public bool RenderSolidProxies { get; set; }
		public bool RenderLineProxies { get; set; }
		public bool TextureAllPolygons { get; set; }
		public float FlatProxyLineWidth { get; set; }
		public float TranslateScaleFactor { get; set; }
		private bool isselectfixed = false;
		private double proxyrendersize = 0.015;
		public bool ShadowOn = false;
		public int RotationSlices = 800;
		public double ballsize { get; set; }
		public double arrowsize { get; set; }
		public bool finalized = false;
		private Color[] colorMall = null;
		public Color[] ColorMall
		{
			get { return colorMall; }
			set { colorMall = value; }
		}
        private Vector2d imgpos = new Vector2d(0, 0);
		private Vector2d oldimgpos = new Vector2d(0, 0);
		private Vector2d[] vp2d = null;
		private Vector3d objavgsize;
		public string imagename;

		private Point2 selectionstartpoint = null, selectionendpoint = null;

		private PolyProxy selectedproxy = null;
		public List<PolyProxy> selectedproxies = new List<PolyProxy>();
		private List<Polygon> selectedfaces = new List<Polygon>();
		private Metaphor selectedmetaphor = null;
		private RigidTransMetaphor rigidmetaphor = null;
		private Polygon selectedpolyface = null;


        private int imgwidth = 0;
        private int imgheight = 0;
        private int imgchannel = 4;
		private int glviewportw = 800;
		private int glviewporth = 600;
		private double[] glmodelviewmatrix = null;
		private double[] glprojectionmatrix = null;
		private Matrix4d roomtrans = null, roomtrans_inv = null;
		private Vector3d lightpos = new Vector3d(5, 5, 5);
		private List<Polygon> hexagons = null;
		private List<Polygon> allpolygons = null;
		private List<PolyProxy> polyproxies = new List<PolyProxy>();
		public List<ProxyGroup> proxygroups = new List<ProxyGroup>();
		private List<PolyProxy> importedproxies = new List<PolyProxy>();
		private List<Hull> alignedhulls = new List<Hull>();
		private List<List<Point2[]>> objectprofiles = new List<List<Point2[]>>();
		private List<Polygon[]> objectpolygons = new List<Polygon[]>();
		private List<Point2[]> objectconvexhulls = null; 
        private uint[] layertxtid = new uint[1];
		private double sacleRatio = 1.0;
		private double whomogenous = 1;
		private Vector3d draggingpos;
		private NonLocalEditor editor = null;
		private bool has3dobjects = false;
		private int currinsertingindex = -1;
		private bool disablepropatation = false;
		public Matrix4d CurrentTransformation = Matrix4d.IdentityMatrix();

		// opencv utilities
		public Image<Bgr, byte> Image { get { return this.image; } }
		private Image<Bgr, byte> image = null;
		private Image<Bgr, byte> imlayer = null;
		private Image<Bgr, byte> resultimage = null;
		private List<Image<Gray, byte>> alphamasks = new List<Image<Gray, byte>>();
		private List<Image<Bgr, byte>> foregrounds = new List<Image<Bgr, byte>>();
		private int[,] polylabelmap = null;
		private int[,] imagelabel = null; // useage: imagelabel[y,x] ;)
		private List<List<Vector2d>> segments = null;
		private ImageViewer imageview = null;


		// constructor
		public ImageManipulator()
		{
			this.InitialRandomColors(36);
			BoxSelectColor = Color.Lime;
			ProxyColor = Color.Yellow;
			this.FlatProxyLineWidth = 1.0f;
			this.TranslateScaleFactor = 10;
			this.ballsize = 0.05;
			this.arrowsize = 0.009;
		//	this.RenderLineProxies = true;
			this.RenderSolidProxies = false;
		}


		// functionalities for object replacements
		public void Paste(List<PolyProxy> objs, Vector2d mousepos)
		{
			// this function paste proxies from another scene to current scene
			// if no object is selected in the current scene, the objects will
			// be pasted on the ground, otherwise replacements are performed
			if (objs == null || objs.Count == 0) return;

			if (this.selectedproxy != null)
			{
				// do replacement
				this.Replace(this.selectedproxy, objs);
				this.selectedproxies.Clear();
				this.selectedproxy = null;
				this.selectedproxies.Clear();
				this.selectedproxies.AddRange(objs);
			}
			else
			{
				Vector3d pos = this.cameracalibrator.ComputePointGround3dPos(mousepos);
				this.Insert(objs, pos);
			}
			this.GroupProxies();
			this.CreateNonLocalEditor();
			this.disablepropatation = true;
		}
		private void Replace(PolyProxy original, List<PolyProxy> objs)
		{
			// replace p1 with p2

			PolyProxy p1 = original, p2 = objs[0];
			List<PolyProxy> buddy = this.GetAllBroProxies(original);
		//	RepContainer bbox1 = CreateBigBox(buddy);
		//	RepContainer bbox2 = CreateBigBox(objs);

			Vector3d[] cs1 = GetBoundingBoxCenterAndScale(buddy);
			Vector3d[] cs2 = GetBoundingBoxCenterAndScale(objs);

			bool switched = false;
			double s1, s2, s3;
			double x1 = cs1[1].x, y1 = cs1[1].y;
			double x2 = cs2[1].x, y2 = cs2[1].y;
			if ((x1 - y1) * (x2 - y2) < 0) // switch x and y scale
			{
				switched = true;
				s2 = x1 / y2;
				s1 = y1 / x2;
			}
			else
			{
				s1 = x1 / x2;
				s2 = y1 / y2;
			}
			s3 = cs1[1].z / cs2[1].z;

			Matrix4d P = new Matrix4d(p2.frame.R_inv);
			P[3, 3] = 1;
			Matrix4d S = Matrix4d.ScalingMatrix(s1, s2, s3);
			Matrix4d Q = P.Inverse() * S * P;

			// do some alignment accordingly
			Vector3d axis1 = p1.frame.x.Normalize();
			Vector3d axis2 = p2.frame.x.Normalize();
			if (switched) axis2 = p2.frame.y.Normalize();
			Vector3d rotaxis = (axis2.Cross(axis1)).Normalize();
			if (double.IsNaN(rotaxis.x))
				rotaxis = new Vector3d(0, 0, 1);
			double cos = Math.Abs(axis1.Dot(axis2));
			if (cos > 1) cos = 1;
			double rotangle = Math.Acos(cos);
			Matrix4d R = Matrix4d.RotationMatrix(rotaxis, rotangle);
			Vector3d t = (cs1[0] - cs2[0]);
			t.z = 0;
			Matrix4d T = Matrix4d.TranslationMatrix(t);

			foreach (PolyProxy pp in objs)
			{
				DeformProxy(pp, T*R*Q, cs2[0], 1);
				this.UpdateProxy(pp, null);
				this.CreateProxyEditMetaphors(pp);
			}

			// align to the ground
			double minz = double.MaxValue; Point3 lowest = null;
			foreach (PolyProxy px in objs)
			{
				foreach (Point3 p3 in px.points3)
				{
					if (p3.pos.z < minz)
					{
						lowest = p3;
						minz = p3.pos.z;
					}
				}
			}
			Vector3d tt = new Vector3d(0, 0, -minz);
			foreach (PolyProxy px in objs)
			{
				foreach (Point3 p3 in px.points3)
				{
					p3.pos += tt;
				}
				UpdateProxy(px,null);
			}

			// assign segindex
			int count = this.polyproxies.Count;
			foreach (PolyProxy px in objs)
			{
				px.segindex = currinsertingindex;
			}
			currinsertingindex--;

			this.CreatePropagationPoints(objs);
			foreach (PolyProxy px in objs)
				this.CreateProxyEditMetaphors(px);
			//
			foreach (PolyProxy px in buddy)
				this.polyproxies.Remove(px);
			objs[0].onobjects = original.onobjects;
			this.polyproxies.AddRange(objs);
		}
		private void Insert(List<PolyProxy> objs, Vector3d pos)
		{
			Vector3d[] s = GetBoundingBoxCenterAndScale(objs);
			Vector3d ss = this.objavgsize;
			Vector3d scale = new Vector3d(ss.x/ s[1].x, ss.y/s[1].y, ss.z/s[1].z);
			double sc = Math.Min(scale.x, Math.Min(scale.y, scale.z));
			Matrix4d S = Matrix4d.ScalingMatrix(sc, sc, sc);
			Matrix4d P = new Matrix4d(objs[0].frame.R_inv);
			P[3, 3] = 1;
			Matrix4d Q = P.Inverse() * S * P;
			Vector3d t = pos - s[0];
			t.z = 0;
			Matrix4d T = Matrix4d.TranslationMatrix(t);
			Matrix4d M = T * Q;
			foreach (PolyProxy px in objs)
			{
				DeformProxy(px, M, s[0], 1);
				UpdateProxy(px, null);
				this.CreateProxyEditMetaphors(px);
			}

			// align to the ground
			double minz = double.MaxValue; Point3 lowest = null;
			foreach (PolyProxy px in objs)
			{
				foreach (Point3 p3 in px.points3)
				{
					if (p3.pos.z < minz)
					{
						lowest = p3;
						minz = p3.pos.z;
					}
				}
			}
			Vector3d tt = new Vector3d(0, 0, -minz);
			foreach (PolyProxy px in objs)
			{
				foreach (Point3 p3 in px.points3)
				{
					p3.pos += tt;
				}
				UpdateProxy(px, null);
			}

			// assign segindex
			int count = this.polyproxies.Count;
			foreach (PolyProxy px in objs)
			{
				px.segindex = currinsertingindex;
			}
			currinsertingindex--;

			foreach (PolyProxy px in objs)
				this.CreateProxyEditMetaphors(px);
			this.polyproxies.AddRange(objs);
		}
		private void GetEnvironmentObjAvgSize()
		{
			Vector3d scale = new Vector3d();
			foreach (ProxyGroup g in this.proxygroups)
			{
				Vector3d[] s = GetBoundingBoxCenterAndScale(g.proxies);
				scale += s[1];
			}
			this.objavgsize = scale / this.proxygroups.Count;
		}
		

		// mem functions
		public void EnlargeSacle()
		{
			sacleRatio /= 0.9;
		}
		public void ReduceSacle()
		{
			sacleRatio *= 0.9;
			if (sacleRatio < 0.1094189891315125)
			{
				sacleRatio = 0.1094189891315124;
			}
		}
		public Vector2d ImgPos
		{
			get { return imgpos; }
			set { imgpos = value; }
		}
		public void CopyImgPos()
		{
			this.oldimgpos = new Vector2d(imgpos.x, imgpos.y);
		}
		public Polygon CurrentSelectedFace { get { return this.selectedpolyface; } }


		// initial
		private void AssignImagePixelLabel(Image<Gray, Byte> imgseg, int label)
		{
			if (this.imagelabel == null) return;
			for (int i = 0; i < this.imgheight; ++i)
			{
				for (int j = 0; j < this.imgwidth; ++j)
				{
					if (imgseg[i, j].Intensity > 0)
					{
						this.imagelabel[i, j] = label;
					}
				}
			}
		}
		private void GetSegmentsPixelPos()
		{
			if (this.imagelabel == null) return;
			int n = this.objectprofiles.Count + 1;
			this.segments = new List<List<Vector2d>>();
			for (int i = 0; i < n; ++i)
			{
				this.segments.Add(new List<Vector2d>());
			}
			for (int i = 0; i < this.imgheight; ++i)
			{
				for (int j = 0; j < this.imgwidth; ++j)
				{
					int k = this.imagelabel[i, j];
					this.segments[k].Add(new Vector2d(j, i));
				}
			}
		}
		private void SetGLViewPort(int w, int h)
		{
			this.glviewportw = w;
			this.glviewporth = h;
		}		
		private void InitialRandomColors(int K)
		{
			Random rand = new Random();
			this.colorMall = new Color[K];
			for (int i = 0; i < K; ++i)
			{
				this.colorMall[i] = Color.FromArgb(
					rand.Next(0, 255),
					rand.Next(0, 255),
					rand.Next(0, 255)
				);
			}
			this.colorMall[3] = Color.FromArgb(197, 27, 125);
			this.colorMall[0] = Color.FromArgb(233, 163, 201);
			this.colorMall[4] = Color.FromArgb(241, 60, 45);
			this.colorMall[6] = Color.FromArgb(253, 224, 239);
			this.colorMall[2] = Color.FromArgb(230, 245, 208);
			this.colorMall[1] = Color.FromArgb(161, 215, 106);
			this.colorMall[5] = Color.FromArgb(80, 127, 218);
		}
		private void SetTranslationCenter()
		{
			Vector3d transcenter = new Vector3d();
			foreach (PolyProxy box in this.polyproxies)
			{
				transcenter += box.center.pos;
			}
			transcenter /= this.polyproxies.Count;
			this.roomtrans = Matrix4d.TranslationMatrix(transcenter);
			this.roomtrans_inv = Matrix4d.TranslationMatrix(new Vector3d() - transcenter);

			// set the light position
			Vector3d maxCoord = new Vector3d(double.MinValue, double.MinValue, double.MinValue);
		//	Vector3d minCoord = new Vector3d(double.MaxValue, double.MaxValue, double.MaxValue);
			foreach (PolyProxy box in this.polyproxies)
			{
				foreach (Point3 v in box.points3)
				{
					maxCoord = Vector3d.Max(maxCoord, v.pos);
			//		minCoord = Vector3d.Min(minCoord, v.pos);
				}
			}
			this.lightpos = 1.5 * maxCoord;
		}
		private Matrix3d ComputePolygonHomography(Polygon face)
		{
			if (face.localframe == null) face.ComputeLocalFrame();
			int n = face.points3.Count;
			Vector2d[] src = new Vector2d[n];
			Vector2d[] tar = new Vector2d[n];
			for (int i = 0; i < n; ++i)
			{
				src[i] = face.points[i].pos;
				tar[i] = face.localframe.GetPointLocalCoord(face.points3[i].pos).ToVector2d();
			}
			return CameraCalibrator.ComputeHomographyMatrixCV(src, tar);
		}
		private Vector3d ProjectScreenPoint2PolyFace(Vector2d pt, Polygon face)
		{
			if (face.H == null)
			{
				face.H = ComputePolygonHomography(face);
			}
			Matrix3d H = face.H;
			Vector3d planepoint = H * new Vector3d(pt, 1);
			planepoint.HomogenousNormalize();
			planepoint.z = 0;
			return (face.localframe.R_inv * planepoint + face.localframe.o);
		}
		private void GetActualGLPoint(ref Vector2d p)
		{
			p -= this.imgpos;
			p /= this.sacleRatio;
		}
		private void GetActualGLVector(ref Vector2d p)
		{
			p /= this.sacleRatio;
		}
		private void ReleaseSelection()
		{
			this.selectedpolyface = null;
			this.has3dobjects = true;
		}		
		private void GetProxyLineSegmentsP3(PolyProxy proxy)
		{
			foreach (Polygon plg in proxy.polygons)
			{
				plg.ComputeLinesegmentsP3();
			}
		}
		private void GetAllPolygons()
		{
			List<Polygon> plys = new List<Polygon>();
			foreach (PolyProxy px in this.polyproxies)
			{
				plys.AddRange(px.polygons);
			}
			int index = 0;
			foreach (Polygon plg in plys)
			{
				plg.globalindex = index++;
			}
			this.allpolygons = plys;
		}
		private void AssignProxyIndex()
		{
			int boxindex = 0;
			foreach (PolyProxy box in this.polyproxies)
			{
				box.index = boxindex;
				if (box.segindex == -1)
					box.segindex = (boxindex + 1);
				boxindex++;
			}
		}
		private void FindRotationBallSize()
		{
			double size = double.MinValue;
			foreach (PolyProxy px in this.polyproxies)
			{
				double s = Math.Max(Math.Max(px.xlen, px.ylen), px.zlen);
				if (s > 0 && s > size)
				{
					size = s;
				}
			}
			this.ballsize = size / 35;
		}
		private void ComputeProxyRenderSize()
		{
			if (this.cameracalibrator == null) return;
			Vector3d c = this.cameracalibrator.camera_R.Inverse() * this.cameracalibrator.camera_t;
			double l = c.Length();
			this.proxyrendersize = l * 1e-3;
		}


		/* algorithms */
		private void FindSegmentsPolygonHulls()
		{
			this.objectconvexhulls = new List<Point2[]>();
			using (MemStorage storage = new MemStorage()) {
				foreach (List<Point2[]> contours in this.objectprofiles) {
					Contour<Point> contpoints = new Contour<Point>(storage);
					foreach (Point2[] contour in contours) {
						List<Point> points = new List<Point>();
						foreach (Point2 pt in contour)
						{
							points.Add(Vector2Point(pt.pos));
						}
						contpoints.PushMulti(points.ToArray(), Emgu.CV.CvEnum.BACK_OR_FRONT.BACK);
					}
			//		Program.OutputText("contour poly points: " + contpoints.Total, true);
					Seq<Point> poly = contpoints.GetConvexHull(Emgu.CV.CvEnum.ORIENTATION.CV_COUNTER_CLOCKWISE, storage);

			//		Program.OutputText("convex hull points: " + poly.Total, true);
					//int index = 0; MCvFont font = new MCvFont(Emgu.CV.CvEnum.FONT.CV_FONT_HERSHEY_PLAIN, 2, 1); Bgr bgr = new Bgr(0, 255, 0);
					//foreach (Point p in poly)
					//{
					//    img.Draw(index++.ToString(), ref font, p, bgr);
					//}
					//img.Draw(poly, new Bgr(0, 255, 0), new Bgr(0, 255, 0), 0, 2);

					List<Point2> pts2 = new List<Point2>();
					foreach (Point pt in poly)
					{
						pts2.Add(new Point2(Point2Vecter2d(pt)));
					}
					this.objectconvexhulls.Add(pts2.ToArray());
				}
			}

			// draw
			//Image<Bgr, byte> img = this.image.Copy();
			//foreach (Point2[] pts in this.objectconvexhulls)
			//{
			//    this.DrawPolygon(img, pts);
			//}
			//ImageViewer vw = new ImageViewer(img, "obj convex hulls");
			//vw.Show();

			// get the box hexagons before we do the subdivision to the convex hulls
		//	this.PolygonsHexagonApproximate();

			// unify the points
			this.UniformObjConvexHulls();
		}
		private void UniformObjConvexHulls()
		{
			if (this.objectconvexhulls == null) return;
			double interval = 5;
			int k = this.objectconvexhulls.Count;
			for (int l = 0; l < k; ++l)
			{
				Point2[] pts = this.objectconvexhulls[l];
				List<Point2> newpts = new List<Point2>();
				int n = pts.Length;
				for (int i = 0; i < n; ++i)
				{
					Point2 p = pts[i], q = pts[(i + 1) % n];
					newpts.Add(p);
					int steps = (int)Math.Round((q.pos - p.pos).Length() / interval);
					if (steps < 2) continue;
					double lambda = 1.0 / steps; 
					for (int j = 1; j < steps; ++j)
					{
						double r = lambda * j;
						Vector2d t = q.pos*r + p.pos*(1-r);
						newpts.Add(new Point2(t));
					}
				}
				this.objectconvexhulls[l] = newpts.ToArray();
			}

			//Image<Bgr, byte> img = this.image.Copy();
			//foreach (Point2[] pts in this.objectconvexhulls)
			//{
			//    this.DrawPolygon(img, pts);
			//}
			//ImageViewer vw = new ImageViewer(img, "unified convex hulls");
			//vw.Show();
		}
		private Vector2d[] FindTangentPoints(Vector2d pt, Vector2d[] poly)
		{
			// find the two tangent points of the polygon according to the point pt
			List<Vector2d> tangentpoints = new List<Vector2d>();
			List<int> tangentsigns = new List<int>();
			int n = poly.Length;
			for (int i = 0; i < n; ++i) {
				Vector3d t = new Vector3d(poly[i],1).Cross(new Vector3d(pt, 1));
				Vector2d r = t.ToVector2d();
				bool find = true; double val = 0;
				for (int j = 0; j < n; ++j) {
					if (j == i) continue;
					val = poly[j].Dot(r) + t.z;
					break;
				}
				double tval = 0;
				for (int j = 0; j < n; ++j) {
					if (j == i) continue;
					double tt = poly[j].Dot(r) + t.z;
					tval += tt;
					if (Math.Abs(tt) > 1e-5 && tt * val < 0)
					{
						find = false;
						break;
					}
				}
				if (find) {
					int sign = (int)Math.Sign(tval);
					if (!tangentsigns.Contains(sign))
					{
						tangentpoints.Add(poly[i]);
						tangentsigns.Add(sign);
					}
				}
			}
			if (tangentpoints.Count != 2)
			{
				Program.OutputText("less than 2 tangent pints found!, vp has problems, ", true);
				return null;
			}
			return tangentpoints.ToArray();
		}
		private Vector3d GLUProject(Vector3d p)
		{
			int[] viewport = new int[4];// {0,0,this.imgwidth,this.imgheight};
			double x, y, z;
			Gl.glGetIntegerv(Gl.GL_VIEWPORT, viewport);
			Glu.gluProject(p.x, p.y, p.z, this.glmodelviewmatrix, this.glprojectionmatrix, 
				viewport, out x, out y, out z);
			return new Vector3d(x, y, z);
		}
		private Matrix3d GetHomography(Shape2D src, Shape2D tar)
		{
			if (src.points.Count != tar.points.Count) return null;
			int n = src.points.Count;
			Vector2d[] srcv2 = new Vector2d[n];
			Vector2d[] tarv2 = new Vector2d[n];
			for (int i = 0; i < n; ++i)
			{
				srcv2[i] = src.points[i].pos;
				tarv2[i] = tar.points[i].pos;
			}
			return CameraCalibrator.ComputeHomographyMatrixCV(tarv2, srcv2);
		}


		/* non-local edits */
		public void CreateNonLocalEditor()
		{
			this.editor = new NonLocalEditor(this.polyproxies,this.proxygroups, null);
		}
		public void InitEdit()
		{
			if (this.editor == null) {
				this.CreateNonLocalEditor();
			}
			this.editor.Init();
		}
		public void Propagate()
		{
			if (this.editor != null && !this.disablepropatation)
			{
				this.editor.Propagate(this.selectedproxy, this.selectedpolyface);
				this.polyproxies = this.editor.proxies;
			}
		}
		public void DeformUpdate()
		{
			if (this.selectedproxy == null) return;

			foreach (PolyProxy proxy in this.polyproxies)
			{
				foreach (Point3 p in proxy.points3)
					p.UpdateOldPos();
				proxy.Compute3DInfo();
				proxy.center.UpdateOldPos();
				// update propagating points
				foreach (Polygon plg in proxy.polygons)
				{
					plg.CreatePropagatingPoints();
					plg.center.UpdateOldPos();
					foreach (Point3 p in plg.propagatepoints3)
					{
						p.R = Matrix3d.IdentityMatrix();
						p.UpdateOldPos();
					}
				}

				// update the img position
				{
					int index = 0;
					foreach (Point3 p in proxy.points3)
					{
						proxy.points[index++].pos =
							this.cameracalibrator.ComputePointImgPos(p.pos);
					}
				}
				foreach (Polygon poly in proxy.polygons)
					poly.ComputeCenter();
			}
			
			this.CreateProxyEditMetaphors();
			this.selectedpolyface = null;
			this.selectedfaces.Clear();
		}


		/* texture maps */
		public void DoTxtureMapping()
		{
			if (this.ShadowOn)
			{
				this.resultimage = this.imlayer.Copy();
				this.DoShadowMapping();
			}
		}
		public void UpdateShadowStatus()
		{
			this.resultimage = this.imlayer.Copy();
			if (!this.ShadowOn)
			{
				this.CreateGLTxture(this.resultimage, this.layertxtid);
			}
			else
			{
				this.DoShadowMapping();
			}
		}

		private void FindProxiesDepth()
		{
			foreach (PolyProxy proxy in this.polyproxies)
			{
				Vector3d d =  this.GLUProject(proxy.center.pos);
				proxy.depth = d.z;
			}
		}
		private void GetTargetEnclosingTxtQuads()
		{
			foreach (PolyProxy proxy in this.polyproxies)
			{
				proxy.tarenclosingtxtquad = GetPointsEnclosingQuad(proxy.points);
			}
		}
		private void GetProxyTxtQuads()
		{
			foreach (PolyProxy proxy in this.polyproxies)
			{
				proxy.txtquads = new List<Polygon>();
				foreach (Polygon plg in proxy.polygons)
				{
					if (plg.visible) proxy.txtquads.Add(plg);
				}
			}
		}
		private void ProjectPixels2Proxies_Direct()
		{
			// this function project the image to the scene objects
			// each pixel is projected to the polygon faces as texel
			// require: depth test
			int txtindex = 1;
			// then, go through the image
			for (int i = 0; i < this.imgheight; ++i)
			{
				for (int j = 0; j < this.imgwidth; ++j)
				{
					int label = this.imagelabel[i, j];
					if (label != 0) // assign to the proxy faces
					{
						Vector2d p = new Vector2d(j,i);
						double mindepth = double.MaxValue; Polygon tarpolygon = null;
						foreach (PolyProxy proxy in this.polyproxies)
						{
							foreach (Polygon plg in proxy.polygons)
							{
								if (Shape2D.PointInPoly(p, plg))
								{
									double depth = this.GLUProject(plg.center.pos).z;
									if (depth < mindepth)
									{
										tarpolygon = plg;
										mindepth = depth;
									}
								}
							}
						}
						if (tarpolygon != null)
						{
							if (tarpolygon.textureindex == -1)
							{
								tarpolygon.textureindex = txtindex++;
							}
							this.imagelabel[i, j] = tarpolygon.textureindex;
						}
					}
				}
			}
			// find the proxy texture polygons based on their labels
			// to do..
		}
		private void ProjectPixels2Proxies()
		{
			// this function project the image to the scene objects
			// each pixel is projected to the polygon faces as texel
			// require: depth test
			if (this.segments == null) 
				this.ProjectPixels2Proxies_Direct();

			List<PolyProxy> allproxies = new List<PolyProxy>();
			foreach (PolyProxy proxy in this.polyproxies)
			{
				if (proxy.slicingboxes == null)
				{
					allproxies.Add(proxy);
				}
				else
				{
					allproxies.AddRange(proxy.slicingboxes);
				}
			}

			
			foreach (PolyProxy px in allproxies)
				foreach (Polygon plg in px.polygons)
					plg.textureindex = -1;

			int txtindex = 1;
			for (int i = 0; i < this.segments.Count; ++i)
			{
				List<Vector2d> seg = segments[i];
				List<PolyProxy> proxies = new List<PolyProxy>();
				foreach (PolyProxy box in allproxies)
				{
					if (box.segindex == i)
					{
						proxies.Add(box);
					}
				}				
				
				if (proxies.Count < 1) continue;
				List<Vector2d> residuals = new List<Vector2d>();
				int assignedcount = 0;
				foreach (Vector2d v in seg)
				{
					List<Polygon> plgs = new List<Polygon>();
					foreach (PolyProxy px in proxies)
					{
						foreach (Polygon plg in px.polygons)
						{
							if (Shape2D.PointInPoly(v, plg))
							{
								plgs.Add(plg);
							}
						}
					}
					Polygon tar = null;
					if (plgs.Count > 1)
					{
						double mindepth = double.MaxValue;
						foreach (Polygon plg in plgs)
						{
							Vector3d v3 = this.ProjectScreenPoint2PolyFace(v, plg);
							double depth = this.GLUProject(v3).z;
							if (mindepth > depth)
							{
								mindepth = depth;
								tar = plg;
							}
						}
					}
					else if (plgs.Count == 1)
						tar = plgs[0];
					if (tar != null)
					{
						if (tar.textureindex == -1)
						{
							tar.textureindex = txtindex++;
						}
						this.imagelabel[(int)v.y, (int)v.x] = tar.textureindex;
						assignedcount++;
					}
					else
					{
						residuals.Add(v);
					}
				}
				if (i == 0)
				{
					continue;
				}
				foreach (Vector2d v in residuals)
				{
					// find a cloest polygon
					double mindis = double.MaxValue; Polygon tar = null;
					foreach (PolyProxy px in proxies)
					{
						foreach (Polygon plg in px.polygons)
						{
							if (plg.textureindex == -1) continue; // only assign to those already assigned ones
							int sindex;
							double dis = Shape2D.Point2PolyBoundaryDist(v, plg.points, out sindex);
							if (dis < mindis)
							{
								mindis = dis;
								tar = plg;
							}
						}
					}
					if (tar != null && tar.textureindex != -1)
					{
						this.imagelabel[(int)v.y, (int)v.x] = tar.textureindex;
						assignedcount++;
					}
				//	else
				//		Program.OutputText("@projectPixels2Proxies::unassigned pixel: (" + v.x + " " + v.y + ")", true);
				}

				if (assignedcount != seg.Count)
					Program.OutputText("@projectPixels2Proxies: has unassigned pixel!", true);
			}
			// find proxy txtquads
			foreach (PolyProxy px in allproxies)
			{
				px.txtquads = new List<Polygon>();
				foreach (Polygon plg in px.polygons)
				{
					if (plg.textureindex != -1)
					{
						px.txtquads.Add(plg);
					}
				}
			}
		}
		private Image<Bgra, byte> WarpPolygonRegion2AlphaTexture(Image<Bgr, byte> srcimg, Polygon plg)
		{
			// get texture w and h
			int scale = 2;
			int width1 = (int)((plg.points[1].texcoord - plg.points[0].texcoord).Length()) * scale;
			int height1 = (int)((plg.points[2].texcoord - plg.points[1].texcoord).Length()) * scale;
			int width2 = (int)((plg.points[3].texcoord - plg.points[2].texcoord).Length()) * scale;
			int height2 = (int)((plg.points[3].texcoord - plg.points[0].texcoord).Length()) * scale;
			int w = width1 > width2 ? width1 : width2;
			int h = height1 > height2 ? height1 : height2;
			while (w < 10 || h < 10)
			{
				w *= 2;
				h *= 2;
			}

			// compute homography
			int ext = 0;
			Vector2d[] imgpts = new Vector2d[4] {
				new Vector2d(ext, ext),
				new Vector2d(w - ext, ext),
				new Vector2d(w - ext, h - ext),
				new Vector2d(ext, h - ext)
			};

			Vector3d n = plg.normal;
			List<Vector3d> p3s = plg.points3.Select(p => p.pos).ToList();
			Vector3d dir = (p3s[1] - p3s[0]).Cross(p3s[2] - p3s[1]).Normalize();
			List<Vector2d> p2s = plg.points.Select(p => p.texcoord).ToList();

			Vector2d[] refpts = new Vector2d[4] {
				p2s[0],
				p2s[1],
				p2s[2],
				p2s[3]
			};

			Matrix3d H = CameraCalibrator.ComputeHomographyMatrixCV(imgpts, refpts);
			Matrix<double> mat = new Matrix<double>(3, 3);
			for (int i = 0; i < 3; ++i)
				for (int j = 0; j < 3; ++j)
					mat[i, j] = H[i, j];


			// warp alpha mask and the foreground object image
			Image<Bgr, byte> fg = this.image;
			Image<Bgr, byte> fgwarp = fg.WarpPerspective<double>(mat, w, h, Emgu.CV.CvEnum.INTER.CV_INTER_CUBIC, 
				Emgu.CV.CvEnum.WARP.CV_WARP_INVERSE_MAP, new Bgr(0, 0, 0));

			// copy pixels
			int[] xx = new int[2]; int[] yy = new int[2];
			Image<Bgra, byte> newimage = new Image<Bgra, byte>(w, h);
			newimage.SetValue(new Bgra(0, 0, 0, 0));
			for (int j = 0; j < h; ++j)
			{
				for (int i = 0; i < w; ++i)
				{
					Vector3d pp = H * new Vector3d(i, j, 1);
					pp.HomogenousNormalize();
					xx[0] = (int)Math.Floor(pp.x); xx[1] = (int)Math.Ceiling(pp.x);
					yy[0] = (int)Math.Floor(pp.y); yy[1] = (int)Math.Ceiling(pp.y);

					// texture.
					for (int l = 0; l < 2; ++l)
					{
						int y = yy[l];
						bool ismappingdone = false;
						for (int m = 0; m < 2; ++m)
						{
							int x = xx[m];
							if (x >= 0 && x < srcimg.Width && y >= 0 && y < srcimg.Height && this.imagelabel[y, x] == plg.textureindex)
							{
								ismappingdone = true;

								Bgr fgcolor = fgwarp[j, i];
								Bgra newcolor = new Bgra(0, 0, 0, 255);

								newcolor.Blue = fgcolor.Blue;
								newcolor.Green = fgcolor.Green;
								newcolor.Red = fgcolor.Red;

								newimage[j, i] = newcolor;
								break;
							}
						}
						if (ismappingdone) break;
					}
				}
			}
			return newimage;
		}
		private void CalculatePolygonVisibilities()
		{
			this.UpdateAllProxy2DInfo();
			this.GeneratePolygonIds();
			List<Polygon> allpolyList = new List<Polygon>();
			foreach (PolyProxy proxy in this.polyproxies)
			{
				if (this.selectedproxy != null && this.selectedproxy != proxy)
					continue;
				allpolyList.AddRange(proxy.polygons);
			}
			Polygon[] allpolygons = allpolyList.ToArray();

			foreach (Polygon poly in allpolygons)
			{
				poly.H = null;
				poly.localframe = null;
			}

			foreach (Polygon poly in allpolygons)
			{
				Vector2d impos = this.GLUProject(poly.center.pos).ToVector2d();
				int label = this.FindNewPixelLabel(
				(int)impos.y,
				(int)impos.x,
				poly.hostproxy.polygons.ToArray());
				poly.visible = (label == this.polyids[poly]);
			}

			foreach (Polygon poly in allpolygons)
			{
				poly.H = null;
				poly.localframe = null;
			}
		}
		private void UpdateAllProxy2DInfo()
		{
			foreach (PolyProxy proxy in this.polyproxies)
			{
				this.UpdateProxy2DInfo(proxy);
			}
		}


		// interface feedback
		public void GetSelectionStartPoint(Vector2d mousepos)
		{
			GetActualGLPoint(ref mousepos);
			this.selectionstartpoint = new Point2(mousepos);
		}
		public void GetSelectionEndPoint(Vector2d mousepos)
		{
			GetActualGLPoint(ref mousepos);
			this.selectionendpoint = new Point2(mousepos);
		}
		public void ReleaseSelectionPoints()
		{
			this.selectionendpoint = this.selectionstartpoint = null;
		}
		public bool SelectBoxByRect(bool iscontroldown)
		{
			if (this.selectionstartpoint == null || 
				this.selectionendpoint == null) return false;
			Vector2d u = this.selectionstartpoint.pos, v = this.selectionendpoint.pos;
			Vector2d d = (u - v);
			double minx = Math.Min(u.x, v.x), miny = Math.Min(u.y, v.y);
			Rectangle rect = new Rectangle((int)minx,(int)miny, (int)Math.Abs(d.x), (int)Math.Abs(d.y));
			foreach (PolyProxy proxy in this.polyproxies)
			{
				foreach (Point2 p in proxy.points)
				{
					if (rect.Contains((int)p.pos.x, (int)p.pos.y))
					{
						this.selectedproxy = proxy;
		//				Program.OutputText("polygon point in rect", true);
						goto ADD;
					}
				}
			}
			List<Polygon> plist = new List<Polygon>();
			foreach (PolyProxy proxy in this.polyproxies)
			{
				foreach (Polygon p in proxy.polygons)
				{
					if (Shape2D.PointInPoly(this.selectionstartpoint.pos, p))
					{
						plist.Add(p);
					}
				}
			}
			if (plist.Count > 0)
			{
				double mindepth = double.MaxValue; Polygon sel = null;
				foreach (Polygon plg in plist)
				{
				 	Vector3d vv = this.ProjectScreenPoint2PolyFace(this.selectionstartpoint.pos, plg);
					double dpth = this.GLUProject(vv).z;
					if (dpth < mindepth)
					{
						mindepth = dpth;
						sel = plg;
					}
				}
				if (sel != null)
				{
					this.selectedproxy = sel.hostproxy;
					goto ADD;
				}
			}
			double mindis = double.MaxValue;
			foreach (PolyProxy proxy in this.polyproxies)
			{
				foreach (Point2 p in proxy.points)
				{
					double d1 = (p.pos - this.selectionstartpoint.pos).Length();
					double d2 = (p.pos - this.selectionendpoint.pos).Length();
					double dd = Math.Max(d1, d2);
					if (dd < mindis)
					{
						this.selectedproxy = proxy;
						mindis = dd;
					}
				}
			}
	//		Program.OutputText("both out, null selection",true);
			if (mindis > 100) this.selectedproxy = null;

			ADD:
			if (!iscontroldown)
			{
				this.selectedproxies.Clear();
				this.AddSelectedProxy();
			}
			else
			{
				this.selectedproxies.Remove(this.selectedproxy);
	//			Program.OutputText("current selected proxies: " + this.selectedproxies.Count,true);
				this.selectedproxy = null;
			}

			if (this.selectedproxy == null)
			{
				return false;
			}
			this.selectedpolyface = null;

			int index = this.polyproxies.IndexOf(this.selectedproxy);
	//		Program.OutputText("selected proxy index: " + index, true);

			return true;
		}
		public bool SelectBoxByPoint(Vector2d mousepos)
		{
			this.GetActualGLPoint(ref mousepos);
			double mindepth = double.MaxValue; PolyProxy sel = null;
			foreach (PolyProxy proxy in this.polyproxies)
			{
				foreach (Polygon p in proxy.polygons)
				{
					if (Shape2D.PointInPoly(mousepos, p))
					{
						Vector3d v3 = this.ProjectScreenPoint2PolyFace(mousepos, p);
						double depth = this.GLUProject(v3).z;
						if (depth < mindepth)
						{
							mindepth = depth;
							sel = proxy;
						}
					}
				}
			}

			this.selectedproxy = sel;

			int index = this.polyproxies.IndexOf(this.selectedproxy);

			return true;
		}
		public bool SelectBoxesByRect()
		{
			if (this.selectionstartpoint == null ||
				this.selectionendpoint == null) return false;

			List<PolyProxy> sels = this.SelBoxesByRect();
			foreach (PolyProxy s in sels)
			{
				if (!this.selectedproxies.Contains(s))
				{
					this.selectedproxies.Add(s);
				}
			}
			return true;
		}
		public void DeselectBoxesByRect()
		{
			List<PolyProxy> sels = this.SelBoxesByRect();
			foreach (PolyProxy s in sels)
			{
				this.selectedproxies.Remove(s);
			}
		}
		private List<PolyProxy> SelBoxesByRect()
		{
			if (this.selectionstartpoint == null || this.selectionendpoint == null) return null;

			Vector2d u = this.selectionstartpoint.pos, v = this.selectionendpoint.pos;
			Vector2d d = (u - v);
			double minx = Math.Min(u.x, v.x), miny = Math.Min(u.y, v.y);
			Rectangle rect = new Rectangle((int)minx, (int)miny, (int)Math.Abs(d.x), (int)Math.Abs(d.y));
			List<PolyProxy> sels = new List<PolyProxy>();
			foreach (PolyProxy proxy in this.polyproxies)
			{
				foreach (Point2 p in proxy.points)
				{
					if (rect.Contains((int)p.pos.x, (int)p.pos.y))
					{
						sels.Add(proxy);
						break;
					}
				}
			}
			return sels;
		}
		public void SelectBoxFaceByRect()
		{
			if (this.selectionstartpoint == null || this.selectionendpoint == null) return;
			Vector2d d = (this.selectionendpoint.pos - this.selectionstartpoint.pos);
			Rectangle rect = new Rectangle((int)this.selectionstartpoint.pos.x,
				(int)this.selectionstartpoint.pos.y, (int)Math.Abs(d.x), (int)Math.Abs(d.y));
			List<PolyProxy> proxes = new List<PolyProxy>();
			List<Polygon> faces = new List<Polygon>();
			foreach (PolyProxy proxy in this.polyproxies)
			{
				foreach (Polygon p in proxy.polygons)
				{
					if (Shape2D.PointInPoly(this.selectionstartpoint.pos, p) ||
						Shape2D.PointInPoly(this.selectionendpoint.pos, p))
					{
						proxes.Add(proxy);
						faces.Add(p);
					}
				}
			}
			double mindepth = double.MaxValue; Polygon s = null; PolyProxy pro = null;
			int index = 0;
			foreach (Polygon p in faces)
			{
				double depth = this.GLUProject(p.center.pos).z;
				if (depth < mindepth)
				{
					mindepth = depth;
					pro = proxes[index];
					s = p;
				}
				index++;
			}
			if (s != null)
			{
				this.selectedproxy = pro;
				this.selectedpolyface = s;
				Program.OutputText("selected face", true);
			}
			else
			{
				this.selectedpolyface = null;
				this.selectedproxy = null;
			}
		}
		private PolyProxy Quads2BoxProxy(Quad A, Quad B)
		{
			// for simplicity, it is assumed that they share a default line segment[0], 
			// /**otherwise we can simply perform search**/
			LineSegment la = A.linesegments[0];
			LineSegment lb = A.linesegments[2];
			LineSegment lc = B.linesegments[2];

			Vector3d vx = new Vector3d(this.vp2d[B.linesegments[1].vanishingindex], this.whomogenous);
			Vector3d vy = new Vector3d(this.vp2d[A.linesegments[1].vanishingindex], this.whomogenous);
			Vector3d x = new Vector3d(lb.u.pos, this.whomogenous);
			Vector3d y = new Vector3d(lc.u.pos, this.whomogenous);
			Vector3d intersect1 = vx.Cross(x).Cross(vy.Cross(y));
			intersect1.HomogenousNormalize();

			x = new Vector3d(lb.v.pos, this.whomogenous);
			y = new Vector3d(lc.v.pos, this.whomogenous);
			Vector3d intersect2 = vx.Cross(x).Cross(vy.Cross(y));
			intersect2.HomogenousNormalize();

			Point2 u = new Point2(intersect1.ToVector2d());
			Point2 v = new Point2(intersect2.ToVector2d());
			Point2 a = la.u, b = lb.u, c = lc.u, e = la.v, f = lb.v, g = lc.v;

			Polygon toppoly = null;
			Polygon botpoly = null;
			switch (la.vanishingindex)
			{
				case 2: // z
					{
						Point2[] pts2 = new Point2[4] { a, b, u, c };
						botpoly = new Polygon(pts2);
						pts2 = new Point2[4] { e, f, v, g };
						toppoly = new Polygon(pts2);
					}
					break;
				case 1: // y
					{
						Point2[] pts2 = new Point2[4] { a, c, g, e };
						botpoly = new Polygon(pts2);
						pts2 = new Point2[4] { b, u, v, f };
						toppoly = new Polygon(pts2);
					}
					break;
				case 0: // x
					{
						Point2[] pts2 = new Point2[4] { a, b, f, e };
						botpoly = new Polygon(pts2);
						pts2 = new Point2[4] { c, u, v, g };
						toppoly = new Polygon(pts2);
					}
					break;
			}
			double ymax = 0; Polygon bottom = toppoly;
			foreach (Point2 p in toppoly.points)
			{
				if (p.pos.y > ymax)
				{
					ymax = p.pos.y;
				}
			}
			foreach (Point2 p in botpoly.points)
			{
				if (p.pos.y > ymax)
				{
					ymax = p.pos.y;
					bottom = botpoly;
					break;
				}
			}
			if (bottom != botpoly)
			{
				toppoly = botpoly;
				botpoly = bottom;
			}
			// compute 3d coords
			Vector3d top0 = this.cameracalibrator.Compute3DCoordinate(
				toppoly.points[0].pos, botpoly.points[0].pos
				);
			double height = top0.z;
			foreach (Point2 p in botpoly.points)
			{
				Vector3d pt = this.cameracalibrator.ComputePointGround3dPos(p.pos);
				botpoly.points3.Add(new Point3(pt));
				toppoly.points3.Add(new Point3(new Vector3d(pt.x,pt.y,height)));
			}

			return PolyProxy.MakePolyProxy(botpoly,toppoly);
		}
		private void FinalizeProxy(PolyProxy proxy)
		{
			this.GetProxyTxtQuads();
			this.GetProxyLineSegmentsP3(proxy);
		}
		private void SelectBox(Vector2d mousepos)
		{
			this.GetActualGLPoint(ref mousepos);
			this.selectedproxy = null;
			PolyProxy closest = null;
			double min = double.MaxValue;
			foreach (PolyProxy proxy in this.polyproxies)
			{
				foreach (Polygon plg in proxy.polygons)
				{
					if (!Shape2D.PointInPoly(mousepos, plg)) continue;
					double dis = (plg.center2 - mousepos).Length();
					if (dis < min)
					{
						closest = proxy;
						min = dis;
					}
				}
			}
			if (min > this.imgwidth / 2) closest = null;
			this.selectedproxy = closest;
		}
		
		public void SelectFace(Vector2d mousepos, bool allownullselection)
		{
			GetActualGLPoint(ref mousepos);
			this.SelectBox(mousepos);
			if (this.selectedproxy != null)
			{
				double mindis = double.MaxValue; Polygon selected = null;
				foreach (Polygon poly in selectedproxy.polygons)
				{
					if (!Shape2D.PointInPoly(mousepos, poly)) continue;
					double dis = (poly.center2 - mousepos).Length();
					if (dis < mindis)
					{
						mindis = dis;
						selected = poly;
					}
				}
				LineSegment sel = null;
				if (this.selectedproxy.polygons.Count == 1)
				{
					double min = double.MaxValue;
					foreach (LineSegment line in this.selectedproxy.polygons[0].linesegments)
					{
						double dis = ((line.u.pos + line.v.pos) / 2 - mousepos).Length();
						if (dis < min)
						{
							min = dis;
							sel = line;
						}
					}
				}
				if (allownullselection && mindis > this.imgwidth / 2)
				{
					this.selectedpolyface = null;
				}
				else
				{
					this.selectedpolyface = selected;
				}
			}
			else
			{
				this.selectedpolyface = null;
			}
		}
		public void SelectFace(Vector2d mousepos)
		{
			this.GetActualGLPoint(ref mousepos);
			List<Polygon> plist = new List<Polygon>();
			foreach (PolyProxy proxy in this.polyproxies)
			{
				foreach (Polygon p in proxy.polygons)
				{
					if (Shape2D.PointInPoly(mousepos, p))
					{
						plist.Add(p);
					}
				}
			}
			if (plist.Count > 0)
			{
				double mindepth = double.MaxValue; Polygon sel = null;
				foreach (Polygon plg in plist)
				{
					Vector3d vv = this.ProjectScreenPoint2PolyFace(mousepos, plg);
					double dpth = this.GLUProject(vv).z;
					if (dpth < mindepth)
					{
						mindepth = dpth;
						sel = plg;
					}
				}
				if (sel != null)
				{
					this.selectedproxy = sel.hostproxy;
					this.selectedpolyface = sel;
				}
			}

			int index = this.polyproxies.IndexOf(this.selectedproxy);
			Program.OutputText("selected proxy index: " + index, true);
		}
		public void AddSelectedProxy()
		{
			if (this.selectedproxy != null &&
				this.selectedproxies.Contains(this.selectedproxy) == false)
				this.selectedproxies.Add(this.selectedproxy);
		}
		public void AddSelectedFace()
		{
			if (this.selectedfaces == null)
				this.selectedfaces = new List<Polygon>();
			if (this.selectedpolyface != null && 
				this.selectedfaces.Contains(this.selectedpolyface) == false)
				this.selectedfaces.Add(this.selectedpolyface);
		}
		public void ClearSelectedProxies()
		{
			if (this.selectedproxies != null)
				this.selectedproxies.Clear();
		}
		public void ClearSelectedFaces()
		{
			if (this.selectedfaces != null)
				this.selectedfaces.Clear();
		}
		public void CreateProxyEditMetaphors()
		{
			FindRotationBallSize();
			foreach (PolyProxy proxy in this.polyproxies)
			{
				CreateProxyEditMetaphors(proxy);
			}
		}
		public void CreateProxyEditMetaphors(PolyProxy proxy)
		{
			if (proxy == null) return;
			double length = 0.15; double ratio = 0.5;
			proxy.transmetaphors.Clear();
			proxy.rotationmetaphor = null;
			switch (proxy.type)
			{
				case ProxyType.QUAD:
					{
						Point3 center = proxy.center;
						Vector3d norm = proxy.polygons[0].normal;
						TranslationMetaphor metaphor = new TranslationMetaphor(center.pos, norm, length);
						proxy.transmetaphors.Add(metaphor);
						foreach (LineSegment line in proxy.polygons[0].linesegments)
						{
							Vector3d c = (line.u3.pos + line.v3.pos) / 2;
							Vector3d uv = (line.u3.pos - line.v3.pos).Normalize();
							Vector3d ax = norm.Cross(uv).Normalize();
							if (ax.Dot(c - center.pos) < 0)
								ax = new Vector3d() - ax;
							TranslationMetaphor meta = new TranslationMetaphor(c, ax, length);
							proxy.transmetaphors.Add(meta);
						}
					}
					break;
				case ProxyType.BOX:
					{
						foreach (Polygon plg in proxy.polygons)
						{
							//		if (plg.visible)
							{
								TranslationMetaphor metaphor =
									new TranslationMetaphor(plg.center.pos, plg.normal, length);
								proxy.transmetaphors.Add(metaphor);
							}
						}
						Polygon g = proxy.polygons[1];
						double r1 = (g.linesegments[0].u3.pos - g.linesegments[0].v3.pos).Length() * ratio;
						double r2 = (g.linesegments[1].u3.pos - g.linesegments[1].v3.pos).Length() * ratio;
						double rr = r1 > r2 ? r2 : r1;
						RotationMetaphor rmetaphor = new RotationMetaphor(g.center.pos, g.normal, rr, this.RotationSlices);
						proxy.rotationmetaphor = rmetaphor;
					}
					break;
				case ProxyType.JOINT:
					{
						foreach (Polygon plg in proxy.polygons)
						{
							//		if (plg.visible)
							{
								TranslationMetaphor metaphor =
									new TranslationMetaphor(plg.center.pos, plg.normal, length);
								proxy.transmetaphors.Add(metaphor);
							}
						}
						if (proxy.polygons.Count == 1) // a quad
						{
							Point3 center = proxy.center;
							Vector3d norm = proxy.polygons[0].normal;
							foreach (LineSegment line in proxy.polygons[0].linesegments)
							{
								Vector3d c = (line.u3.pos + line.v3.pos) / 2;
								Vector3d uv = (line.u3.pos - line.v3.pos).Normalize();
								Vector3d ax = norm.Cross(uv).Normalize();
								if (ax.Dot(c - center.pos) < 0)
									ax = new Vector3d() - ax;
								TranslationMetaphor meta = new TranslationMetaphor(c, ax, length);
								proxy.transmetaphors.Add(meta);
							}
						}
						if (proxy.isjointhoster)
						{
							Polygon g = proxy.polygons[1];
							double r1 = (g.linesegments[0].u3.pos - g.linesegments[0].v3.pos).Length() * ratio;
							double r2 = (g.linesegments[1].u3.pos - g.linesegments[1].v3.pos).Length() * ratio;
							double rr = r1 > r2 ? r2 : r1;
							RotationMetaphor rmetaphor = new RotationMetaphor(g.center.pos, g.normal, rr, this.RotationSlices);
							proxy.rotationmetaphor = rmetaphor;
						}
						else
						{
							LineSegment line = proxy.jointline;
							Point3 v = line.u3, u = line.v3;
							Vector3d axis = (u.pos - v.pos).Normalize();
							double r = 0;
							foreach (Polygon plg in proxy.polygons)
							{
								if (proxy.polygons.Count == 1 || plg.points3.Contains(u) && !plg.points3.Contains(v))
								{
									double l1 = (plg.points3[0].pos - plg.points3[1].pos).Length();
									double l2 = (plg.points3[3].pos - plg.points3[0].pos).Length();
									r = l1 > l2 ? l1 : l2;
									break;
								}
							}
							RotationMetaphor rmetaphor = new RotationMetaphor(u.pos, axis, r * 0.3, this.RotationSlices);
							proxy.rotationmetaphor = rmetaphor;
						}
					}
					break;
			}
			foreach (TranslationMetaphor mtp in proxy.transmetaphors)
			{
				this.ComputeMetaphorScreenDir(mtp);
			}
			this.ComputeMetaphorScreenDir(proxy.rotationmetaphor);
		}
		public void ComputeMetaphorScreenDir(Metaphor mtphor)
		{
			Vector3d u = this.GLUProject(mtphor.center.pos);
			Vector3d v = this.GLUProject(mtphor.center.pos + mtphor.axis);
			mtphor.screendir = (v - u).ToVector2d();
		}
		public void DeleteSelectProxy()
		{
			if (this.selectedproxy != null)
			{
				this.polyproxies.Remove(this.selectedproxy);
				this.selectedproxies.Clear();
				this.selectedproxy = null;
				this.selectedpolyface = null;
			}
		}		
		public void SetFixedProxies()
		{
			foreach (PolyProxy px in this.selectedproxies)
			{
				px.isfixed = !px.isfixed;
			}
		}


		// deformation operations
		public void FixSelect()
		{
			this.isselectfixed = true;
		}
		public void RelaseFixSelect()
		{
			this.selectedproxies.Clear();
			this.selectedmetaphor = null;
			this.selectedproxy = null;
			this.selectedpolyface = null;
			this.isselectfixed = false;
		}
		public void Select(Vector2d mousepos, bool isrotation)
		{
			if (this.selectedproxy == null || !this.isselectfixed)
			{
				if (isrotation)
					this.SelectBoxByPoint(mousepos);
				else
					this.SelectFace(mousepos,true);
			}
			else if (isrotation)
			{
				this.SelectMetaphorRotationPoint(mousepos);
			}
			else
			{
				this.SelectTranslateMetaphor(mousepos);
			}
		}
		public void DeformProxy(PolyProxy proxy, Matrix4d T, TransformStyle style, int d)
		{
			if (proxy == null) return;
			if (proxy.jointproxy != null)
			{
				DeformJointProxy(proxy, T, style, d);
			}
			else
				DeformProxy(proxy, T, d);
			proxy.Compute3DInfo();
			this.UpdateProxy2DInfo(proxy);
		}
		public void DeformSelectedProxy(Matrix4d T, TransformStyle style, int d)
		{
			if (this.selectedproxy == null) return;
			if (this.selectedproxy.jointproxy != null)
			{
				DeformJointProxy(this.selectedproxy, T, style, d);
			}
			else
			{
				int index = this.selectedproxy.segindex;
				List<PolyProxy> proxies = new List<PolyProxy>();
				foreach (PolyProxy px in this.polyproxies)
				{
					if (px.segindex == index)
					{
						proxies.Add(px);
					}
				}
				foreach (PolyProxy px in proxies)
					DeformProxy(px, T, this.selectedproxy.center.pos, d);
				if (style == TransformStyle.Scaling)
				{
					double minz = double.MaxValue; Point3 lowest = null;
					foreach (PolyProxy px in proxies)
					{
						foreach (Point3 p3 in px.points3)
						{
							if (p3.pos.z < minz)
							{
								lowest = p3;
								minz = p3.pos.z;
							}
						}
					}
					Vector3d t = new Vector3d(0,0,lowest.oldpos.z-minz);
					foreach (PolyProxy px in proxies)
					{
						foreach (Point3 p3 in px.points3)
						{
							p3.pos += t;
						}
					}
				}
			}
			this.selectedproxy.Compute3DInfo();
			this.UpdateProxy2DInfo(this.selectedproxy);
		}
		public void DeformJointProxy(PolyProxy proxy, Matrix4d T, TransformStyle t, int d)
		{
			switch (t)
			{
				case TransformStyle.Translation:
				case TransformStyle.Scaling:
					{
						DeformProxy(proxy, T, proxy.center.pos, d);
						DeformProxy(proxy.jointproxy, T, proxy.center.pos, d);
					}
					break;
				case TransformStyle.Rotation:
					{
						if (proxy.isjointhoster) break;
						Point3 u = proxy.points3[proxy.jointline.u.index];
						Point3 v = proxy.points3[proxy.jointline.v.index];
						Vector4d c = new Vector4d((u.oldpos + v.oldpos)/2);
						foreach (Point3 p in proxy.points3)
						{
							Vector4d q = new Vector4d(p.oldpos, d);
							q = T * (q - c) + c;
							p.pos = q.XYZ();
						}
					}
					break;
			}
			this.selectedproxy.Compute3DInfo();
			this.UpdateProxy2DInfo(this.selectedproxy);
		}
		public void InitialDragging(Vector2d mousedpos)
		{
			if (this.selectedproxy == null) return;
			this.GetActualGLPoint(ref mousedpos);
			if (this.selectedmetaphor != null)
			{
				this.draggingpos = this.selectedmetaphor.start.pos;
			}
			else
			{
				Polygon poly = this.selectedproxy.polygons[0];
				this.draggingpos = this.ProjectScreenPoint2PolyFace(mousedpos, poly);
			}
		}
		public void SelectMetaphorRotationPoint(Vector2d mousepos)
		{
			if (this.selectedproxy == null) return;
			this.GetActualGLPoint(ref mousepos);
			double mindis = double.MaxValue; Vector3d p = new Vector3d(); Metaphor selected = null;
			RotationMetaphor meta = this.selectedproxy.rotationmetaphor;
			if (meta != null)
			{
				Vector3d pt; double d;
				this.SelectMetaphorPoint(mousepos, meta, out pt, out d);
				if (d < mindis)
				{
					mindis = d;
					selected = meta;
					p = pt;
				}
			}
			if (mindis > 15) selected = null;
			if (selected != null)
			{
				this.selectedmetaphor = selected;
				selected.start = new Point3(p);

				if (selected != meta && meta != null)
				{
					meta.ReleaseMetaphorPoints();
				}
			}
			else if (meta != null)
			{
				this.selectedmetaphor = null;
				meta.ReleaseMetaphorPoints();
			}
		}
		public void SelectTranslateMetaphor(Vector2d mousepos)
		{
			if (this.selectedproxy == null) return;
			this.GetActualGLPoint(ref mousepos);
			double mindis = double.MaxValue; Vector3d p = new Vector3d();
			foreach (TranslationMetaphor metaphor in this.selectedproxy.transmetaphors)
			{
				Vector3d pt; double d;
				this.SelectMetaphorPoint(mousepos, metaphor, out pt, out d);
				if (d < mindis)
				{
					this.selectedmetaphor = metaphor;
					mindis = d;
					p = pt;
				}
			}
			if (mindis > 100)
			{
				this.selectedmetaphor = null;
			}
			else
			{
				Vector3d axis = this.selectedmetaphor.axis;
				foreach (Polygon plg in this.selectedproxy.polygons)
				{
					if (plg.normal.Dot(axis) > 0.8)
					{
						this.selectedpolyface = plg;
						break;
					}
				}
			}
		}
		public void SelectMetaphorStartingPoint(Vector2d mousepos)
		{
			Metaphor meta = this.selectedmetaphor;
			if (this.selectedproxy == null || meta == null) return;

			this.GetActualGLPoint(ref mousepos);

			Vector3d pt; double d;
			this.SelectMetaphorPoint(mousepos, meta, out pt, out d);
			meta.start = new Point3(pt);
		}
		public void SelectMetaphorEndingPoint(Vector2d mousepos)
		{
			Metaphor meta = this.selectedmetaphor;
			if (this.selectedproxy == null || meta == null) return;
			
			this.GetActualGLPoint(ref mousepos);
			
			Vector3d pt; double d;
			this.SelectMetaphorPoint(mousepos, meta, out pt, out d);
			meta.end = new Point3(pt);
		}
		public void SelectMetaphorTranslatePoint(Vector2d mousedpos, Vector2d mousepos)
		{
			if (this.selectedmetaphor == null) return;
			this.GetActualGLPoint(ref mousepos);
			this.GetActualGLPoint(ref mousedpos);
			mousedpos.y = -mousedpos.y;
			mousepos.y = -mousepos.y;
			Vector2d w = mousepos - mousedpos;
			double l1 = this.TranslateScaleFactor;
			double l2 = this.selectedmetaphor.screendir.Normalize().Dot(w);
			double ratio = l2 / l1 * (this.selectedmetaphor.points[1] - this.selectedmetaphor.points[0]).Length();
			Vector3d p = this.selectedmetaphor.center.oldpos + ratio * this.selectedmetaphor.axis;
			this.selectedmetaphor.end = new Point3(p);
		}

		public void TranslateSelectedProxy()
		{
			if (this.selectedmetaphor == null || this.selectedmetaphor.start == null ||
				this.selectedmetaphor.end == null)
				return;
		}
		public void TranslateSelectedFace()
		{
			if (this.selectedmetaphor == null || this.selectedmetaphor.start == null ||
				this.selectedmetaphor.end == null)
				return;
			if (this.disablepropatation) return;
			if (this.selectedproxy.jointproxy != null) return;
			if (this.selectedproxy != null)
			{
				Transform t = this.selectedmetaphor.GetTransformation();
				DeformPolygon(this.selectedpolyface, t.T, 1);
			}
		}
		public void DragSelected(Vector2d mousepos)
		{
			if (this.selectedproxy == null) return;
			this.GetActualGLPoint(ref mousepos);
			this.DragProxy(mousepos);
		}
		
		public void DragProxy(Vector2d mousepos) // problem
		{
			// we do not drag planar proxies

			this.GetActualGLPoint(ref mousepos);
			Matrix4d T = null;
			if (this.selectedmetaphor != null)
			{
				Vector3d pt; double dis;
				this.SelectMetaphorPoint(mousepos, this.selectedmetaphor, out pt, out dis);
				this.selectedmetaphor.end = new Point3(pt);
				Transform t = this.selectedmetaphor.GetTransformation();
				T = t.T;
			}
			else
			{
				if (this.selectedproxy == null || this.selectedproxy.points.Count < 8)
					return;
				Polygon poly = this.selectedproxy.polygons[0];
				Vector3d v = this.ProjectScreenPoint2PolyFace(mousepos, poly);
				Vector3d off = v - this.draggingpos;
				T = Matrix4d.TranslationMatrix(off);
			}

			Vector3d c = this.selectedmetaphor == null ? new Vector3d() : this.selectedmetaphor.center.pos;
			List<PolyProxy> bros = this.GetAllCoupledProxies(this.selectedproxy);
			foreach (PolyProxy px in bros)
				DeformProxy(px, T, c, 1);
			if (this.selectedproxy.isjointhoster)
			{
				DeformProxy(this.selectedproxy.jointproxy, T, c, 1);
			}
		}
		private List<PolyProxy> GetAllCoupledProxies(PolyProxy proxy)
		{
			int index = proxy.segindex;
			List<PolyProxy> proxies = new List<PolyProxy>();
			foreach (PolyProxy px in this.polyproxies)
			{
				if (px.segindex == index)
				{
					proxies.Add(px);
					if (px.onobjects != null)
					{
						proxies.AddRange(px.onobjects);
					}
				}
			}
			return proxies;
		}
		private List<PolyProxy> GetAllBroProxies(PolyProxy proxy)
		{
			int index = proxy.segindex;
			List<PolyProxy> proxies = new List<PolyProxy>();
			foreach (PolyProxy px in this.polyproxies)
			{
				if (px.segindex == index)
				{
					proxies.Add(px);
				}
			}
			if (proxy.jointproxy != null && !proxies.Contains(proxy.jointproxy))
			{
				proxies.Add(proxy.jointproxy);
			}
			return proxies;
		}
		public void DeformUpdateSelected(Matrix4d T)
		{
			if (this.selectedproxy == null) return;
			this.selectedproxy.Compute3DInfo();
			this.CreateProxyEditMetaphors(this.selectedproxy);
			// update the propagation points
			foreach (Polygon plg in this.selectedproxy.polygons)
			{
				if (T != null)
					foreach (Point3 p in plg.propagatepoints3)
					{
						p.R = T.ToMatrix3d();
					}
			}
		}
		public void UpdateProxy(PolyProxy proxy, Matrix4d T)
		{
			if (proxy != null)
			{
				foreach (Point3 p in proxy.points3)
				{
					p.UpdateOldPos();
				}
				proxy.Compute3DInfo();
				proxy.center.UpdateOldPos();

				
				// update the propagation points
				foreach (Polygon plg in proxy.polygons)
				{
					plg.CreatePropagatingPoints();
					if (T!=null)
					foreach (Point3 p in plg.propagatepoints3)
					{
						p.R = T.ToMatrix3d();
					}
				}

				// update 2d 
				this.UpdateProxy2DInfo(proxy);
			}
		}
		public void UpdateProxy2DInfo(PolyProxy proxy)
		{
			// update the img position
			int index = 0;
			foreach (Point3 p in proxy.points3)
			{
				proxy.points[index].pos = this.cameracalibrator.ComputePointImgPos(p.pos);
				proxy.points[index++].UpdateOldPos();
			}
			foreach (Polygon poly in proxy.polygons)
				poly.ComputeCenter();
		}
		public void SelectMetaphorPoint(Vector2d mousepos, Metaphor metaphor, out Vector3d point, out double dis)
		{
			// select point on the metaphor
			double mindis = double.MaxValue; point = new Vector3d();
			mousepos.y = parentview.Height - mousepos.y;
			foreach (Vector3d p in metaphor.points)
			{
				Vector2d scrpos = this.GLUProject(p).ToVector2d();
				double l = (mousepos - scrpos).Length();
				if (l < mindis)
				{
					mindis = l;
					point = p;
				}
			}
			dis = mindis;
		}
		public void CreatePropagationPoints()
		{
			// first find the propatation density per line segment
			double minL = double.MaxValue;
			foreach (PolyProxy proxy in this.polyproxies)
			{
				foreach (Polygon plg in proxy.polygons)
				{
					foreach (LineSegment line in plg.linesegments)
					{
						double l = (line.u3.pos - line.v3.pos).Length();
						if (l < minL)
							minL = l;
					}
				}
			}
			Polygon.propagationpointdensity = minL / 10;
			foreach (PolyProxy proxy in this.polyproxies)
			{
				foreach (Polygon plg in proxy.polygons)
				{
					plg.CreatePropagatingPoints();
				}
			}
		}
		public void CreatePropagationPoints(List<PolyProxy> proxies)
		{
			// first find the propatation density per line segment
			double minL = double.MaxValue;
			foreach (PolyProxy proxy in proxies)
			{
				foreach (Polygon plg in proxy.polygons)
				{
					foreach (LineSegment line in plg.linesegments)
					{
						double l = (line.u3.pos - line.v3.pos).Length();
						if (l < minL)
							minL = l;
					}
				}
			}
			Polygon.propagationpointdensity = minL / 10;
			foreach (PolyProxy proxy in proxies)
			{
				foreach (Polygon plg in proxy.polygons)
				{
					plg.CreatePropagatingPoints();
				}
			}
		}


		// rendering entities
		public void RenderSense()
		{
			this.Render2dScene();
			this.Render3dScene();
			this.HighLightSelection();
		}
        public void Render2dScene()
        {	
			Gl.glPushMatrix();

			Gl.glViewport(0, this.parentview.Size.Height-this.glviewporth, this.glviewportw, this.glviewporth);

			Gl.glMatrixMode(Gl.GL_PROJECTION);
			Gl.glLoadIdentity();

			Glu.gluOrtho2D(0, this.glviewportw, this.glviewporth, 0);

			Gl.glMatrixMode(Gl.GL_MODELVIEW);
			Gl.glLoadIdentity();

            Gl.glPushMatrix();

			Gl.glTranslated(imgpos.x, imgpos.y, 0.0);
			Gl.glScaled(this.sacleRatio, this.sacleRatio, this.sacleRatio);
			Gl.glColor3f(1.0f, 1.0f, 1.0f);

			if (this.renderLayerImage)
				this.RenderVanishingLines();
			
			#region  Render Image
			if (renderLayerImage)
			{
				Gl.glEnable(Gl.GL_TEXTURE_2D);
				Gl.glTexEnvi(Gl.GL_TEXTURE_ENV, Gl.GL_TEXTURE_ENV_MODE, (int)Gl.GL_REPLACE);
				Gl.glBindTexture(Gl.GL_TEXTURE_2D, layertxtid[0]);
				Gl.glBegin(Gl.GL_QUADS);
				Gl.glTexCoord2d(0, 0);
				Gl.glVertex2d(0, 0);
				Gl.glTexCoord2d(1, 0);
				Gl.glVertex2d(this.imgwidth, 0);
				Gl.glTexCoord2d(1, 1);
				Gl.glVertex2d(this.imgwidth, this.imgheight);
				Gl.glTexCoord2d(0, 1);
				Gl.glVertex2d(0, this.imgheight);
				Gl.glEnd();
				Gl.glDisable(Gl.GL_TEXTURE_2D);
			}
			#endregion

	//		this.HighlightElements2D();

			Gl.glPopMatrix();

			Gl.glPopMatrix();

        }
		public void Render3dScene()
		{
			if (this.glprojectionmatrix == null) return;

			Gl.glViewport(0, this.parentview.Size.Height - this.glviewporth, this.glviewportw, this.glviewporth);

			// -------------------------------------------------------------------------
			// draw 3d sense
			Matrix4d m = this.parentview.GetTransMatrix();
			m = this.roomtrans * m *this.roomtrans_inv;


			Gl.glPushMatrix();
			Gl.glMatrixMode(Gl.GL_PROJECTION);
			Gl.glLoadIdentity();
			Gl.glLoadMatrixd(this.glprojectionmatrix);

			Gl.glMatrixMode(Gl.GL_MODELVIEW);
			Gl.glLoadIdentity();
			Gl.glLoadMatrixd(this.glmodelviewmatrix);


			this.InitLighting();

			Gl.glPushMatrix();
			Gl.glMultMatrixd((m.Transpose()).ToArray());


			if (this.RenderBoxWithTexutreMap)
				this.RenderTextured3DScene();

			this.RenderPolyProxies3D(this.polyproxies, this.ProxyColor);// Color.FromArgb(230, 219, 24));
				
			this.HighlightElements3D();

			Gl.glPopMatrix();
			this.DisableLighting();
			Gl.glPopMatrix();
		}

		// render elements
		private void RenderTextured3DScene()
		{
			Gl.glShadeModel(Gl.GL_SMOOTH);
			
			Gl.glPolygonMode(Gl.GL_FRONT_AND_BACK, Gl.GL_FILL);
			Gl.glEnable(Gl.GL_LIGHTING);
			Gl.glEnable(Gl.GL_NORMALIZE);

		//	Gl.glDisable(Gl.GL_CULL_FACE);
			Gl.glEnable(Gl.GL_DEPTH_TEST);

			Gl.glEnable(Gl.GL_BLEND);
			Gl.glBlendFunc(Gl.GL_SRC_ALPHA, Gl.GL_ONE_MINUS_SRC_ALPHA);
			
			Gl.glEnable(Gl.GL_ALPHA_TEST);
			Gl.glAlphaFunc(Gl.GL_GREATER, 0.4f);

			foreach (PolyProxy b in this.polyproxies)
			{
				foreach (Polygon poly in b.polygons)
				{
					if (!this.TextureAllPolygons && poly.textureindex == -1) continue;
					if (poly.textid == null) continue;

					uint texid = usegltexturemapping ? this.layertxtid[0] : poly.textid[0];

					Gl.glEnable(Gl.GL_TEXTURE_2D);
					Gl.glTexEnvi(Gl.GL_TEXTURE_ENV, Gl.GL_TEXTURE_ENV_MODE, (int)Gl.GL_REPLACE);
					Gl.glBindTexture(Gl.GL_TEXTURE_2D, texid);

					Gl.glColor3ub(255, 255, 255);
					Gl.glBegin(Gl.GL_POLYGON);

					Vector3d p, n;
					Vector2d t;
					t = poly.assignedtextures[0];
					if (usegltexturemapping)
						Gl.glTexCoord2d(poly.points[0].texcoord.x, poly.points[0].texcoord.y);
					else
						Gl.glTexCoord2d(t.x, t.y);
					p = poly.points3[0].pos;
					n = poly.points3[0].normal;
					Gl.glNormal3d(n.x, n.y, n.z);
					Gl.glVertex3d(p.x, p.y, p.z);


					t = poly.assignedtextures[1];
					if (usegltexturemapping)
						Gl.glTexCoord2d(poly.points[1].texcoord.x, poly.points[1].texcoord.y);
					else
						Gl.glTexCoord2d(t.x, t.y);
					p = poly.points3[1].pos;
					n = poly.points3[1].normal;
					Gl.glNormal3d(n.x, n.y, n.z);
					Gl.glVertex3d(p.x, p.y, p.z);


					t = poly.assignedtextures[2];
					if (usegltexturemapping)
						Gl.glTexCoord2d(poly.points[2].texcoord.x, poly.points[2].texcoord.y);
					else
						Gl.glTexCoord2d(t.x, t.y);
					p = poly.points3[2].pos;
					n = poly.points3[2].normal;
					Gl.glNormal3d(n.x, n.y, n.z);
					Gl.glVertex3d(p.x, p.y, p.z);


					t = poly.assignedtextures[3];
					if (usegltexturemapping)
						Gl.glTexCoord2d(poly.points[3].texcoord.x, poly.points[3].texcoord.y);
					else
						Gl.glTexCoord2d(t.x, t.y);
					p = poly.points3[3].pos;
					n = poly.points3[3].normal;
					Gl.glNormal3d(n.x, n.y, n.z);
					Gl.glVertex3d(p.x, p.y, p.z);

					Gl.glEnd();
					Gl.glDisable(Gl.GL_TEXTURE_2D);

				}
			}

			Gl.glDisable(Gl.GL_ALPHA_TEST);

			Gl.glDisable(Gl.GL_BLEND);
			Gl.glDisable(Gl.GL_DEPTH_TEST);
		//	Gl.glEnable(Gl.GL_CULL_FACE);

			Gl.glDisable(Gl.GL_LIGHTING);
			Gl.glDisable(Gl.GL_NORMALIZE);
		}
		private void RenderLineSegment(LineSegment s, int colorid)
		{
			Gl.glEnable(Gl.GL_LINE_SMOOTH);
			switch (colorid)
			{
				case 0:
					Gl.glColor3ub(255, 0, 0);
					break;
				case 1:
					Gl.glColor3ub(0, 255, 0);
					break;
				case 2:
					Gl.glColor3ub(0, 0, 255);
					break;
				case 3:
					Gl.glColor3ub(255, 255, 255);
					break;
				default:
					Gl.glColor3ub(255, 20, 147);
					break;
			}
			Vector2d u = s.u.pos, v = s.v.pos;
			double L = (u - v).Length();
			int intervals = (int)(L / 10);
			int k = intervals;
			double step = 1.0/k;
			Gl.glLineWidth(1.0f);
			Gl.glBegin(Gl.GL_LINES);
			for (int i = 0; i < k; i+=2)
			{
				double r1 = step * i, r2 = step*(i+1);
				Vector2d p = u * (1 - r1) + v * r1;
				Vector2d q = u * (1 - r2) + v * r2;
				Gl.glVertex2d(p.x, p.y);
				Gl.glVertex2d(q.x, q.y);
			}
			Gl.glEnd();
			Gl.glColor3ub(255, 255, 255);
			Gl.glPointSize(2.0f);
			Gl.glBegin(Gl.GL_POINTS);
			Gl.glVertex2d(s.u.pos.x, s.u.pos.y);
			Gl.glVertex2d(s.v.pos.x, s.v.pos.y);
			Gl.glEnd();
		}
		private void RenderPoint(Point2 p, float size, Color c)
		{
			Gl.glColor3ub(c.R,c.G,c.B);
			Gl.glPointSize(size);
			Gl.glBegin(Gl.GL_POINTS);
			Gl.glVertex2d(p.pos.x, p.pos.y);
			Gl.glEnd();
			Gl.glPointSize(1.0f);
		}
		private void RenderCalibrateNode(Point2 p, float size, Color c)
		{
			double x = p.pos.x, y = p.pos.y;
			Gl.glColor3ub(c.R, c.G, c.B);
			Gl.glLineWidth(1.0f);
			Gl.glBegin(Gl.GL_LINES);
			Gl.glVertex2d(x - size, y - size);
			Gl.glVertex2d(x + size, y - size);
			Gl.glVertex2d(x + size, y - size);
			Gl.glVertex2d(x + size, y + size);
			Gl.glVertex2d(x + size, y + size);
			Gl.glVertex2d(x - size, y + size);
			Gl.glVertex2d(x - size, y + size);
			Gl.glVertex2d(x - size, y - size);
			Gl.glEnd();

			Gl.glLineWidth(2.0f);
			Gl.glBegin(Gl.GL_LINES);
			Gl.glVertex2d(x - size, y - size);
			Gl.glVertex2d(x + size, y + size);
			
			Gl.glVertex2d(x - size, y + size);
			Gl.glVertex2d(x + size, y - size);
			Gl.glEnd();
			Gl.glPointSize(1.0f);
		}
		private void HighlightElements2D()
		{
			if (this.selectionstartpoint != null &&
					this.selectionendpoint != null)
			{
				Gl.glColor3ub(0, 192, 0);
				Gl.glRectd(selectionstartpoint.pos.x, selectionstartpoint.pos.y,
					selectionendpoint.pos.x, selectionendpoint.pos.y);
			}

			if (this.selectedpolyface != null)
			{
				this.RenderShape2D(this.selectedpolyface, 1.0f, Color.Red, true);
			}
		}
		private void HighlightElements3D()
		{
			if (true)
			{
				foreach (PolyProxy px in this.selectedproxies)
				{
					this.RenderPolyProxy3D(px, Color.Pink, FlatProxyLineWidth);
				}
				{
					int j = 1;
					foreach (Polygon plg in this.selectedfaces)
					{
						this.RenderPolygon3D(plg, this.colorMall[j++], true);
					}
				}
				foreach (PolyProxy proxy in this.selectedproxies)
				{
					this.RenderPolyProxy3D(proxy, this.BoxSelectColor, FlatProxyLineWidth);
					foreach (PolyProxy symm in proxy.mirrorsymmetry)
					{
						this.RenderPolyProxy3D(symm, this.BoxSelectColor, FlatProxyLineWidth);
					}
				}

			}
			if (this.rigidmetaphor != null)
				this.RenderRigidMetaphor(this.rigidmetaphor);
			if (this.selectedproxy != null)
			{
			//	this.RenderPolyProxy3D(this.selectedproxy, this.BoxSelectColor, FlatProxyLineWidth);
				if (this.selectedproxy.bendingmetaphors.Count == 0)
				{
			//		this.HighlightSelectedMetaphor();
					this.RenderProxyEditingMetaphor(this.selectedproxy);
				}

				foreach (PolyProxy proxy in this.polyproxies)
				{
					if (proxy.bendingmetaphors.Count > 0)
					{
						foreach (BendingMetaphor bmetaphor in proxy.bendingmetaphors)
						{
							this.Render3DBall(bmetaphor.center.pos, Color.DodgerBlue);
							this.RenderBendingMetaphor(bmetaphor);
						}
					}
					//if (proxy.bendingslices != null)
					//    foreach (Polygon plg in proxy.bendingslices)
					//    {
					//        this.RenderPolygon3D(plg, Color.Lime, true);
					//    }
					if (proxy.slicingboxes != null)
						foreach (PolyProxy vbox in proxy.slicingboxes)
						{
							this.RenderPolyProxy3D(vbox, Color.Lime, FlatProxyLineWidth);
						}
				}
			}


			if (this.selectedpolyface != null)
			{
				this.RenderPolygon3D(this.selectedpolyface, Color.Red, false);
			}
		}
		private void HighLightSelection()
		{
			Gl.glPushMatrix();

			Gl.glViewport(0, this.parentview.Size.Height - this.glviewporth, this.glviewportw, this.glviewporth);

			Gl.glMatrixMode(Gl.GL_PROJECTION);
			Gl.glLoadIdentity();

			Glu.gluOrtho2D(0, this.glviewportw, this.glviewporth, 0);

			Gl.glMatrixMode(Gl.GL_MODELVIEW);
			Gl.glLoadIdentity();

			Gl.glPushMatrix();

			Gl.glTranslated(imgpos.x, imgpos.y, 0.0);
			Gl.glScaled(this.sacleRatio, this.sacleRatio, this.sacleRatio);
			Gl.glColor3f(1.0f, 1.0f, 1.0f);

			this.HighlightElements2D();

			Gl.glPopMatrix();

			Gl.glPopMatrix();
		}

		// for tests
		public static List<Vector3d> userballs = new List<Vector3d>();
		private void Render3DBall(Vector3d position, Color c)
		{
			Gl.glShadeModel(Gl.GL_SMOOTH);

			Gl.glClearDepth(1.0f);							// Depth buffer setup             
			Gl.glEnable(Gl.GL_DEPTH_TEST);					// Enables Depth Testing             
			Gl.glDepthFunc(Gl.GL_LEQUAL);					// The Type Of Depth Test To Do     
			Gl.glHint(Gl.GL_PERSPECTIVE_CORRECTION_HINT, Gl.GL_NICEST);     /* Really Nice Perspective Calculations */
			Gl.glEnable(Gl.GL_CULL_FACE);
			Gl.glPolygonOffset(1f, 1f);

			Gl.glPushAttrib(Gl.GL_POLYGON_BIT | Gl.GL_LIGHTING_BIT);
			Gl.glPolygonMode(Gl.GL_FRONT_AND_BACK, Gl.GL_FILL);


			Glu.GLUquadric quad = Glu.gluNewQuadric();
			double radius = this.ballsize;

			Gl.glColor3ub(c.R, c.G, c.B);
			Gl.glPushMatrix();
			Gl.glTranslated(position.x, position.y, position.z);
			Glu.gluSphere(quad, radius, 30, 20);
			Gl.glPopMatrix();

			Glu.gluDeleteQuadric(quad);
			Gl.glPopAttrib();
			Gl.glDisable(Gl.GL_CULL_FACE);
			Gl.glDisable(Gl.GL_DEPTH_TEST);
		}
		private void RenderCalibratingSegment(Point2 s, Point2 t, Color cc)
		{
			Gl.glDisable(Gl.GL_LIGHTING);
			Gl.glEnable(Gl.GL_LINE_SMOOTH);
			Gl.glHint(Gl.GL_LINE_SMOOTH_HINT, Gl.GL_NICEST);
			Vector2d u = s.pos, v = t.pos;
			double L = (u - v).Length();
			int intervals = (int)(L / 10);
			int k = intervals;
			double step = 1.0 / k;
			Color[] c = new Color[2] {
				Color.WhiteSmoke,
				Color.Black
			};
			Gl.glLineWidth(3.0f);
			Gl.glBegin(Gl.GL_LINES);
			for (int i = 0; i < k - 1; ++i)
			{
				Color clr = c[i % 2];
				Gl.glColor3ub(clr.R, clr.G, clr.B);
				double r1 = i * step, r2 = (i + 1) * step;
				Vector2d pos1 = s.pos * (1 - r1) + r1 * t.pos;
				Vector2d pos2 = s.pos * (1 - r2) + r2 * t.pos;
				Gl.glVertex2d(pos1.x, pos1.y);
				Gl.glVertex2d(pos2.x, pos2.y);
			}
			Gl.glEnd();
			Gl.glLineWidth(1.0f);
			
		//	this.RenderCalibrateNode(s, this.CalibratingNodeSize, cc);
		//	this.RenderCalibrateNode(t, this.CalibratingNodeSize, cc);

			Gl.glDisable(Gl.GL_LIGHTING);
		}
		private void RenderSolidSegment(LineSegment s, int colorid)
		{
			Gl.glEnable(Gl.GL_LINE_SMOOTH);
			Gl.glHint(Gl.GL_LINE_SMOOTH_HINT, Gl.GL_NICEST);
			switch (colorid)
			{
				case 0:
					Gl.glColor3ub(255, 0, 0);
					break;
				case 1:
					Gl.glColor3ub(0, 255, 0);
					break;
				case 2:
					Gl.glColor3ub(0, 0, 255);
					break;
				case 3:
					Gl.glColor3ub(255, 255, 0);
					break;
				default:
					Gl.glColor3ub(255, 20, 147);
					break;
			}
			Vector2d u = s.u.pos, v = s.v.pos;
			double L = (u - v).Length();
			int intervals = (int)(L / 10);
			int k = intervals;
			double step = 1.0/k;
			Gl.glLineWidth(2.0f);
			Gl.glBegin(Gl.GL_LINES);
			Gl.glVertex2d(s.u.pos.x, s.u.pos.y);
			Gl.glVertex2d(s.v.pos.x, s.v.pos.y);
			Gl.glEnd();
			//Gl.glColor3ub(255, 164, 255);
			//Gl.glPointSize(5.0f);
			//Gl.glBegin(Gl.GL_POINTS);
			//Gl.glVertex2d(s.u.pos.x, s.u.pos.y);
			//Gl.glVertex2d(s.v.pos.x, s.v.pos.y);
			//Gl.glEnd();
		}
		private void RenderVanishingLines()
		{
			if (this.vp2d == null) return;
			int n = this.vp2d.Length;
			Gl.glColor3ub(0, 255, 0);
			Gl.glLineWidth(2.0f);
			Gl.glBegin(Gl.GL_LINES);
			for (int i = 0; i < n; ++i)
			{
				Gl.glVertex2d(this.vp2d[i].x, this.vp2d[i].y);
				Gl.glVertex2d(this.vp2d[(i+1)%n].x, this.vp2d[(i+1)%n].y);
			}
			Gl.glEnd();
			Gl.glLineWidth(1.0f);
			Gl.glPointSize(8.0f);
			Gl.glBegin(Gl.GL_POINTS);
			Gl.glColor3ub(255, 0, 0);
			Gl.glVertex2d(this.vp2d[0].x, this.vp2d[0].y);
			Gl.glColor3ub(0, 255, 0);
			Gl.glVertex2d(this.vp2d[1].x, this.vp2d[1].y);
			Gl.glColor3ub(0, 0, 255);
			Gl.glVertex2d(this.vp2d[2].x, this.vp2d[2].y);
			Gl.glEnd();
			Gl.glPointSize(1.0f);
		}
		private void RenderShape3D(Vector3d[] points, Color c)
		{
			if (points == null) return;

			//Gl.glEnable(Gl.GL_BLEND);
			//Gl.glBlendFunc(Gl.GL_SRC_ALPHA, Gl.GL_ONE_MINUS_SRC_ALPHA);
			//Gl.glEnable(Gl.GL_LINE_SMOOTH);
			//Color temp = Color.FromArgb(60, c);
			//Gl.glColor4ub(temp.R, temp.G, temp.B, temp.A);
			//Gl.glBegin(Gl.GL_POLYGON);
			//foreach (Vector3d p in points)
			//{
			//    Gl.glVertex3d(p.x, p.y,p.z);
			//}
			//Gl.glEnd();
			//Gl.glDisable(Gl.GL_BLEND);

			Gl.glDisable(Gl.GL_LIGHTING);
			int k = points.Length;
			Gl.glLineWidth(2.0f);
			Gl.glColor3ub(c.R, c.G, c.B);
			Gl.glBegin(Gl.GL_LINES);
			for (int j = 0; j < k; ++j)
			{
				Vector3d u = points[j], v = points[(j + 1) % k];
				Gl.glVertex3d(u.x, u.y,u.z);
				Gl.glVertex3d(v.x, v.y,v.z);
			}
			Gl.glEnd();
			Gl.glLineWidth(1.0f);
			Gl.glEnable(Gl.GL_LIGHTING);
		}
		private void RenderShape2D(Shape2D shape, float linewidth, Color c, bool outlineonly)
		{
			if (shape == null) return;
			Gl.glDisable(Gl.GL_LIGHTING);
			if (!outlineonly)
			{
				Gl.glEnable(Gl.GL_BLEND);
				Gl.glBlendFunc(Gl.GL_SRC_ALPHA, Gl.GL_ONE_MINUS_SRC_ALPHA);
				Gl.glEnable(Gl.GL_LINE_SMOOTH);
				Color temp = Color.FromArgb(60, c);
				Gl.glColor4ub(temp.R, temp.G, temp.B, temp.A);
				switch (shape.shaptype)
				{
					case ShapeType2D.QUAD:
						Gl.glBegin(Gl.GL_QUADS);
						foreach (Point2 p in shape.points)
						{
							Gl.glVertex2d(p.pos.x, p.pos.y);
						}
						Gl.glEnd();
						break;
					case ShapeType2D.POLYGON:
						Gl.glBegin(Gl.GL_POLYGON);
						foreach (Point2 p in shape.points)
						{
							Gl.glVertex2d(p.pos.x, p.pos.y);
						}
						Gl.glEnd();
						break;
				}
				Gl.glDisable(Gl.GL_BLEND);
			}
			
			List<Point2> pts = shape.points;
			int k = pts.Count;
			Gl.glColor3ub(c.R, c.G, c.B);
			Gl.glLineWidth(linewidth);
			Gl.glBegin(Gl.GL_LINES);
			for (int j = 0; j < k; ++j)
			{
				Vector2d u = pts[j].pos, v = pts[(j + 1) % k].pos;
				Gl.glVertex2d(u.x, u.y);
				Gl.glVertex2d(v.x, v.y);
			}
			Gl.glEnd();
			Gl.glLineWidth(1.0f);
			Gl.glEnable(Gl.GL_LIGHTING);
		}
		private void RenderPolygon3D(Polygon plg, Color c, bool outlineonly)
		{
			if (plg == null) return;
			Gl.glDisable(Gl.GL_LIGHTING);
			if (!outlineonly)
			{
				Gl.glEnable(Gl.GL_BLEND);
				Gl.glBlendFunc(Gl.GL_SRC_ALPHA, Gl.GL_ONE_MINUS_SRC_ALPHA);
				Gl.glEnable(Gl.GL_LINE_SMOOTH);
				Color temp = Color.FromArgb(60, c);
				Gl.glColor4ub(temp.R, temp.G, temp.B, temp.A);

				Gl.glBegin(Gl.GL_POLYGON);
				foreach (Point3 p in plg.points3)
				{
					Gl.glVertex3d(p.pos.x, p.pos.y, p.pos.z);
				}
				Gl.glEnd();
				Gl.glDisable(Gl.GL_BLEND);
			}

			List<Point3> pts = plg.points3;
			int k = pts.Count;
			Gl.glBegin(Gl.GL_LINES);
			Gl.glColor3ub(c.R, c.G, c.B);
			for (int j = 0; j < k; ++j)
			{
				Vector3d u = pts[j].pos, v = pts[(j + 1) % k].pos;
				Gl.glVertex3d(u.x, u.y, u.z);
				Gl.glVertex3d(v.x, v.y, v.z);
			}
			Gl.glEnd();
			Gl.glEnable(Gl.GL_LIGHTING);
		}
		private void RenderPolyProxies3D(List<PolyProxy> proxies, Color c)
		{
			if (proxies.Count < 1) return; // int index = 0;
			if (this.RenderLineProxies)
			{
				Color cc = this.colorMall[5];
				foreach (PolyProxy proxy in proxies)
				{
					this.RenderPolyProxy3D(proxy, cc, FlatProxyLineWidth);
				}
			}
			if (this.RenderSolidProxies)
			{
				Gl.glEnable(Gl.GL_LIGHTING);
				foreach (PolyProxy proxy in proxies)
					this.RenderProxySolidOutLine(proxy, c);

				foreach (PolyProxy proxy in proxies)
					this.RenderPolyProxy3D_Solid(proxy, c, 20, false);

				foreach (PolyProxy proxy in proxies)
				    this.RenderPolyProxy3D_Solid(proxy, c, 10, true);

				foreach (PolyProxy proxy in proxies)
					this.RenderProxySolidOutLine(proxy, c);
				Gl.glDisable(Gl.GL_LIGHTING);
			}
		}
		private void RenderPolyProxy3D(PolyProxy proxy, Color c, float width)
		{
			Gl.glDisable(Gl.GL_LIGHTING);
			Gl.glColor3ub(c.R, c.G, c.B);
			Gl.glEnable(Gl.GL_LINE_SMOOTH);
			Gl.glLineWidth(width);
			Gl.glBegin(Gl.GL_LINES);
			foreach (Polygon poly in proxy.polygons)
			{
				for (int i = 0; i < poly.points3.Count; ++i)
				{
					Point3 p = poly.points3[i];
					Point3 q = poly.points3[(i + 1) % poly.points3.Count];
					Gl.glVertex3d(p.pos.x, p.pos.y, p.pos.z);
					Gl.glVertex3d(q.pos.x, q.pos.y, q.pos.z);
				}
			}
			Gl.glEnd();
			Gl.glLineWidth(1.0f);

			//Gl.glEnable(Gl.GL_BLEND);
			//Gl.glBlendFunc(Gl.GL_SRC_ALPHA, Gl.GL_ONE_MINUS_SRC_ALPHA);
			//Color temp = Color.FromArgb(alpha, c);
			//Gl.glColor4ub(temp.R, temp.G, temp.B, temp.A);
			//for (int i = proxy.polygons.Count - 1; i >= 0; --i)
			//{
			//    Gl.glBegin(Gl.GL_POLYGON);
			//    Polygon poly = proxy.polygons[i];
			//    foreach (Point3 pt in poly.points3)
			//    {
			//        Gl.glVertex3d(pt.pos.x, pt.pos.y, pt.pos.z);
			//    }
			//    Gl.glEnd();
			//}
			//Gl.glDisable(Gl.GL_BLEND);
			Gl.glEnable(Gl.GL_LIGHTING);
		}
		private void InitLighting()
		{
			Gl.glEnable(Gl.GL_LIGHTING);
			Gl.glEnable(Gl.GL_NORMALIZE);

			float diffuse = 1.0f;
			float specular = 0.1f;
			int shinness = 128;

			float[] LightDiffuse = { diffuse, diffuse, diffuse, 1.0f };
			float[] LightSpecular = { specular, specular, specular, 1f };
			float[] LightPosition = { (float)lightpos.x, (float)lightpos.y, (float)lightpos.z, 1f };
			Gl.glLightfv(Gl.GL_LIGHT0, Gl.GL_DIFFUSE, LightDiffuse);
			Gl.glLightfv(Gl.GL_LIGHT0, Gl.GL_SPECULAR, LightSpecular);
			Gl.glLightfv(Gl.GL_LIGHT0, Gl.GL_POSITION, LightPosition);
			Gl.glEnable(Gl.GL_LIGHT0);


			Gl.glEnable(Gl.GL_COLOR_MATERIAL);
			Gl.glColorMaterial(Gl.GL_FRONT, Gl.GL_AMBIENT_AND_DIFFUSE);
			//Gl.glColorMaterial(Gl.GL_FRONT, Gl.GL_SPECULAR);
			//Gl.glColorMaterial(Gl.GL_FRONT, Gl.GL_DIFFUSE);
			float[] mdiffuse = { 0.9f, 0.9f, 0.9f, 1f };
			float[] mspecular = { 0.9f, 0.9f, 0.99f, 1f };
			float[] mambient = { 0.9f, 0.9f, 0.99f, 1f };
			Gl.glMaterialfv(Gl.GL_FRONT, Gl.GL_SPECULAR, mspecular);
			Gl.glMaterialfv(Gl.GL_FRONT, Gl.GL_DIFFUSE, mdiffuse);
			Gl.glMaterialfv(Gl.GL_FRONT, Gl.GL_AMBIENT, mambient);
			Gl.glMaterialf(Gl.GL_FRONT, Gl.GL_SHININESS, shinness);

		}
		private void DisableLighting()
		{
			Gl.glDisable(Gl.GL_LIGHTING);
			Gl.glDisable(Gl.GL_NORMALIZE);
			Gl.glDisable(Gl.GL_COLOR_MATERIAL);
		}
		private void RenderProxySolidOutLine(PolyProxy proxy, Color c)
		{
			Gl.glShadeModel(Gl.GL_SMOOTH);

			Gl.glClearDepth(1.0f);							// Depth buffer setup             
			Gl.glEnable(Gl.GL_DEPTH_TEST);					// Enables Depth Testing             
			Gl.glDepthFunc(Gl.GL_LEQUAL);					// The Type Of Depth Test To Do     
			Gl.glHint(Gl.GL_PERSPECTIVE_CORRECTION_HINT, Gl.GL_NICEST);     /* Really Nice Perspective Calculations */
			Gl.glEnable(Gl.GL_CULL_FACE);
			Gl.glPolygonOffset(1f, 1f);

			Gl.glPushAttrib(Gl.GL_POLYGON_BIT | Gl.GL_LIGHTING_BIT);
			Gl.glPolygonMode(Gl.GL_FRONT_AND_BACK, Gl.GL_FILL);

			Color curvecolor = Color.FromArgb(c.R, c.G, c.B);
			Gl.glColor3ub(curvecolor.R, curvecolor.G, curvecolor.B);
			Glu.GLUquadric quad = Glu.gluNewQuadric();
			double radius = this.proxyrendersize;// (this.roomlayout.box.points3[2].pos - this.roomlayout.box.points3[0].pos).Length() * 0.0015;
			foreach (Point3 p in proxy.points3)
			{
				Gl.glPushMatrix();
				Gl.glTranslated(p.pos.x, p.pos.y, p.pos.z);
				Glu.gluSphere(quad, radius, 30, 20);
				Gl.glPopMatrix();
			}

			int count = 0;
			foreach (Polygon poly in proxy.polygons)
			{
				for (int i = 0; i < poly.points3.Count; ++i)
				{
					Point3 p = poly.points3[i];
					Point3 q = poly.points3[(i + 1) % poly.points3.Count];

					Vector3d pq = q.pos - p.pos;
					double h = pq.Length();
					double angle;
					angle = Math.Acos(pq.z / h) / Math.PI * 180;

					Gl.glPushMatrix();
					Gl.glTranslated(p.pos.x, p.pos.y, p.pos.z);
					Gl.glRotated(angle, -pq.y, pq.x, 0);
					Glu.gluCylinder(quad, radius, radius, h, 30, 20);
					Gl.glPopMatrix();
				}
				if (count++ >= 1)
					break;
			}
			if (proxy.polygons.Count > 1)
			{
				Polygon poly1 = proxy.polygons[0], poly2 = proxy.polygons[1];
				for (int i = 0; i < poly1.points.Count; ++i)
				{
					Point3 p = poly1.points3[i];
					Point3 q = poly2.points3[i];

					Vector3d pq = q.pos - p.pos;
					double h = pq.Length();
					double angle;
					angle = Math.Acos(pq.z / h) / Math.PI * 180;

					Gl.glPushMatrix();
					Gl.glTranslated(p.pos.x, p.pos.y, p.pos.z);
					Gl.glRotated(angle, -pq.y, pq.x, 0);
					Glu.gluCylinder(quad, radius, radius, h, 30, 20);
					Gl.glPopMatrix();
				}
			}
			Glu.gluDeleteQuadric(quad);
			Gl.glPopAttrib();
			Gl.glDisable(Gl.GL_CULL_FACE);
			Gl.glDisable(Gl.GL_DEPTH_TEST);
		}
		private void RenderPolyProxy3D_Solid(PolyProxy proxy, Color c, byte alpha, bool withdepthtest)
		{
			Color temp = Color.FromArgb(alpha, Color.Blue);
			Gl.glShadeModel(Gl.GL_SMOOTH);

			Gl.glClearDepth(1.0f);							// Depth buffer setup 
			if (!withdepthtest)
			{
				Gl.glDisable(Gl.GL_DEPTH_TEST);
			}
			else
			{
				Gl.glEnable(Gl.GL_DEPTH_TEST);					// Enables Depth Testing             
				Gl.glDepthFunc(Gl.GL_LEQUAL);					// The Type Of Depth Test To Do     
				temp = Color.FromArgb(0, 0, 0, 0);
			}
			Gl.glHint(Gl.GL_PERSPECTIVE_CORRECTION_HINT, Gl.GL_NICEST);     /* Really Nice Perspective Calculations */
			Gl.glDisable(Gl.GL_CULL_FACE);

			Gl.glEnable(Gl.GL_BLEND);
			Gl.glBlendFunc(Gl.GL_SRC_ALPHA, Gl.GL_ONE_MINUS_SRC_ALPHA);
			Gl.glPolygonMode(Gl.GL_FRONT_AND_BACK, Gl.GL_FILL);
			Gl.glColor4ub(temp.R, temp.G, temp.B, temp.A);
			for (int i = proxy.polygons.Count - 1; i >= 0; --i)
			{
				Polygon poly = proxy.polygons[i];
				Gl.glBegin(Gl.GL_POLYGON);
				Gl.glNormal3d(poly.normal.x, poly.normal.y, poly.normal.z);
				foreach (Point3 pt in poly.points3)
				{
					Gl.glVertex3d(pt.pos.x, pt.pos.y, pt.pos.z);
				}
				Gl.glEnd();
			}
			Gl.glDisable(Gl.GL_BLEND);

			Gl.glEnable(Gl.GL_CULL_FACE);
			Gl.glEnable(Gl.GL_DEPTH_TEST);
			Gl.glDepthFunc(Gl.GL_LEQUAL);
		}
		private void RenderProxyFrames()
		{
			Gl.glShadeModel(Gl.GL_SMOOTH);
			Gl.glClearDepth(1.0f); // Depth buffer setup
			Gl.glEnable(Gl.GL_DEPTH_TEST); // Disables Depth Testing
			Gl.glDepthFunc(Gl.GL_LEQUAL); // The Type Of Depth Test To Do
			Gl.glHint(Gl.GL_PERSPECTIVE_CORRECTION_HINT, Gl.GL_NICEST); /* Really Nice Perspective Calculations */
			Gl.glEnable(Gl.GL_CULL_FACE);
			Gl.glPolygonOffset(1f, 1f);
			Gl.glPushAttrib(Gl.GL_POLYGON_BIT | Gl.GL_LIGHTING_BIT);
			Gl.glPolygonMode(Gl.GL_FRONT_AND_BACK, Gl.GL_FILL);

			foreach (PolyProxy proxy in this.polyproxies)
			{
				Render3dArrow(proxy.center.pos, proxy.frame.x, Color.Red);
				Render3dArrow(proxy.center.pos, proxy.frame.y, Color.Green);
				Render3dArrow(proxy.center.pos, proxy.frame.z, Color.Blue);
			}

			Gl.glPopAttrib();
			Gl.glDisable(Gl.GL_CULL_FACE);
			Gl.glDisable(Gl.GL_DEPTH_TEST);
		}
		private void HighlightSelectedMetaphor()
		{
			if (this.selectedmetaphor == null) return;

			Gl.glShadeModel(Gl.GL_SMOOTH);
			Gl.glClearDepth(1.0f); // Depth buffer setup
			Gl.glEnable(Gl.GL_DEPTH_TEST); // Disables Depth Testing
			Gl.glDepthFunc(Gl.GL_LEQUAL); // The Type Of Depth Test To Do
			Gl.glHint(Gl.GL_PERSPECTIVE_CORRECTION_HINT, Gl.GL_NICEST); /* Really Nice Perspective Calculations */
			Gl.glEnable(Gl.GL_CULL_FACE);
			Gl.glPolygonOffset(1f, 1f);
			Gl.glPushAttrib(Gl.GL_POLYGON_BIT | Gl.GL_LIGHTING_BIT);
			Gl.glPolygonMode(Gl.GL_FRONT_AND_BACK, Gl.GL_FILL);

			this.Render3dArrow(this.selectedmetaphor.center.pos, selectedmetaphor.axis, Color.Yellow);

			if (this.selectedmetaphor.type == 1) // rotation metaphor
			{
				RotationMetaphor rmetephor = (RotationMetaphor)this.selectedmetaphor;
				this.RenderRotationMetaphor(rmetephor.center.pos, rmetephor.axis, Color.LimeGreen, rmetephor.radius);
				if (rmetephor.start != null)
					RenderSolidSphere(rmetephor.start.pos, this.ballsize, Color.Pink);
				if (rmetephor.end != null)
					RenderSolidSphere(rmetephor.end.pos, this.ballsize, Color.Peru);
			}

			Gl.glPopAttrib();
			Gl.glDisable(Gl.GL_CULL_FACE);
			Gl.glDisable(Gl.GL_DEPTH_TEST);
		}
		private void RenderProxyEditingMetaphor(PolyProxy proxy)
		{
			Gl.glShadeModel(Gl.GL_SMOOTH);
			Gl.glClearDepth(1.0f); // Depth buffer setup
			Gl.glEnable(Gl.GL_LIGHTING);
			Gl.glEnable(Gl.GL_DEPTH_TEST); // Disables Depth Testing
			Gl.glDepthFunc(Gl.GL_LEQUAL); // The Type Of Depth Test To Do
			Gl.glHint(Gl.GL_PERSPECTIVE_CORRECTION_HINT, Gl.GL_NICEST); /* Really Nice Perspective Calculations */
			Gl.glEnable(Gl.GL_CULL_FACE);
			Gl.glPolygonOffset(1f, 1f);
			Gl.glPushAttrib(Gl.GL_POLYGON_BIT | Gl.GL_LIGHTING_BIT);
			Gl.glPolygonMode(Gl.GL_FRONT_AND_BACK, Gl.GL_FILL);

			//if (parentview.CurrentMode == ImageView.EnumOperationMode.ControllerAnisotropicScaling)
			//    foreach (TranslationMetaphor metaphor in proxy.transmetaphors)
			//    {
			//        Color c = Color.OrangeRed;
			//        Render3dArrow(metaphor.center.pos, metaphor.axis, c);
			//    }
		//	else 
			if (parentview.CurrentMode == ImageView.EnumOperationMode.ControllerDragging)
			{
				RotationMetaphor rmetephor = proxy.rotationmetaphor;
				if (rmetephor != null)
				{
					RenderRotationMetaphor(rmetephor.center.pos, rmetephor.axis, Color.LimeGreen, rmetephor.radius);
					if (rmetephor.start != null)
						RenderSolidSphere(rmetephor.start.pos, this.ballsize, Color.Pink);
					if (rmetephor.end != null)
						RenderSolidSphere(rmetephor.end.pos, this.ballsize, Color.Peru);
				}
			}

			if (this.selectedmetaphor != null)
				Render3dArrow(selectedmetaphor.center.pos, selectedmetaphor.axis, Color.Yellow);

			Gl.glPopAttrib();
			Gl.glDisable(Gl.GL_CULL_FACE);
			Gl.glDisable(Gl.GL_DEPTH_TEST);
			Gl.glDisable(Gl.GL_LIGHTING);
		}
		private void RenderBendingMetaphor(BendingMetaphor bmetaphor)
		{
			Gl.glShadeModel(Gl.GL_SMOOTH);
			Gl.glClearDepth(1.0f); // Depth buffer setup
			Gl.glEnable(Gl.GL_DEPTH_TEST); // Disables Depth Testing
			Gl.glDepthFunc(Gl.GL_LEQUAL); // The Type Of Depth Test To Do
			Gl.glHint(Gl.GL_PERSPECTIVE_CORRECTION_HINT, Gl.GL_NICEST); /* Really Nice Perspective Calculations */
			Gl.glEnable(Gl.GL_CULL_FACE);
			Gl.glPolygonOffset(1f, 1f);
			Gl.glPushAttrib(Gl.GL_POLYGON_BIT | Gl.GL_LIGHTING_BIT);
			Gl.glPolygonMode(Gl.GL_FRONT_AND_BACK, Gl.GL_FILL);

			foreach (TranslationMetaphor metaphor in bmetaphor.metaphors)
			{
				Color c = Color.OrangeRed;
				Render3dArrow(metaphor.center.pos, metaphor.axis, c);
			}

			if (this.selectedmetaphor != null)
				Render3dArrow(selectedmetaphor.center.pos, selectedmetaphor.axis, Color.Yellow);

			Gl.glPopAttrib();
			Gl.glDisable(Gl.GL_CULL_FACE);
			Gl.glDisable(Gl.GL_DEPTH_TEST);
		}
		private void RenderRigidMetaphor(RigidTransMetaphor rmetaphor)
		{
			Gl.glShadeModel(Gl.GL_SMOOTH);
			Gl.glClearDepth(1.0f); // Depth buffer setup
			Gl.glEnable(Gl.GL_DEPTH_TEST); // Disables Depth Testing
			Gl.glDepthFunc(Gl.GL_LEQUAL); // The Type Of Depth Test To Do
			Gl.glHint(Gl.GL_PERSPECTIVE_CORRECTION_HINT, Gl.GL_NICEST); /* Really Nice Perspective Calculations */
			Gl.glEnable(Gl.GL_CULL_FACE);
			Gl.glPolygonOffset(1f, 1f);
			Gl.glPushAttrib(Gl.GL_POLYGON_BIT | Gl.GL_LIGHTING_BIT);
			Gl.glPolygonMode(Gl.GL_FRONT_AND_BACK, Gl.GL_FILL);

			foreach (TranslationMetaphor metaphor in rmetaphor.transmetaphors)
			{
				Color c = Color.OrangeRed;
				Render3dArrow(metaphor.center.pos, metaphor.axis, c);
			}
			foreach (RotationMetaphor rmt in rmetaphor.rotatmetaphors)
			{
				this.RenderRotationMetaphor(rmt.center.pos, rmt.axis, Color.Yellow, rmt.radius);
			}

			if (this.selectedmetaphor != null)
				Render3dArrow(selectedmetaphor.center.pos, selectedmetaphor.axis, Color.Yellow);

			Gl.glPopAttrib();
			Gl.glDisable(Gl.GL_CULL_FACE);
			Gl.glDisable(Gl.GL_DEPTH_TEST);
		}
		private void RenderRotationMetaphor(Vector3d p, Vector3d dir, Color c, double radius)
		{
			double angle = Math.Acos(dir.z / dir.Length()) / Math.PI * 180;
			double len1 = radius;
			double r1 = radius, r2 = r1 / 100;

			Glu.GLUquadric quad = Glu.gluNewQuadric();

			Gl.glPushMatrix();
			Gl.glTranslated(p.x, p.y, p.z);
			Gl.glRotated(angle, -dir.y, dir.x, 0);

			Gl.glEnable(Gl.GL_BLEND);
			Gl.glBlendFunc(Gl.GL_SRC_ALPHA, Gl.GL_ONE_MINUS_SRC_ALPHA);
			Gl.glPolygonMode(Gl.GL_FRONT_AND_BACK, Gl.GL_FILL);
			Gl.glColor4ub(c.R, c.G, c.B, 100);
			Glu.gluDisk(quad, 0, r1, 40, 1);
			Gl.glColor4ub(c.R, c.G, c.B, 150);
			RenderTorus(r1, r2, 40, 40);
			Gl.glDisable(Gl.GL_BLEND);

			Gl.glPopMatrix();
			Glu.gluDeleteQuadric(quad);

			this.Render3dArrow(p, dir, Color.DodgerBlue);
		}
		private void Render3dArrow(Vector3d p, Vector3d dir, Color c)
		{
			double len1 = this.proxyrendersize * 20;
			double len2 = len1 / 2;
			Vector3d q1 = p + dir * len1;

			double r1 = this.arrowsize, r2 = r1 * 2;
			double angle = Math.Acos(dir.z / dir.Length()) * 180 / Math.PI;
			Glu.GLUquadric quad = Glu.gluNewQuadric();

			Color ocolor = Color.SkyBlue;
			Gl.glColor3ub(ocolor.R, ocolor.G, ocolor.B);
			Gl.glPushMatrix();
			Gl.glTranslated(p.x, p.y, p.z);
			Glu.gluSphere(quad, r2, 40, 40);
			Gl.glPopMatrix();

			Gl.glColor3ub(c.R, c.G, c.B);
			Gl.glPushMatrix();
			Gl.glTranslated(p.x, p.y, p.z);
			Gl.glRotated(angle, -dir.y, dir.x, 0);
			Glu.gluCylinder(quad, r1, r1, len1, 40, 40);
			Gl.glPopMatrix();

			Gl.glPushMatrix();
			Gl.glTranslated(q1.x, q1.y, q1.z);
			Gl.glRotated(angle, -dir.y, dir.x, 0);
			Glu.gluCylinder(quad, r2, 0.0001, len2, 40, 40);
			Gl.glPopMatrix();

			Glu.gluDeleteQuadric(quad);
		}
		private void RenderTorus(double majorRadius, double minorRadius, int numMajor, int numMinor)
		{
			Vector3d vNormal = new Vector3d();
			double majorStep = 2.0f * Math.PI / numMajor;
			double minorStep = 2.0f * Math.PI / numMinor;
			int i, j;

			for (i = 0; i < numMajor; i++)
			{
				double a0 = i * majorStep;
				double a1 = a0 + majorStep;
				float x0 = (float)Math.Cos(a0);
				float y0 = (float)Math.Sin(a0);
				float x1 = (float)Math.Cos(a1);
				float y1 = (float)Math.Sin(a1);

				Gl.glBegin(Gl.GL_TRIANGLE_STRIP);
				for (j = 0; j <= numMinor; j++)
				{
					double b = j * minorStep;
					double c = Math.Cos(b);
					double r = minorRadius * c + majorRadius;
					double z = minorRadius * Math.Sin(b);

					//First point
					Gl.glTexCoord2f((float)i / (float)(numMajor), (float)(j) / (float)(numMinor));
					vNormal[0] = x0 * c;
					vNormal[1] = y0 * c;
					vNormal[2] = z / minorRadius;
					vNormal = vNormal.Normalize();
					Gl.glNormal3d(vNormal.x, vNormal.y, vNormal.z);
					Gl.glVertex3d(x0 * r, y0 * r, z);

					Gl.glTexCoord2f((float)(i + 1) / (float)(numMajor), (float)(j) / (float)(numMinor));
					vNormal[0] = x1 * c;
					vNormal[1] = y1 * c;
					vNormal[2] = z / minorRadius;
					vNormal = vNormal.Normalize();
					Gl.glNormal3d(vNormal.x, vNormal.y, vNormal.z);
					Gl.glVertex3d(x1 * r, y1 * r, z);
				}
				Gl.glEnd();
			}
		}
		private void RenderSolidSphere(Vector3d p, double radius, Color c)
		{
			Glu.GLUquadric quad = Glu.gluNewQuadric();
			Gl.glColor3ub(c.R, c.G, c.B);
			Gl.glPushMatrix();
			Gl.glTranslated(p.x, p.y, p.z);
			Glu.gluSphere(quad, radius, 30, 30);
			Gl.glPopMatrix();
			Glu.gluDeleteQuadric(quad);
		}
		private void RenderLines(List<LineSegment> clusters)
		{
			Gl.glDisable(Gl.GL_LIGHTING);
			Gl.glLineWidth(this.FlatProxyLineWidth);
			Gl.glBegin(Gl.GL_LINES);
			foreach (LineSegment line in clusters)
			{
				Color c = Color.Yellow;
				switch (line.vanishingindex)
				{
					case 0:
						c = this.colorMall[0];
						break;
					case 1:
						c = this.colorMall[1];
						break;
					case 2:
						c = this.colorMall[2];
						break;
					case 3:
						c = this.colorMall[3];
						break;
				}
				Gl.glVertex2d(line.u.pos.x, line.u.pos.y);
				Gl.glVertex2d(line.v.pos.x, line.v.pos.y);
			}
			Gl.glEnd();
			Gl.glLineWidth(2.0f);
			Gl.glEnable(Gl.GL_LIGHTING);
		}
		private void RenderPoint(Vector2d p, Color c, float size)
		{
			Gl.glPointSize(size);
			Gl.glColor3ub(c.R, c.G, c.B);
			Gl.glBegin(Gl.GL_POINTS);
			Gl.glVertex2d(p.x,p.y);
			Gl.glEnd();
			Gl.glPointSize(1.0f);
		}
		private void RenderSelectedSegmentContour()
		{
			if (this.selectedproxy == null) return;
			int index = this.polyproxies.IndexOf(this.selectedproxy);
			List<Point2[]> profile = this.objectprofiles[index];
			Gl.glColor3ub(0, 255, 0);
			Gl.glBegin(Gl.GL_LINES);
			foreach (Point2[] poly in profile)
				for (int i = 0; i < poly.Length; ++i)
				{
					Point2 p = poly[i], q = poly[(i + 1) % poly.Length];
					Gl.glVertex2d(p.pos.x, p.pos.y);
					Gl.glVertex2d(q.pos.x, q.pos.y);
				}
			Gl.glEnd();
		}
		private void RenderObjHexagons()
		{
			if (this.hexagons == null) return;
			int j = 0;
			foreach (Polygon plg in this.hexagons)
			{
				RenderShape2D(plg, 1.0f, this.colorMall[j++ % this.colorMall.Length], false);
			//	RenderShape2D(plg,this.colorMall[j++%this.colorMall.Length]);
			}
		}
		private void RenderShape2D(Shape2D shape, Color c)
		{
			Gl.glColor3ub(c.R, c.G, c.B);
			Gl.glLineWidth(2.0f);
			Gl.glBegin(Gl.GL_LINES);
			foreach (LineSegment line in shape.linesegments)
			{
				Vector2d p = line.u.pos, q = line.v.pos;
				Gl.glVertex2d(p.x, p.y);
				Gl.glVertex2d(q.x, q.y);
			}
			Gl.glEnd();
			Gl.glLineWidth(1.0f);
		}
		private void RenderAllHexagons()
		{
			int j = 0;
			foreach (Polygon plg in this.hexagons)
			{
				Color c = this.colorMall[j++ % this.colorMall.Length];
				RenderHexagon(plg, c);
			}
		}
		private void RenderHexagon(Polygon plg, Color c)
		{
			if (plg.points == null) return;
			//foreach (Point2 pt in plg.points)
			//{
			//    this.RenderCalibrateNode(pt, CalibratingNodeSize, c);
			//}
			int nn = plg.points.Count;
			for (int i = 0; i < nn; ++i)
			{
				Point2 pt1 = plg.points[i], pt2 = plg.points[(i + 1) % nn];
				this.RenderCalibratingSegment(pt1, pt2, c);
			}
		}

		private byte[] Image2Texture(Image<Bgr, byte> image)
		{
			int w = image.Width;
			int h = image.Height;
			byte[] txture = new byte[(w + 1) * (h + 1) * this.imgchannel];
			for (int i = 0; i < w; i++)
			{
				for (int j = 0; j < h; j++)
				{
					int shift = (j * w + i) * imgchannel;
					Bgr bgr = image[j, i];
					txture[shift + 0] = (byte)(bgr.Red);
					txture[shift + 1] = (byte)(bgr.Green);
					txture[shift + 2] = (byte)(bgr.Blue);
				}
			}
			return txture;
		}
		private byte[] Image2Texture(Image<Bgra, byte> image)
		{
			int w = image.Width;
			int h = image.Height;
			byte[] txture = new byte[(w + 1) * (h + 1) * (this.imgchannel+1)];
			for (int i = 0; i < w; i++)
			{
				for (int j = 0; j < h; j++)
				{
					int shift = (j * w + i) * imgchannel;
					Bgra bgr = image[j, i];
					txture[shift + 3] = (byte)(bgr.Alpha);
					txture[shift + 0] = (byte)(bgr.Red);
					txture[shift + 1] = (byte)(bgr.Green);
					txture[shift + 2] = (byte)(bgr.Blue);
					
				}
			}
			return txture;
		}

		#region IDisposable Members
        public void Dispose()
        {
        }
        #endregion

		#region texture mapping
		private bool usegltexturemapping = false;
		public bool UseGLTextureMapping
		{
			get { return usegltexturemapping; }
			set { 
				usegltexturemapping = value;
				this.parentview.Refresh();
			}
		}
		public int currenttexid = 0;
		Matrix<double> cvMat = new Matrix<double>(3, 3);
		private void CreateGLTxture(Image<Bgr,byte> img, uint[] txtid) // create gltexture
		{
			byte[] txture = this.Image2Texture(img);
			// -- create texture --
			Gl.glGenTextures(1, txtid);	// Create The Texture

			// Typical Texture Generation Using Data From The Bitmap
			Gl.glBindTexture(Gl.GL_TEXTURE_2D, txtid[0]);
			Gl.glTexImage2D(Gl.GL_TEXTURE_2D, 0, 4, (int)img.Width, (int)img.Height, 0, Gl.GL_RGBA, Gl.GL_UNSIGNED_BYTE, txture);
			Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_MIN_FILTER, Gl.GL_LINEAR);
			Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_MAG_FILTER, Gl.GL_LINEAR);

			Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_WRAP_S, Gl.GL_CLAMP_TO_EDGE);
			Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_WRAP_T, Gl.GL_CLAMP_TO_EDGE);
			Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_WRAP_R, Gl.GL_CLAMP_TO_EDGE);
			
		}
		private void CreateGLTxture(Image<Bgra, byte> img, uint[] txtid) // create gltexture
		{
			byte[] txture = this.Image2Texture(img);
			Gl.glGenTextures(1, txtid);	// Create The Texture

			// Typical Texture Generation Using Data From The Bitmap
			Gl.glBindTexture(Gl.GL_TEXTURE_2D, txtid[0]);
			Gl.glTexImage2D(Gl.GL_TEXTURE_2D, 0, 4, (int)img.Width, (int)img.Height, 0, Gl.GL_RGBA, Gl.GL_UNSIGNED_BYTE, txture);
			Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_MIN_FILTER, Gl.GL_LINEAR);
			Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_MAG_FILTER, Gl.GL_LINEAR);

			Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_WRAP_S, Gl.GL_CLAMP_TO_EDGE);
			Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_WRAP_T, Gl.GL_CLAMP_TO_EDGE);
			Gl.glTexParameteri(Gl.GL_TEXTURE_2D, Gl.GL_TEXTURE_WRAP_R, Gl.GL_CLAMP_TO_EDGE);

		}
		#endregion


		private void VisualizeCVImage(IImage img, string text)
		{
			ImageViewer viewer = new ImageViewer(img, text);
			viewer.ClientSize = img.Size;
			viewer.Show();
		}
		private void VisualizeCVImage(IImage img)
		{
			this.VisualizeCVImage(img, "temp");
		}
		private void ShowCurrentShadows()
		{
			if (this.resultimage != null)
			{
				Image<Bgr, byte> composition = this.resultimage.Copy();

				this.shadowviewer.Image = composition;
				this.shadowviewer.ClientSize = this.shadowviewer.Image.Size;
				this.shadowviewer.Show();
			}
		}
	}
}
